package coderutil

import (
	"context"
	"net/url"

	"golang.org/x/xerrors"
	"nhooyr.io/websocket"

	"cdr.dev/coder-cli/coder-sdk"
)

// DialEnvWsep dials the executor endpoint using the https://github.com/cdr/wsep message protocol.
// The proper workspace provider envproxy access URL is used.
func DialEnvWsep(ctx context.Context, client *coder.Client, env *coder.Environment) (*websocket.Conn, error) {
	workspaceProvider, err := client.WorkspaceProviderByID(ctx, env.ResourcePoolID)
	if err != nil {
		return nil, xerrors.Errorf("get env workspace provider: %w", err)
	}
	accessURL, err := url.Parse(workspaceProvider.EnvproxyAccessURL)
	if err != nil {
		return nil, xerrors.Errorf("invalid workspace provider envproxy access url: %w", err)
	}

	conn, err := client.DialWsep(ctx, accessURL, env.ID)
	if err != nil {
		return nil, xerrors.Errorf("dial websocket: %w", err)
	}
	return conn, nil
}

// EnvWithWorkspaceProvider composes an Environment entity with its associated WorkspaceProvider.
type EnvWithWorkspaceProvider struct {
	Env               coder.Environment
	WorkspaceProvider coder.WorkspaceProvider
}

// EnvsWithProvider performs the composition of each Environment with its associated WorkspaceProvider.
func EnvsWithProvider(ctx context.Context, client *coder.Client, envs []coder.Environment) ([]EnvWithWorkspaceProvider, error) {
	pooledEnvs := make([]EnvWithWorkspaceProvider, 0, len(envs))
	providers, err := client.WorkspaceProviders(ctx)
	if err != nil {
		return nil, err
	}
	providerMap := make(map[string]coder.WorkspaceProvider, len(providers))
	for _, p := range providers {
		providerMap[p.ID] = p
	}
	for _, e := range envs {
		envProvider, ok := providerMap[e.ResourcePoolID]
		if !ok {
			return nil, xerrors.Errorf("fetch env workspace provider: %w", coder.ErrNotFound)
		}
		pooledEnvs = append(pooledEnvs, EnvWithWorkspaceProvider{
			Env:               e,
			WorkspaceProvider: envProvider,
		})
	}
	return pooledEnvs, nil
}
