// Copyright 2017 The Gitea Authors. All rights reserved.
// Use of this source code is governed by a MIT-style
// license that can be found in the LICENSE file.

package models

import "code.gitea.io/gitea/modules/util"

// IssueWatch is connection request for receiving issue notification.
type IssueWatch struct {
	ID          int64          `xorm:"pk autoincr"`
	UserID      int64          `xorm:"UNIQUE(watch) NOT NULL"`
	IssueID     int64          `xorm:"UNIQUE(watch) NOT NULL"`
	IsWatching  bool           `xorm:"NOT NULL"`
	CreatedUnix util.TimeStamp `xorm:"created NOT NULL"`
	UpdatedUnix util.TimeStamp `xorm:"updated NOT NULL"`
}

// CreateOrUpdateIssueWatch set watching for a user and issue
func CreateOrUpdateIssueWatch(userID, issueID int64, isWatching bool) error {
	iw, exists, err := getIssueWatch(x, userID, issueID)
	if err != nil {
		return err
	}

	if !exists {
		iw = &IssueWatch{
			UserID:     userID,
			IssueID:    issueID,
			IsWatching: isWatching,
		}

		if _, err := x.Insert(iw); err != nil {
			return err
		}
	} else {
		iw.IsWatching = isWatching

		if _, err := x.ID(iw.ID).Cols("is_watching", "updated_unix").Update(iw); err != nil {
			return err
		}
	}
	return nil
}

// GetIssueWatch returns an issue watch by user and issue
func GetIssueWatch(userID, issueID int64) (iw *IssueWatch, exists bool, err error) {
	return getIssueWatch(x, userID, issueID)
}

func getIssueWatch(e Engine, userID, issueID int64) (iw *IssueWatch, exists bool, err error) {
	iw = new(IssueWatch)
	exists, err = e.
		Where("user_id = ?", userID).
		And("issue_id = ?", issueID).
		Get(iw)
	return
}

// GetIssueWatchers returns watchers/unwatchers of a given issue
func GetIssueWatchers(issueID int64) ([]*IssueWatch, error) {
	return getIssueWatchers(x, issueID)
}

func getIssueWatchers(e Engine, issueID int64) (watches []*IssueWatch, err error) {
	err = e.
		Where("`issue_watch`.issue_id = ?", issueID).
		And("`user`.is_active = ?", true).
		And("`user`.prohibit_login = ?", false).
		Join("INNER", "`user`", "`user`.id = `issue_watch`.user_id").
		Find(&watches)
	return
}

func removeIssueWatchersByRepoID(e Engine, userID int64, repoID int64) error {
	iw := &IssueWatch{
		IsWatching: false,
	}
	_, err := e.
		Join("INNER", "issue", "`issue`.id = `issue_watch`.issue_id AND `issue`.repo_id = ?", repoID).
		Cols("is_watching", "updated_unix").
		Where("`issue_watch`.user_id = ?", userID).
		Update(iw)
	return err
}
