/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { SchematicTestRunner, UnitTestTree } from '@angular-devkit/schematics/testing';
import { Schema as ApplicationOptions } from '../application/schema';
import { createAppModule, getFileContent } from '../utility/test';
import { Schema as WorkspaceOptions } from '../workspace/schema';
import { Schema as PipeOptions } from './schema';


describe('Pipe Schematic', () => {
  const schematicRunner = new SchematicTestRunner(
    '@schematics/angular',
    require.resolve('../collection.json'),
  );
  const defaultOptions: PipeOptions = {
    name: 'foo',
    spec: true,
    module: undefined,
    export: false,
    flat: true,
    project: 'bar',
  };

  const workspaceOptions: WorkspaceOptions = {
    name: 'workspace',
    newProjectRoot: 'projects',
    version: '6.0.0',
  };

  const appOptions: ApplicationOptions = {
    name: 'bar',
    inlineStyle: false,
    inlineTemplate: false,
    routing: false,
    style: 'css',
    skipTests: false,
    skipPackageJson: false,
  };
  let appTree: UnitTestTree;
  beforeEach(() => {
    appTree = schematicRunner.runSchematic('workspace', workspaceOptions);
    appTree = schematicRunner.runSchematic('application', appOptions, appTree);
  });

  it('should create a pipe', () => {
    const options = { ...defaultOptions };

    const tree = schematicRunner.runSchematic('pipe', options, appTree);
    const files = tree.files;
    expect(files).toContain('/projects/bar/src/app/foo.pipe.spec.ts');
    expect(files).toContain('/projects/bar/src/app/foo.pipe.ts');
    const moduleContent = getFileContent(tree, '/projects/bar/src/app/app.module.ts');
    expect(moduleContent).toMatch(/import.*Foo.*from '.\/foo.pipe'/);
    expect(moduleContent).toMatch(/declarations:\s*\[[^\]]+?,\r?\n\s+FooPipe\r?\n/m);
  });

  it('should import into a specified module', () => {
    const options = { ...defaultOptions, module: 'app.module.ts' };

    const tree = schematicRunner.runSchematic('pipe', options, appTree);
    const appModule = getFileContent(tree, '/projects/bar/src/app/app.module.ts');

    expect(appModule).toMatch(/import { FooPipe } from '.\/foo.pipe'/);
  });

  it('should fail if specified module does not exist', () => {
    const options = { ...defaultOptions, module: '/projects/bar/src/app/app.moduleXXX.ts' };
    let thrownError: Error | null = null;
    try {
      schematicRunner.runSchematic('pipe', options, appTree);
    } catch (err) {
      thrownError = err;
    }
    expect(thrownError).toBeDefined();
  });

  it('should handle a path in the name and module options', () => {
    appTree = schematicRunner.runSchematic(
      'module',
      { name: 'admin/module', project: 'bar' },
      appTree,
    );

    const options = { ...defaultOptions, module: 'admin/module' };
    appTree = schematicRunner.runSchematic('pipe', options, appTree);

    const content = appTree.readContent('/projects/bar/src/app/admin/module/module.module.ts');
    expect(content).toMatch(/import { FooPipe } from '\.\.\/\.\.\/foo.pipe'/);
  });

  it('should export the pipe', () => {
    const options = { ...defaultOptions, export: true };

    const tree = schematicRunner.runSchematic('pipe', options, appTree);
    const appModuleContent = getFileContent(tree, '/projects/bar/src/app/app.module.ts');
    expect(appModuleContent).toMatch(/exports: \[FooPipe\]/);
  });

  it('should respect the flat flag', () => {
    const options = { ...defaultOptions, flat: false };

    const tree = schematicRunner.runSchematic('pipe', options, appTree);
    const files = tree.files;
    expect(files).toContain('/projects/bar/src/app/foo/foo.pipe.spec.ts');
    expect(files).toContain('/projects/bar/src/app/foo/foo.pipe.ts');
    const moduleContent = getFileContent(tree, '/projects/bar/src/app/app.module.ts');
    expect(moduleContent).toMatch(/import.*Foo.*from '.\/foo\/foo.pipe'/);
    expect(moduleContent).toMatch(/declarations:\s*\[[^\]]+?,\r?\n\s+FooPipe\r?\n/m);
  });

  it('should use the module flag even if the module is a routing module', () => {
    const routingFileName = 'app-routing.module.ts';
    const routingModulePath = `/projects/bar/src/app/${routingFileName}`;
    const newTree = createAppModule(appTree, routingModulePath);
    const options = { ...defaultOptions, module: routingFileName };
    const tree = schematicRunner.runSchematic('pipe', options, newTree);
    const content = getFileContent(tree, routingModulePath);
    expect(content).toMatch(/import { FooPipe } from '.\/foo.pipe/);
  });

  it('should respect the sourceRoot value', () => {
    const config = JSON.parse(appTree.readContent('/angular.json'));
    config.projects.bar.sourceRoot = 'projects/bar/custom';
    appTree.overwrite('/angular.json', JSON.stringify(config, null, 2));

    // should fail without a module in that dir
    expect(() => schematicRunner.runSchematic('pipe', defaultOptions, appTree)).toThrow();

    // move the module
    appTree.rename('/projects/bar/src/app/app.module.ts', '/projects/bar/custom/app/app.module.ts');
    appTree = schematicRunner.runSchematic('pipe', defaultOptions, appTree);
    expect(appTree.files).toContain('/projects/bar/custom/app/foo.pipe.ts');
  });
});
