/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { Architect } from '@angular-devkit/architect/src/index2';
import { TestingArchitectHost } from '@angular-devkit/architect/testing/index2';
import { experimental, join, normalize, schema } from '@angular-devkit/core';
import { NodeJsSyncHost } from '@angular-devkit/core/node';
import * as fs from 'fs';
import fetch from 'node-fetch';  // tslint:disable-line:no-implicit-dependencies
import * as path from 'path';
import { WorkspaceNodeModulesArchitectHost } from '../../../architect/node';
import { DevServerBuildResult } from './index2';

const devkitRoot = normalize((global as any)._DevKitRoot); // tslint:disable-line:no-any


describe('Dev Server Builder', () => {
  let testArchitectHost: TestingArchitectHost;
  let architect: Architect;
  let vfHost: NodeJsSyncHost;

  const webpackTargetSpec = { project: 'app', target: 'serve' };

  async function createArchitect(workspaceRoot: string) {
    vfHost = new NodeJsSyncHost();
    const configContent = fs.readFileSync(path.join(workspaceRoot, 'angular.json'), 'utf-8');
    const workspaceJson = JSON.parse(configContent);

    const registry = new schema.CoreSchemaRegistry();
    registry.addPostTransform(schema.transforms.addUndefinedDefaults);

    const workspace = new experimental.workspace.Workspace(normalize(workspaceRoot), vfHost);
    await workspace.loadWorkspaceFromJson(workspaceJson).toPromise();

    testArchitectHost = new TestingArchitectHost(workspaceRoot, workspaceRoot,
      new WorkspaceNodeModulesArchitectHost(workspace, workspaceRoot));
    architect = new Architect(testArchitectHost, registry);
  }

  beforeEach(async () => {
    await createArchitect(join(devkitRoot, 'tests/angular_devkit/build_webpack/basic-app/'));
  });

  it('works', async () => {
    const run = await architect.scheduleTarget(webpackTargetSpec);
    const output = await run.result as DevServerBuildResult;
    expect(output.success).toBe(true);

    const response = await fetch(`http://${output.address}:${output.port}/bundle.js`);
    expect(await response.text()).toContain(`console.log('hello world')`);

    await run.stop();
  }, 30000);
});
