package logger

import (
	"fmt"
	"io/ioutil"
	"os"
	"runtime"
	"strings"

	"github.com/sirupsen/logrus"
)

// Config holds configuration parameters.
type Config struct {
	// env defines application's current environment that appears in the logs.
	Env string
	// level defines minimum log level that needs to be logged.
	Level string
	// Discard defines if the logs should be discarded or not.
	Discard bool
	// CtxFields defines the list of context key-value pair that should appear
	// in the logs.
	CtxFields map[string]interface{}
}

// Setup sets up the application logger globally.
func Setup(config Config) {
	lev, _ := logrus.ParseLevel(config.Level)

	logrus.SetLevel(lev)
	logrus.AddHook(config)
	logrus.SetReportCaller(true)
	if config.Discard {
		logrus.SetOutput(ioutil.Discard)
	} else {
		logrus.SetOutput(os.Stdout)
	}

	logrus.SetFormatter(&logrus.JSONFormatter{
		DisableHTMLEscape: true,
		CallerPrettyfier:  caller(),
		FieldMap: logrus.FieldMap{
			logrus.FieldKeyMsg:   "message",
			logrus.FieldKeyFile:  "caller",
			logrus.FieldKeyLevel: "level",
		},
	})
}

// Levels allows all log levels to include all default fields.
func (c Config) Levels() []logrus.Level {
	return logrus.AllLevels
}

// Fire runs on each logging call in order to inject one off runtime specific
// static values as well as the dynamic values coming from request context.
func (c Config) Fire(e *logrus.Entry) error {
	e.Data["env"] = c.Env

	if e.Context != nil {
		for k, v := range c.CtxFields {
			e.Data[k] = e.Context.Value(v).(string)
		}
	}

	return nil
}

// caller returns string presentation of log caller which is formatted as
// `/path/to/file.go:line_number`. e.g. `/internal/app/api.go:25`
func caller() func(*runtime.Frame) (function string, file string) {
	return func(f *runtime.Frame) (function string, file string) {
		p, _ := os.Getwd()

		return "", fmt.Sprintf("%s:%d", strings.TrimPrefix(f.File, p), f.Line)
	}
}
