package request

import (
	"context"
	"net/http"

	"github.com/google/uuid"
)

type ctxID int

const CtxRequestIDKey = ctxID(iota)

// ID is a server level middleware. It is called on every request to handle
// `X-Request-ID` header. If the header already exists in the request, nothing
// else is done otherwise a new UUID V4 format is generated and added to request
// `context`.
func ID(h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		id := r.Header.Get("X-Request-ID")
		if id == "" {
			id = uuid.New().String()
		}

		h.ServeHTTP(w, r.WithContext(context.WithValue(r.Context(), CtxRequestIDKey, id)))
	})
}

// CtxRequestID retrieves the `X-Request-ID` from the current request `context`.
func CtxRequestID(r *http.Request) string {
	return r.Context().Value(CtxRequestIDKey).(string)
}

// -----------------------------------------------------------------------------

type ctxInfo int

const (
	CtxRequestProtocolKey ctxInfo = iota
	CtxRequestMethodKey
	CtxRequestPathKey
)

// Info is a server level middleware. It is called on every request to construct
// request protocol, method and path values then added to request `context`.
func Info(h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		ctx := context.WithValue(r.Context(), CtxRequestProtocolKey, r.Proto)
		ctx = context.WithValue(ctx, CtxRequestMethodKey, r.Method)
		ctx = context.WithValue(ctx, CtxRequestPathKey, r.RequestURI)

		h.ServeHTTP(w, r.WithContext(ctx))
	})
}

// CtxRequestProtocol retrieves the request protocol from the current request
// `context`.
func CtxRequestProtocol(r *http.Request) string {
	return r.Context().Value(CtxRequestProtocolKey).(string)
}

// CtxRequestMethod retrieves the request method from the current request
// `context`.
func CtxRequestMethod(r *http.Request) string {
	return r.Context().Value(CtxRequestMethodKey).(string)
}

// CtxRequestPath retrieves the request path from the current request `context`.
func CtxRequestPath(r *http.Request) string {
	return r.Context().Value(CtxRequestPathKey).(string)
}
