package logger

import (
	"fmt"
	"io/ioutil"
	"os"
	"runtime"
	"strings"

	"github.com/sirupsen/logrus"
)

// Config holds configuration parameters.
type Config struct {
	// level defines minimum log level that needs to be logged.
	Level string
	// Discard defines if the logs should be discarded or not. Often dedicated
	// to test environment.
	Discard bool
	// DefaultFields defines the list of static default key-value pair that
	// should appear in the logs.
	DefaultFields map[string]interface{}
	// ContextFields defines the list of dynamic context key-value pair that
	// should appear in the logs.
	ContextFields map[string]interface{}
}

// Setup sets up the application logger globally.
func Setup(config Config) error {
	lev, err := logrus.ParseLevel(config.Level)
	if err != nil {
		return err
	}

	logrus.SetLevel(lev)
	logrus.AddHook(config)
	logrus.SetReportCaller(true)
	if config.Discard {
		logrus.SetOutput(ioutil.Discard)
	} else {
		logrus.SetOutput(os.Stdout)
	}

	logrus.SetFormatter(&logrus.JSONFormatter{
		DisableHTMLEscape: true,
		CallerPrettyfier:  caller(),
		FieldMap: logrus.FieldMap{
			logrus.FieldKeyMsg:  "message",
			logrus.FieldKeyFile: "caller",
		},
	})

	return nil
}

// Levels allows all log levels to include all key-value pair.
func (c Config) Levels() []logrus.Level {
	return logrus.AllLevels
}

// Fire runs on each logging call in order to add static and dynamic key-value
// pair to logs.
func (c Config) Fire(e *logrus.Entry) error {
	for k, v := range c.DefaultFields {
		e.Data[k] = v
	}

	if e.Context != nil {
		for k, v := range c.ContextFields {
			if e.Context.Value(v) != nil {
				e.Data[k] = e.Context.Value(v).(string)
			}
		}
	}

	return nil
}

// caller returns string presentation of log caller which is formatted as
// `/path/to/file.go:line`. e.g. `/internal/app/api.go:25`
func caller() func(*runtime.Frame) (function string, file string) {
	return func(f *runtime.Frame) (function string, file string) {
		p, _ := os.Getwd()

		return "", fmt.Sprintf("%s:%d", strings.TrimPrefix(f.File, p), f.Line)
	}
}
