package middleware

import (
	"context"
	"net/http"

	"github.com/google/uuid"
)

type ctxRequestID string

const CtxRequestIDKey = ctxRequestID("request_id")

// RequestID is a server level middleware which creates a unique request
// identifier and links it to its corresponding key in the request context. If
// the `X-Request-ID` header already presents, its value is used instead. The
// purpose of this middleware is to inject the value to the logs.
func RequestID(h http.Handler) http.Handler {
	return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		id := r.Header.Get("X-Request-ID")
		if id == "" {
			id = uuid.New().String()
		}

		h.ServeHTTP(w, r.WithContext(context.WithValue(r.Context(), CtxRequestIDKey, id)))
	})
}

// RequestIDFromContext returns the current request identifier from the request
// context.
func RequestIDFromContext(ctx context.Context) string {
	if ctx.Value(CtxRequestIDKey) != nil {
		return ctx.Value(CtxRequestIDKey).(string)
	}

	return ""
}
