package rest

import (
	"encoding/json"
	"errors"
	"net/http"
)

// All the codes and messages here are exposed to the client so care must be
// taken when adding a new one or editing an existing one.

type ErrorCode string

const (
	ErrCodeInvalidRequest   ErrorCode = "invalid_request"
	ErrCodeRouteNotFound    ErrorCode = "route_not_found"
	ErrCodeResourceNotFound ErrorCode = "resource_not_found"
	ErrCodeMethodNotAllowed ErrorCode = "method_not_allowed"
	ErrCodeConflict         ErrorCode = "conflict"
	ErrCodeInternal         ErrorCode = "internal"
)

type ErrorMessage string

const (
	ErrMsgInvalidRequest   ErrorMessage = "An invalid request was provided"
	ErrMsgRouteNotFound    ErrorMessage = "The route cannot be found"
	ErrMsgResourceNotFound ErrorMessage = "The resource cannot be found"
	ErrMsgMethodNotAllowed ErrorMessage = "The method is not allowed"
	ErrMsgConflict         ErrorMessage = "A conflict has been detected"
	ErrMsgInternal         ErrorMessage = "An internal error has occurred"
)

type ErrorFieldMessage string

const (
	MsgInvalidValue ErrorFieldMessage = "Invalid value"
	MsgUnknownField ErrorFieldMessage = "Unknown field"
)

type Response struct {
	// Shared common fields.
	Status  int               `json:"-"`
	Headers map[string]string `json:"-"`

	// Success specific fields.
	Data interface{} `json:"data,omitempty"`
	Meta interface{} `json:"meta,omitempty"`

	// Failure specific fields.
	Code    ErrorCode                    `json:"code,omitempty"`
	Message ErrorMessage                 `json:"message,omitempty"`
	Errors  map[string]ErrorFieldMessage `json:"errors,omitempty"`
}

func Write(w http.ResponseWriter, r Response) error {
	if r.Status < 100 || r.Status > 599 {
		return errors.New("invalid status")
	}

	for k, v := range r.Headers {
		w.Header().Add(k, v)
	}

	if !isBodyAllowed(r.Status) {
		w.WriteHeader(r.Status)
		return nil
	}

	bdy, err := json.Marshal(r)
	if err != nil {
		return err
	}

	w.Header().Set("Content-Type", "application/json; charset=utf-8")
	w.WriteHeader(r.Status)

	if _, err := w.Write(bdy); err != nil {
		return err
	}

	return nil
}

// See RFC 7230, section 3.3.
func isBodyAllowed(status int) bool {
	if (status >= 100 && status <= 199) || status == 204 || status == 304 {
		return false
	}

	return true
}
