package rest

import (
	"context"
	"net/http"

	"github.com/julienschmidt/httprouter"
)

// Middleware represents server and route level middlewares that apply to all or
// specific routes respectively.
type Middleware func(http.Handler) http.Handler

// Route is used by application routes in order to get them registered into the
// `Router` type with the help of `AddRoutes` method.
type Route struct {
	Method      string
	Path        string
	Handler     http.HandlerFunc
	Middlewares []Middleware
}

// Router represents HTTP router. It helps adding application routes, server
// level middlewares and some custom handler.
type Router struct {
	handler     http.Handler
	middlewares []Middleware
}

// NewRouter returns `Router` type with some default configurations.
func NewRouter() *Router {
	r := httprouter.New()
	r.RedirectTrailingSlash = false
	r.RedirectFixedPath = false

	return &Router{handler: r}
}

// SetRouteNotFound sets a custom handler that handles default 404 responses.
func (r *Router) SetRouteNotFound(handler http.Handler) {
	r.handler.(*httprouter.Router).NotFound = handler
}

// SetMethodNotAllowed sets a custom handler that handles default 405 responses.
func (r *Router) SetMethodNotAllowed(handler http.Handler) {
	r.handler.(*httprouter.Router).MethodNotAllowed = handler
}

// UseMiddleware adds a new middleware to the collection. The given middleware
// is applicable to all registered routes.
func (r *Router) UseMiddleware(middleware Middleware) {
	r.middlewares = append(r.middlewares, middleware)
}

// AddRoutes adds new routes to the collection. It registers route specific
// middlewares if provided.
func (r *Router) AddRoutes(routes []Route) {
	for _, route := range routes {
		for _, middleware := range route.Middlewares {
			route.Handler = middleware(route.Handler).(http.HandlerFunc)
		}
		r.handler.(*httprouter.Router).Handler(route.Method, route.Path, route.Handler)
	}
}

// Handler applies previously added middlewares to all routes and returns HTTP
// handler for HTTP server to use.
func (r *Router) Handler() http.Handler {
	for _, middleware := range r.middlewares {
		r.handler = middleware(r.handler)
	}

	return r.handler
}

// RouteParamFromContext returns the value for the given route parameter name.
func RouteParamFromContext(ctx context.Context, name string) string {
	if name == "" {
		return ""
	}

	return httprouter.ParamsFromContext(ctx).ByName(name)
}
