﻿[CmdletBinding(SupportsShouldProcess=$true, ConfirmImpact="High")]
param ()
Set-StrictMode -Version 2.0
If ($PSVersiontable.PSVersion.Major -le 2) {$PSScriptRoot = Split-Path -Parent $MyInvocation.MyCommand.Path}
Import-Module $PSScriptRoot\OpenSSHUtils -Force

#check sshd config file
$sshdConfigPath = join-path $env:ProgramData\ssh "sshd_config"
if(Test-Path $sshdConfigPath -PathType Leaf)
{
    Repair-SshdConfigPermission -FilePath $sshdConfigPath @psBoundParameters
}
else
{
    Write-host "$sshdConfigPath does not exist"  -ForegroundColor Yellow
}
 
#check host keys
<#
$warning = @"
To keep the host private keys secure, it is recommended to register them with ssh-agent following
steps in link https://github.com/PowerShell/Win32-OpenSSH/wiki/Install-Win32-OpenSSH.
If you choose not to register the keys with ssh-agent, please grant sshd read access to the private host keys after run this script.
"@
$prompt = "Did you register host private keys with ssh-agent?"
$description = "Grant sshd read access to the private host keys"

if($pscmdlet.ShouldProcess($description, $prompt, $warning))
{
    $warning = @"
To keep the host private keys secure, it is recommended to register them with ssh-agent following
steps in link https://github.com/PowerShell/Win32-OpenSSH/wiki/Install-Win32-OpenSSH.
If you choose not to register the keys with ssh-agent, please grant sshd read access to the private host keys after run this script.
"@
    Write-Warning $warning
    Write-Host " "
}#>

Get-ChildItem $env:ProgramData\ssh\ssh_host_*_key -ErrorAction SilentlyContinue | % {
    Repair-SshdHostKeyPermission -FilePath $_.FullName @psBoundParameters
}


#check authorized_keys
Get-ChildItem "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion\ProfileList"  -ErrorAction SilentlyContinue | % {
    $properties =  Get-ItemProperty $_.pspath  -ErrorAction SilentlyContinue
    $userProfilePath = ""
    if($properties)
    {
        $userProfilePath =  $properties.ProfileImagePath
    }
    $filePath = Join-Path $userProfilePath .ssh\authorized_keys
    if(Test-Path $filePath -PathType Leaf)
    {
        Repair-AuthorizedKeyPermission -FilePath $filePath @psBoundParameters
    }
}

Write-Host "   Done."
Write-Host " "

# SIG # Begin signature block
# MIIpeAYJKoZIhvcNAQcCoIIpaTCCKWUCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCNZxzRshrfmXF6
# WvQywldKGwPNcoiU39jyVtHuk5IarKCCDYIwggYAMIID6KADAgECAhMzAAAAxa4M
# aP4Yy0aNAAAAAADFMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMTcwODExMjAyMDI1WhcNMTgwODExMjAyMDI1WjB0MQsw
# CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9u
# ZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMR4wHAYDVQQDExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIB
# AQCmTM+CNIcTiNU9avGPbZnSEGiTyBoBQloAzdqM3Hb4YCqzBBcStCdXwi52Tlu5
# FgDfayDSrYbY+9+zzP+xbQRZYD4TsGeDNYK3TKfDpHVbL38JyS1tG0eLKXY4jA89
# PM6x5TPforEKesh0CLX5hkM/zKTCjG6JBPeFALqomY4pr8GYM8HdqQJ6jrs90zzH
# vEwpZxObn30WFpYKpXbVuSqwxxxU2ZOuON7eQvNkJBCxwFJ49d9BtCbz287SD9k3
# W7XxTAo6jbwGuXB9Afc+1rVI8NbuhUofeDJPQLR7NrSvZvmlPvjq2EA8OQ6hTbhw
# f8Jf1AR9LR7qdA/Wc4vRBuLnAgMBAAGjggF/MIIBezAfBgNVHSUEGDAWBgorBgEE
# AYI3TAgBBggrBgEFBQcDAzAdBgNVHQ4EFgQU164eIpQwDitpNYitLuAaFjrtNIMw
# UQYDVR0RBEowSKRGMEQxDDAKBgNVBAsTA0RCWDE0MDIGA1UEBRMrMjMwMDEyK2M4
# MDRiNWVhLTQ5YjQtNDIzOC04MzYyLWQ4NTFmYTIyNTRmYzAfBgNVHSMEGDAWgBRI
# bmTlUAXTgqoXNzcitW2oynUClTBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vd3d3
# Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NybC9NaWNDb2RTaWdQQ0EyMDExXzIwMTEt
# MDctMDguY3JsMGEGCCsGAQUFBwEBBFUwUzBRBggrBgEFBQcwAoZFaHR0cDovL3d3
# dy5taWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0cy9NaWNDb2RTaWdQQ0EyMDExXzIw
# MTEtMDctMDguY3J0MAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcNAQELBQADggIBAJwr
# NGIdIyziSKi4wFLnk1YgC0F2lGCE8UYIJgzNiuuJlmMwshYHmOnpz7EtYuJdbS6Y
# jGEnFnDrZTGwVDUr0SReggUtw7eUBJrRu3BQtnRL/d/zMW1B1kpHR/iOG0MWgz5N
# 0AvGdo0tAlqAfj5a5Zxw3CgGq0irHYNPULPZD8rV5wc/qbVyb5sMxyl1N3TR21CL
# 8MbJkkiW1BE0e2RkBDHoKtQKSA2WJZkaBDmBxXupltI0RdPm7bdjQa5DDZWpYmwW
# 8mCp0wfP8PY42jiI5or6ZXOCj5gLfkLsq1NfZdNNQDLR+FEHr8RXGpANbeLIKI5K
# RyF8OE6fjvc6VqhvnMKWg4E7szCTtKuWa9g5nVyTuQzGe3MeWRTZQCO0SdhAuWKQ
# 1b+11ZDff+bgL00krNl9Nh3eA99W79SmJDXSkwb2ji6+twkREVkQK6DHSFnC90cx
# FUPVJ5REGPaYjUfp5YyXyXSNAErTEKwZy7BS3kMuhaiOoOcxMQZi/4qQpJm0yK7E
# 3f8oGdK990kJyF8DXm8Mj3ahq+9wEznuhazPsrigHiy9M7XqF77wV11WxeTuHMOm
# MddeLmTcYXQgFVuR8CvxGh9KJPLyQwF5UJVyWLrf0tJpsPn/WgEWTDHYLuPz/pVi
# dOmBIs9cH5zy9P6KbkAoU0gDhcJyWG6lzizzVDjhMIIHejCCBWKgAwIBAgIKYQ6Q
# 0gAAAAAAAzANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgT
# Cldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29m
# dCBDb3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2VydGlmaWNh
# dGUgQXV0aG9yaXR5IDIwMTEwHhcNMTEwNzA4MjA1OTA5WhcNMjYwNzA4MjEwOTA5
# WjB+MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
# UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSgwJgYDVQQD
# Ex9NaWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQSAyMDExMIICIjANBgkqhkiG9w0B
# AQEFAAOCAg8AMIICCgKCAgEAq/D6chAcLq3YbqqCEE00uvK2WCGfQhsqa+laUKq4
# BjgaBEm6f8MMHt03a8YS2AvwOMKZBrDIOdUBFDFC04kNeWSHfpRgJGyvnkmc6Whe
# 0t+bU7IKLMOv2akrrnoJr9eWWcpgGgXpZnboMlImEi/nqwhQz7NEt13YxC4Ddato
# 88tt8zpcoRb0RrrgOGSsbmQ1eKagYw8t00CT+OPeBw3VXHmlSSnnDb6gE3e+lD3v
# ++MrWhAfTVYoonpy4BI6t0le2O3tQ5GD2Xuye4Yb2T6xjF3oiU+EGvKhL1nkkDst
# rjNYxbc+/jLTswM9sbKvkjh+0p2ALPVOVpEhNSXDOW5kf1O6nA+tGSOEy/S6A4aN
# 91/w0FK/jJSHvMAhdCVfGCi2zCcoOCWYOUo2z3yxkq4cI6epZuxhH2rhKEmdX4ji
# JV3TIUs+UsS1Vz8kA/DRelsv1SPjcF0PUUZ3s/gA4bysAoJf28AVs70b1FVL5zmh
# D+kjSbwYuER8ReTBw3J64HLnJN+/RpnF78IcV9uDjexNSTCnq47f7Fufr/zdsGbi
# wZeBe+3W7UvnSSmnEyimp31ngOaKYnhfsi+E11ecXL93KCjx7W3DKI8sj0A3T8Hh
# hUSJxAlMxdSlQy90lfdu+HggWCwTXWCVmj5PM4TasIgX3p5O9JawvEagbJjS4NaI
# jAsCAwEAAaOCAe0wggHpMBAGCSsGAQQBgjcVAQQDAgEAMB0GA1UdDgQWBBRIbmTl
# UAXTgqoXNzcitW2oynUClTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNV
# HQ8EBAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAWgBRyLToCMZBDuRQF
# TuHqp8cx0SOJNDBaBgNVHR8EUzBRME+gTaBLhklodHRwOi8vY3JsLm1pY3Jvc29m
# dC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFfMDNf
# MjIuY3JsMF4GCCsGAQUFBwEBBFIwUDBOBggrBgEFBQcwAoZCaHR0cDovL3d3dy5t
# aWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXQyMDExXzIwMTFfMDNf
# MjIuY3J0MIGfBgNVHSAEgZcwgZQwgZEGCSsGAQQBgjcuAzCBgzA/BggrBgEFBQcC
# ARYzaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9kb2NzL3ByaW1hcnlj
# cHMuaHRtMEAGCCsGAQUFBwICMDQeMiAdAEwAZQBnAGEAbABfAHAAbwBsAGkAYwB5
# AF8AcwB0AGEAdABlAG0AZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQBn8oal
# mOBUeRou09h0ZyKbC5YR4WOSmUKWfdJ5DJDBZV8uLD74w3LRbYP+vj/oCso7v0ep
# o/Np22O/IjWll11lhJB9i0ZQVdgMknzSGksc8zxCi1LQsP1r4z4HLimb5j0bpdS1
# HXeUOeLpZMlEPXh6I/MTfaaQdION9MsmAkYqwooQu6SpBQyb7Wj6aC6VoCo/KmtY
# SWMfCWluWpiW5IP0wI/zRive/DvQvTXvbiWu5a8n7dDd8w6vmSiXmE0OPQvyCInW
# H8MyGOLwxS3OW560STkKxgrCxq2u5bLZ2xWIUUVYODJxJxp/sfQn+N4sOiBpmLJZ
# iWhub6e3dMNABQamASooPoI/E01mC8CzTfXhj38cbxV9Rad25UAqZaPDXVJihsMd
# YzaXht/a8/jyFqGaJ+HNpZfQ7l1jQeNbB5yHPgZ3BtEGsXUfFL5hYbXw3MYbBL7f
# QccOKO7eZS/sl/ahXJbYANahRr1Z85elCUtIEJmAH9AAKcWxm6U/RXceNcbSoqKf
# enoi+kiVH6v7RyOA9Z74v2u3S5fi63V4GuzqN5l5GEv/1rMjaHXmr/r8i+sLgOpp
# O6/8MO0ETI7f33VtY5E90Z1WTk+/gFcioXgRMiF670EKsT/7qMykXcGhiJtXcVZO
# SEXAQsmbdlsKgEhr/Xmfwb1tbWrJUnMTDXpQzTGCG0wwghtIAgEBMIGVMH4xCzAJ
# BgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
# MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jv
# c29mdCBDb2RlIFNpZ25pbmcgUENBIDIwMTECEzMAAADFrgxo/hjLRo0AAAAAAMUw
# DQYJYIZIAWUDBAIBBQCgga4wGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYK
# KwYBBAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIEh8qYgJ
# Gf/gqSlejASK0tRt/wADFFyP4s1oJvUUX7kUMEIGCisGAQQBgjcCAQwxNDAyoBSA
# EgBNAGkAYwByAG8AcwBvAGYAdKEagBhodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20w
# DQYJKoZIhvcNAQEBBQAEggEAipgEIyt8HtlxRjn3h9FA/rissHMABpGg/7Lg1Ywl
# GlpdtHha+8ww437lEu/tIh54cqajFWBDEyLMYRkKquuD8elEr9rll9CcMcNPaEr6
# oYvP5Beiv4UM8TcePbxIPYS9E8qhzUq3UmG8EuPRwLRp/6MeKlhyaGrPQH0b3kYy
# Srq2O9bEAmCJnprx40/2OtByWc8wEfMhaKhBysJ7K13dtN75OTPqy7awNcDCLElF
# R2pur3NhG2OqWbr/svID+ugy4QiMdhivFDSTyTWycrt4Y1FZN8ZIcH6JUFb3RjyR
# WkwDBF7rKiqj0KcXR7RLvCdFRJbR4KJF2e7bgzr3rYhrF6GCGNYwghjSBgorBgEE
# AYI3AwMBMYIYwjCCGL4GCSqGSIb3DQEHAqCCGK8wghirAgEDMQ8wDQYJYIZIAWUD
# BAIBBQAwggFRBgsqhkiG9w0BCRABBKCCAUAEggE8MIIBOAIBAQYKKwYBBAGEWQoD
# ATAxMA0GCWCGSAFlAwQCAQUABCCw2P2FXYFNxE54nbECSjB3grNLzBj/Z5zdHyQn
# U9kObwIGW1Wgf3nvGBMyMDE4MDcyNjE4MjAzMS4wMDhaMASAAgH0oIHQpIHNMIHK
# MQswCQYDVQQGEwJVUzELMAkGA1UECBMCV0ExEDAOBgNVBAcTB1JlZG1vbmQxHjAc
# BgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEtMCsGA1UECxMkTWljcm9zb2Z0
# IElyZWxhbmQgT3BlcmF0aW9ucyBMaW1pdGVkMSYwJAYDVQQLEx1UaGFsZXMgVFNT
# IEVTTjo4RTlFLTRCRDAtMkVEMDElMCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3Rh
# bXAgc2VydmljZaCCFC0wggTxMIID2aADAgECAhMzAAAAwxrtitINPstRAAAAAADD
# MA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5n
# dG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9y
# YXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwMB4X
# DTE4MDEzMTE5MDA0NloXDTE4MDkwNzE5MDA0NlowgcoxCzAJBgNVBAYTAlVTMQsw
# CQYDVQQIEwJXQTEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMS0wKwYDVQQLEyRNaWNyb3NvZnQgSXJlbGFuZCBPcGVyYXRp
# b25zIExpbWl0ZWQxJjAkBgNVBAsTHVRoYWxlcyBUU1MgRVNOOjhFOUUtNEJEMC0y
# RUQwMSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBzZXJ2aWNlMIIBIjAN
# BgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAjDf3SRVVqR6aLJkKB6yPZXX+VLq/
# lO+UqzeFD7fkMZVv7yhcpWy1zc3xIrc22s5iuCkIq7Vlpi3AR+CaqY8U4WKUo1PO
# 25oUrVEO7aas4lFigPxm2EEh+ImTqtk1mOEDeH7G9/S98u8UGtOtpqZYhoMAeYJ2
# Jij0PVMHvielyMyjeyCixozwZ3GODqEuD4z1Fwu5/8mIM3L5CQ3OFof5WImbZ840
# nJVUY53+jFJyZ4P0juMvcAY7uiTdeeYnQDGwb57iZ2W7WWct5kf6Z1/Aip7rmI6U
# gufODE7B/v6r2s+Bia5J4d5/U8BnmOXqCQTsFSEXY2Z2vQ7oqpaz3IIrEwIDAQAB
# o4IBGzCCARcwHQYDVR0OBBYEFKil/GFCx+yFicRlovgD0rky37XaMB8GA1UdIwQY
# MBaAFNVjOlyKMZDzQ3t8RhvFM2hahW1VMFYGA1UdHwRPME0wS6BJoEeGRWh0dHA6
# Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1RpbVN0YVBD
# QV8yMDEwLTA3LTAxLmNybDBaBggrBgEFBQcBAQROMEwwSgYIKwYBBQUHMAKGPmh0
# dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kvY2VydHMvTWljVGltU3RhUENBXzIw
# MTAtMDctMDEuY3J0MAwGA1UdEwEB/wQCMAAwEwYDVR0lBAwwCgYIKwYBBQUHAwgw
# DQYJKoZIhvcNAQELBQADggEBAKgBhzcPOkU4X5BgPbVavRsk7uBRA8PQFt3cUgVh
# FAmPwoih1W1eO7Uwt6aIA4B9DI2JBE/a0XlBFs6UHCC0b56c/sSUtD2t+DK4X+Ht
# wUQIV8loNogOkG0SAaXVoDWVgz0/wfaEczfb1AOotSKFy3ODxQTKwe0zaUC0j8xb
# 5ZIA7t0txUJpXrTH9PZrGcvFiY6GCZ0SBEEW356rSHSd+zUD5n5rSe6VCXci3eoV
# 5HlsEbQJjekm5hWyniMWchhE7WUGaWtpkEoO58hL6a/NsAQmMKkfu9cR/hAojAyT
# GenxezV7csHXDrwAPU6c55f6EhGu/MQXhA6oLd86oGIR1JQwggXtMIID1aADAgEC
# AhAozDolv7pErESam1hrQzmqMA0GCSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9v
# dCBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkgMjAxMDAeFw0xMDA2MjMyMTU3MjRaFw0z
# NTA2MjMyMjA0MDFaMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0ZSBBdXRob3Jp
# dHkgMjAxMDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBALkInijk5OwG
# TlBos0HFe+uuto6vgboiRB9lNGlMvnBAF/IWe+J5/YbtDTn0G6itkpAeyz12j1rZ
# tZEQLjwFjYptJFTnH+1WrYO0UJwVpRd0iFkg/AjFhHbTaNRvKHjOXLjzUJBE/+Nj
# X76hmiyWFQTWB/4ehCHgQjERxCg2lM9QpGKeydarcQCyWwzmltQKJJb1/8bVtxvX
# y7chYq8S3KFdN+Ma+xpGmMCbwOdjHyoIkwJ+HmqO8p8YieQihaKxhFdA//UO2G+c
# 7eJFMQHNF+l/sIFF46ohQCahcqqnTzwBBX7ug1ixXgZjmWKReIK3DZMMJGq0G9sn
# 7F+VBD+TSjD1lxizp/kZp5MzHQHI2yJSXNclyUb5ovuHWUO+m2KxjS2GRBpGrHhh
# fjAJ+q6JxEEqImYDkTlFnMeLDKjKDS/7UuoM92MzI53+sB+tZ9anUAPGBHBjtSyx
# hlpDt/uu+W4pbiEhQSYGjMnD7rDChZOhuYXZ5jJsS0w/1l2j5bWdd8OcwFW3dADj
# uDirg5dQ4ZpCJB3GwKMw0RpayFI093PxxxgfM6167MtBYPMjlCDCSEWsXFHGLoDC
# 4ncVvYWH7TadlpHuALWjcOyf442AaIN2uq9dcFIiFuJm+7qzxcL3Pi93psrewabG
# SEzDN1Ej0yfXuE5wlvChRHaveM+a4WYTAgMBAAGjUTBPMAsGA1UdDwQEAwIBhjAP
# BgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTV9lbLj+iiXGJo0T2UkFvXzpoYxDAQ
# BgkrBgEEAYI3FQEEAwIBADANBgkqhkiG9w0BAQsFAAOCAgEArKWWjL+7rqb213GH
# QzFWiP0cMnFbNbfU8JHyrzfiFPHzAiYFPhYUfxS6uE/7ibKy59QJzG25WztkZXBm
# t/KxWt8aAvP1UbhnbXnzv1Z75IS5Kx6bQJwmNPlHGJhp2BzXttG/j2HCZ8S172BD
# jhAbNknkIMqtp8GxJ2UJ+M31WyrQhDPz7x/y9ZwLWJM3oHWg3nLebHUqZiL1jAYw
# Vp9AuTCqQHcVgteL7MDTsr2DxXcMHq6vGVOgTXlxnw+vMM5n+dYszCJBegfyl0IY
# zll5EFXebxDkuNqDZkAWCWgjW5cuJpoCu1eMxbi6aWIygImeof3Aknx7KzMZhCpj
# xQBoYvqfR42ZekU6p+nt7mlCtfOBm0dWEHv8cDaEGHPq7/mXTZ4zI90mC7oqtz9E
# 3IMn/71hWSsRt8pP28WLDBwxrjL4+LlC93/cYZp2sVoE4RE9ZkW3GHG+ySSF1vPU
# ukE0XRItJbmNphNIbUuwB32ZkwlhgXRXJoqraePk2ceIzCTY7FIkXB68kRTilt7r
# Ctqe3V+zW9vUguzGIFCHJUA6+8fuzf4z5W7DhAlVAyU5wOk1XWUxqPa/oAnNKcez
# NjIu3JXzg8Faz4uN9uqzIfik7R4xDrZMEatgC6QSIyIXozZkgpEEEuCrbx7LUAVh
# tED/WYZx0dUzaXypc4o412QM8WkwggZxMIIEWaADAgECAgphCYEqAAAAAAACMA0G
# CSqGSIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
# bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
# aW9uMTIwMAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0ZSBBdXRob3Jp
# dHkgMjAxMDAeFw0xMDA3MDEyMTM2NTVaFw0yNTA3MDEyMTQ2NTVaMHwxCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4w
# HAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29m
# dCBUaW1lLVN0YW1wIFBDQSAyMDEwMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIB
# CgKCAQEAqR0NvHcRijog7PwTl/X6f2mUa3RUENWlCgCChfvtfGhLLF/Fw+Vhwna3
# PmYrW/AVUycEMR9BGxqVHc4JE458YTBZsTBED/FgiIRUQwzXTbg4CLNC3ZOs1nMw
# VyaCo0UN0Or1R4HNvyRgMlhgRvJYR4YyhB50YWeRX4FUsc+TTJLBxKZd0WETbijG
# GvmGgLvfYfxGwScdJGcSchohiq9LZIlQYrFd/XcfPfBXday9ikJNQFHRD5wGPmd/
# 9WbAA5ZEfu/QS/1u5ZrKsajyeioKMfDaTgaRtogINeh4HLDpmc085y9Euqf03GS9
# pAHBIAmTeM38vMDJRF1eFpwBBU8iTQIDAQABo4IB5jCCAeIwEAYJKwYBBAGCNxUB
# BAMCAQAwHQYDVR0OBBYEFNVjOlyKMZDzQ3t8RhvFM2hahW1VMBkGCSsGAQQBgjcU
# AgQMHgoAUwB1AGIAQwBBMAsGA1UdDwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MB8G
# A1UdIwQYMBaAFNX2VsuP6KJcYmjRPZSQW9fOmhjEMFYGA1UdHwRPME0wS6BJoEeG
# RWh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jv
# b0NlckF1dF8yMDEwLTA2LTIzLmNybDBaBggrBgEFBQcBAQROMEwwSgYIKwYBBQUH
# MAKGPmh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kvY2VydHMvTWljUm9vQ2Vy
# QXV0XzIwMTAtMDYtMjMuY3J0MIGgBgNVHSABAf8EgZUwgZIwgY8GCSsGAQQBgjcu
# AzCBgTA9BggrBgEFBQcCARYxaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL1BLSS9k
# b2NzL0NQUy9kZWZhdWx0Lmh0bTBABggrBgEFBQcCAjA0HjIgHQBMAGUAZwBhAGwA
# XwBQAG8AbABpAGMAeQBfAFMAdABhAHQAZQBtAGUAbgB0AC4gHTANBgkqhkiG9w0B
# AQsFAAOCAgEAB+aIUQ3ixuCYP4FxAz2do6Ehb7Prpsz1Mb7PBeKp/vpXbRkws8LF
# Zslq3/Xn8Hi9x6ieJeP5vO1rVFcIK1GCRBL7uVOMzPRgEop2zEBAQZvcXBf/XPle
# FzWYJFZLdO9CEMivv3/Gf/I3fVo/HPKZeUqRUgCvOA8X9S95gWXZqbVr5MfO9sp6
# AG9LMEQkIjzP7QOllo9ZKby2/QThcJ8ySif9Va8v/rbljjO7Yl+a21dA6fHOmWaQ
# jP9qYn/dxUoLkSbiOewZSnFjnXshbcOco6I8+n99lmqQeKZt0uGc+R38ONiU9Mal
# CpaGpL2eGq4EQoO4tYCbIjggtSXlZOz39L9+Y1klD3ouOVd2onGqBooPiRa6YacR
# y5rYDkeagMXQzafQ732D8OE7cQnfXXSYIghh2rBQHm+98eEA3+cxB6STOvdlR3jo
# +KhIq/fecn5ha293qYHLpwmsObvsxsvYgrRyzR30uIUBHoD7G4kqVDmyW9rIDVWZ
# eodzOwjmmC3qjeAzLhIp9cAvVCch98isTtoouLGp25ayp0Kiyc8ZQU3ghvkqmqMR
# ZjDTu3QyS99je/WZii8bxyGvWbWu3EQ8l1Bx16HSxVXjad5XwdHeMMD9zOZN+w2/
# XU/pnR4ZOC+8z1gFLu8NoFA12u8JJxzVs341Hgi62jbb01+P3nSISRKhggLOMIIC
# NwIBATCB+KGB0KSBzTCByjELMAkGA1UEBhMCVVMxCzAJBgNVBAgTAldBMRAwDgYD
# VQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xLTAr
# BgNVBAsTJE1pY3Jvc29mdCBJcmVsYW5kIE9wZXJhdGlvbnMgTGltaXRlZDEmMCQG
# A1UECxMdVGhhbGVzIFRTUyBFU046OEU5RS00QkQwLTJFRDAxJTAjBgNVBAMTHE1p
# Y3Jvc29mdCBUaW1lLVN0YW1wIHNlcnZpY2WiIwoBATAHBgUrDgMCGgMVAMm7RYky
# ezNuJMFuWBPXehil4gjDoIGDMIGApH4wfDELMAkGA1UEBhMCVVMxEzARBgNVBAgT
# Cldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29m
# dCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENB
# IDIwMTAwDQYJKoZIhvcNAQEFBQACBQDfBBNtMCIYDzIwMTgwNzI2MTczMTI1WhgP
# MjAxODA3MjcxNzMxMjVaMHcwPQYKKwYBBAGEWQoEATEvMC0wCgIFAN8EE20CAQAw
# CgIBAAICIksCAf8wBwIBAAICEQowCgIFAN8FZO0CAQAwNgYKKwYBBAGEWQoEAjEo
# MCYwDAYKKwYBBAGEWQoDAqAKMAgCAQACAwehIKEKMAgCAQACAwGGoDANBgkqhkiG
# 9w0BAQUFAAOBgQBdwW9Hl+pqCJ35kDT57Y9XYaCwQbc1m3wIwD2zpcfhpQmVxFEh
# ksOD+v8variYGnZrDBbbEa1yr1RBYPxzGP2Gvv4RZrNxAwMceTq8Ki1yw7+xHLMx
# 7nTTZF12wUG7+Bo16lQ7LDH9zGhYCkxK9pv6+ejASCVwlahfu924+u2v8DGCAw0w
# ggMJAgEBMIGTMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAw
# DgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
# JjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAAwxrt
# itINPstRAAAAAADDMA0GCWCGSAFlAwQCAQUAoIIBSjAaBgkqhkiG9w0BCQMxDQYL
# KoZIhvcNAQkQAQQwLwYJKoZIhvcNAQkEMSIEIL17SohxANGeNOe+z+s0fI9BAF7T
# VNB3myZDeWKFw2VJMIH6BgsqhkiG9w0BCRACLzGB6jCB5zCB5DCBvQQgs/lor7aR
# UiRBlKqohPTCBeOEl7NTrTUeAQEAOjDateowgZgwgYCkfjB8MQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UE
# ChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGlt
# ZS1TdGFtcCBQQ0EgMjAxMAITMwAAAMMa7YrSDT7LUQAAAAAAwzAiBCDMi/luzdfT
# V5+CMfRpmTkcTc9zP8+WouGPUEeCo3sqZTANBgkqhkiG9w0BAQsFAASCAQACXDVc
# t8N4GtBJLUnSFMeKWENUoSRgouwEe0zRnlLdna/k2yPyS7B439xiAXCG9P4KubwL
# 5uUAli2nYnbbhWH1jPfdqmOVgUh7RLgJ17T9+yoTIrIxKdrgBs3OwQjTDg4wWyj+
# oa6eQaji47CIE21IWSpUSdysQiWpgL9kcraS/Ho5fRqw6gbBgmSTEKhv2X9YoCh8
# yW40LyHsRQm+1wL5kaxMUq1H2uPBq2Obd68P3wir4VhmKlXSo9ADK0is195vnJ1w
# S7TusPPem/Ut4JBL8VsBSZWDirSEeuVbywqfnm2xYroZs5c+yz37/L6uTgFk7vtR
# fqo419bn0ScfM7OA
# SIG # End signature block
