package gomodvendor

import (
	"github.com/paketo-buildpacks/packit"
	"github.com/paketo-buildpacks/packit/scribe"
)

//go:generate faux --interface BuildProcess --output fakes/build_process.go
type BuildProcess interface {
	ShouldRun(workingDir string) (ok bool, reason string, err error)
	Execute(path, workingDir string) error
}

func Build(buildProcess BuildProcess, logs scribe.Emitter) packit.BuildFunc {
	return func(context packit.BuildContext) (packit.BuildResult, error) {
		logs.Title("%s %s", context.BuildpackInfo.Name, context.BuildpackInfo.Version)

		ok, reason, err := buildProcess.ShouldRun(context.WorkingDir)
		if err != nil {
			return packit.BuildResult{}, err
		}

		if !ok {
			logs.Process("Skipping build process: %s", reason)
			logs.Break()

			return packit.BuildResult{}, nil
		}

		modCacheLayer, err := context.Layers.Get("mod-cache")
		if err != nil {
			return packit.BuildResult{}, err
		}

		modCacheLayer.Cache = true

		err = buildProcess.Execute(modCacheLayer.Path, context.WorkingDir)
		if err != nil {
			return packit.BuildResult{}, err
		}

		return packit.BuildResult{
			Plan:   context.Plan,
			Layers: []packit.Layer{modCacheLayer},
		}, nil
	}
}
