#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Shows some **Color** package color plotting related examples.
"""

from numpy import matrix
import color
from color.implementations.matplotlib.plots import *

# Plotting a single color.
single_color_plot(color_parameter(name="Neutral 5 (.70 D)", RGB=(0.32315746, 0.32983556, 0.33640183)),
                  text_size=32.)

# Plotting multiples colors.
multi_color_plot([color_parameter(name="Dark Skin", RGB=(0.45293517, 0.31732158, 0.26414773)),
                  color_parameter(name="Light Skin", RGB=(0.77875824, 0.5772645, 0.50453169))],
                 spacing=0.,
                 text_size=32.)

# Plotting color checkers.
print(sorted(color.COLORCHECKERS.keys()))
color_checker_plot("ColorChecker 1976")
color_checker_plot("BabelColor Average",
                   text_display=False)
color_checker_plot("ColorChecker 1976",
                   text_display=False)
color_checker_plot("ColorChecker 2005",
                   text_display=False)

# Plotting a single illuminant relative spectral power distribution.
single_illuminant_relative_spd_plot("F1")

# Plotting multiple illuminants relative spectral power distributions.
print(sorted(color.ILLUMINANTS_RELATIVE_SPD.keys()))
multi_illuminants_relative_spd_plot(["A", "C", "D50", "D55", "D60", "D65", "D75", "F1"])

# Plotting an arbitrary single spectral power distribution.
sample_spd_data = {360: 0.0000,
                   380: 0.0000,
                   400: 0.0641,
                   420: 0.0645,
                   440: 0.0562,
                   460: 0.0537,
                   480: 0.0559,
                   500: 0.0651,
                   520: 0.0705,
                   540: 0.0772,
                   560: 0.0870,
                   580: 0.1128,
                   600: 0.1360,
                   620: 0.1511,
                   640: 0.1688,
                   660: 0.1996,
                   680: 0.2397,
                   700: 0.2852,
                   720: 0.0000,
                   740: 0.0000,
                   760: 0.0000,
                   780: 0.0000,
                   800: 0.0000,
                   820: 0.0000}

# http://physics.nist.gov/PhysRefData/ASD/lines_form.html
# Following data is just given as example and not accurate.
boron_spd = {365.559: 33.028396,
             366.01: 33.032729,
             369.23: 19.178634,
             369.534: 21.150628,
             378.841: 21.150628,
             394.628: 24.51399,
             395.037: 24.514172,
             399.024: 21.150628,
             400.007: 228.47965,
             412.193: 18.678179,
             414.725: 33.028396,
             419.479: 17.866487,
             419.777: 250.30083,
             424.298: 30.107049,
             424.361: 30.107534,
             427.273: 20.624137,
             429.571: 21.150628,
             436.147: 233.11958,
             436.611: 233.12144,
             443.15: 24.857254,
             445.944: 250.54088,
             447.21: 17.852553,
             447.25: 32.627653,
             447.285: 17.852994,
             448.705: 30.269855,
             449.08: 202.80222,
             449.773: 30.276908,
             450.481: 30.276831,
             459.73: 232.65454,
             461.113: 21.268987,
             463.249: 30.269703,
             464.713: 250.65717,
             465.815: 250.66419,
             468.5: 253.3251,
             471.611: 24.514172,
             471.97: 250.69717,
             477.33: 250.65717,
             478.42: 18.678179,
             481.277: 250.54088,
             491.743: 30.107049,
             491.843: 30.107534,
             494.037: 19.178634,
             494.46: 330.7779940153,
             498.866: 32.945545,
             512.579: 21.582341,
             515.785: 33.028396,
             516.596: 33.032203,
             522.686: 33.028396,
             526.312: 21.685248,
             529.14: 32.945545,
             534.765: 20.821324,
             539.317: 21.150628,
             550.456: 5.933527,
             556.318: 5.933527,
             563.306: 4.96428874,
             564.432: 5.933527,
             576.194: 5.933527,
             581.84: 8.991693,
             582.107: 8.993138,
             582.233: 8.993138,
             594.263: 5.9334878,
             594.272: 5.933527,
             601.346: 21.268987,
             608.044: 15.82797,
             612.502: 218.65175,
             612.79: 218.65175,
             614.892: 21.582341,
             624.457: 5.9334878,
             624.466: 5.933527,
             628.547: 19.178634,
             634.915: 21.685248,
             635.681: 21.687549,
             657.13: 22.53607,
             671.765: 21.765054,
             678.612: 21.150628,
             703.02: 16.089904,
             703.19: 16.089904,
             703.254: 16.089904,
             783.525: 22.34219,
             784.141: 22.34219,
             821.183: 5.9334878,
             821.204: 5.933527}

carbon_spd = {391.898: 16.33174,
              392.069: 16.333122,
              405.606: 40.197224,
              406.794: 39.923366,
              406.891: 39.924098,
              407.026: 39.925138,
              416.286: 40.064293,
              418.69: 40.00995,
              426.7: 18.045809,
              426.726: 18.045986,
              432.556: 38.435883,
              464.742: 29.534647,
              465.025: 29.534647,
              465.147: 29.534647,
              466.586: 38.22647,
              477.175: 7.487795,
              493.205: 7.684766,
              505.217: 7.684766,
              513.294: 20.701285,
              513.328: 20.704211,
              514.349: 20.704211,
              514.516: 20.709788,
              515.109: 20.709788,
              538.034: 7.684766,
              564.807: 20.704211,
              566.247: 20.709788,
              569.592: 32.103387,
              580.133: 37.54849,
              581.198: 37.54849,
              582.642: 40.197224,
              588.977: 18.045986,
              589.159: 18.045809,
              600.113: 8.64302,
              600.603: 8.647157,
              600.718: 8.640394,
              601.068: 8.640394,
              601.322: 8.647157,
              601.484: 8.64302,
              657.805: 14.448826,
              658.288: 14.448826,
              658.761: 8.537097,
              674.438: 38.22647,
              678.39: 20.709788,
              703.725: 38.435883,
              711.318: 8.647157,
              711.519: 8.640394,
              711.563: 22.532398,
              711.699: 8.647157,
              711.99: 22.536906,
              723.132: 16.33174,
              723.642: 16.333122,
              761.265: 41.342187,
              786.089: 8.850659,
              805.862: 8.850659}

single_spectral_power_distribution_plot(color.SpectralPowerDistribution(name="Custom",
                                                                        spd=sample_spd_data))

single_spectral_power_distribution_plot(color.SpectralPowerDistribution(name="Boron",
                                                                        spd=boron_spd))

# Plotting multiple relative spectral power distributions.
multi_spectral_power_distribution_plot([color.SpectralPowerDistribution(name="Boron",
                                                                        spd=boron_spd),
                                        color.SpectralPowerDistribution(name="Carbon",
                                                                        spd=carbon_spd)],
                                       y_label="Ei (eV)")

# Spectral bandpass correction.
bandpass_corrected_carbon_spd = color.bandpass_correction(
    color.SpectralPowerDistribution("Carbon (Bandpass Corrected)", carbon_spd), method="Stearns")
multi_spectral_power_distribution_plot([color.SpectralPowerDistribution("Carbon", carbon_spd),
                                        bandpass_corrected_carbon_spd],
                                       title="Stearns Bandpass Correction")

# Plotting given single standard observer 'CIE XYZ' or 'CIE RGB' color matching functions.
print(sorted(color.STANDARD_OBSERVERS_COLOR_MATCHING_FUNCTIONS.keys()))
single_color_matching_functions_plot("Standard CIE 1931 2 Degree Observer")
single_color_matching_functions_plot("Standard CIE 1964 10 Degree Observer")
single_color_matching_functions_plot("Stiles & Burch 1955 2 Degree Observer",
                                     legend_location="upper left",
                                     bounding_box=[390, 830, -0.5, 3.5])
single_color_matching_functions_plot("Stiles & Burch 1959 10 Degree Observer",
                                     legend_location="upper left",
                                     bounding_box=[390, 830, -0.5, 3.5])

# Comparing given multiple standard observers degrees 'CIE XYZ* color matching functions.
multi_color_matching_functions_plot(["Standard CIE 1931 2 Degree Observer",
                                     "Standard CIE 1964 10 Degree Observer"])
multi_color_matching_functions_plot(["Standard CIE 2006 10 Degree Observer",
                                     "Standard CIE 1964 10 Degree Observer"])
multi_color_matching_functions_plot(["Stiles & Burch 1955 2 Degree Observer",
                                     "Stiles & Burch 1959 10 Degree Observer"])

# Plotting visible colors under given standard observer.
visible_spectrum_plot("Standard CIE 1931 2 Degree Observer")
visible_spectrum_plot("Standard CIE 2006 2 Degree Observer")

# Plotting photopic luminous efficiency functions.
multi_spectral_power_distribution_plot(color.PHOTOPIC_LUMINOUS_EFFICIENCY_FUNCTIONS.values(),
                                       title="Luminous Efficiency Functions",
                                       y_label="Luminous Efficiency",
                                       legend_location="upper right",
                                       y_tighten=True,
                                       margins=[0., 150., 0., .1])

# Comparing photopic and scotopic luminous efficiency functions.
multi_spectral_power_distribution_plot(
    [color.PHOTOPIC_LUMINOUS_EFFICIENCY_FUNCTIONS["Stockman, Jagle, Pirzer & Sharpe CIE 2008 2 Degree Observer"],
     color.SCOTOPIC_LUMINOUS_EFFICIENCY_FUNCTIONS["Wald & Crawford CIE 1951 2 Degree Observer"]],
    title="Photopic & Scotopic Luminous Efficiency Functions",
    y_label="Luminous Efficiency",
    legend_location="upper right",
    y_tighten=True,
    margins=[0., 150., 0., .1])

# Plotting *CIE 1931 Chromaticity Diagram*.
CIE_1931_chromaticity_diagram_plot()

# Plotting colorspaces in *CIE 1931 Chromaticity Diagram*.
print(sorted(color.COLORSPACES.keys()))
colorspaces_CIE_1931_chromaticity_diagram_plot(["sRGB", "ACES RGB", "Adobe RGB 1998"])

# Plotting a single custom colorspace in *CIE 1931 Chromaticity Diagram*.
color.COLORSPACES["Awful RGB"] = color.Colorspace("Awful RGB",
                                                  primaries=matrix([[0.1, 0.2], [0.3, 0.15], [0.05, 0.6]]),
                                                  whitepoint=(1. / 3., 1. / 3.))
print(sorted(color.COLORSPACES.keys()))
colorspaces_CIE_1931_chromaticity_diagram_plot(["sRGB", "Awful RGB"])

# Plotting planckian locus in *CIE 1931 Chromaticity Diagram*.
planckian_locus_CIE_1931_chromaticity_diagram_plot()

# Plotting *CIE 1960 UCS Chromaticity Diagram*.
CIE_1960_UCS_chromaticity_diagram_plot()

# Plotting planckian locus in *CIE 1960 UCS Chromaticity Diagram*.
planckian_locus_CIE_1960_UCS_chromaticity_diagram_plot()

# Plotting *CIE 1976 UCS Chromaticity Diagram*.
CIE_1976_UCS_chromaticity_diagram_plot()

# Plotting a single *Munsell value* function.
single_munsell_value_function_plot("Munsell Value 1955")

# Plotting multiple *Munsell value* functions.
multi_munsell_value_function_plot(["Munsell Value 1955", "Munsell Value 1944"])

# Plotting a single *Lightness* function.
single_lightness_function_plot("Lightness 1976")

# Plotting multiple *Lightness* functions.
multi_lightness_function_plot(["Lightness 1976", "Lightness 1958"])

# Plotting a single colorspace transfer function.
single_transfer_function_plot("sRGB")

# Plotting multiple colorspaces transfer functions.
multi_transfer_function_plot(["sRGB", "Rec. 709"])

# Plotting various blackbody spectral radiance.
blackbody_spectral_radiance_plot(temperature=3500, blackbody="VY Canis Major")
blackbody_spectral_radiance_plot(temperature=5778, blackbody="The Sun")
blackbody_spectral_radiance_plot(temperature=12130, blackbody="Rigel")

# Comparing theoretical and experiment *Sun* spectral distributions.
ASTM_G_173 = {280.0: 8.2000e-02,
              280.5: 9.9000e-02,
              281.0: 1.5000e-01,
              281.5: 2.1200e-01,
              282.0: 2.6700e-01,
              282.5: 3.0300e-01,
              283.0: 3.2500e-01,
              283.5: 3.2300e-01,
              284.0: 2.9900e-01,
              284.5: 2.5024e-01,
              285.0: 1.7589e-01,
              285.5: 1.5500e-01,
              286.0: 2.4200e-01,
              286.5: 3.3300e-01,
              287.0: 3.6200e-01,
              287.5: 3.3900e-01,
              288.0: 3.1100e-01,
              288.5: 3.2500e-01,
              289.0: 3.9200e-01,
              289.5: 4.7900e-01,
              290.0: 5.6300e-01,
              290.5: 6.0600e-01,
              291.0: 6.1800e-01,
              291.5: 5.9800e-01,
              292.0: 5.6700e-01,
              292.5: 5.2900e-01,
              293.0: 5.3800e-01,
              293.5: 5.4900e-01,
              294.0: 5.3300e-01,
              294.5: 5.2000e-01,
              295.0: 5.2700e-01,
              295.5: 5.5900e-01,
              296.0: 5.7300e-01,
              296.5: 5.2100e-01,
              297.0: 4.7800e-01,
              297.5: 5.2900e-01,
              298.0: 5.2800e-01,
              298.5: 4.6800e-01,
              299.0: 4.7200e-01,
              299.5: 5.0100e-01,
              300.0: 4.5794e-01,
              300.5: 4.3300e-01,
              301.0: 4.6300e-01,
              301.5: 4.7756e-01,
              302.0: 4.4900e-01,
              302.5: 5.0800e-01,
              303.0: 6.1200e-01,
              303.5: 6.4600e-01,
              304.0: 6.2100e-01,
              304.5: 6.2600e-01,
              305.0: 6.4200e-01,
              305.5: 6.1100e-01,
              306.0: 5.6500e-01,
              306.5: 5.7500e-01,
              307.0: 6.0500e-01,
              307.5: 6.3100e-01,
              308.0: 6.4500e-01,
              308.5: 6.4050e-01,
              309.0: 5.8000e-01,
              309.5: 5.0000e-01,
              310.0: 5.3300e-01,
              310.5: 6.5200e-01,
              311.0: 7.6200e-01,
              311.5: 7.4400e-01,
              312.0: 7.0615e-01,
              312.5: 6.7600e-01,
              313.0: 6.9400e-01,
              313.5: 7.2400e-01,
              314.0: 7.1700e-01,
              314.5: 6.7600e-01,
              315.0: 6.8450e-01,
              315.5: 6.3200e-01,
              316.0: 5.8700e-01,
              316.5: 6.4900e-01,
              317.0: 7.3900e-01,
              317.5: 8.0189e-01,
              318.0: 7.2361e-01,
              318.5: 6.8000e-01,
              319.0: 7.0580e-01,
              319.5: 7.3318e-01,
              320.0: 7.7500e-01,
              320.5: 8.2600e-01,
              321.0: 7.6546e-01,
              321.5: 7.1900e-01,
              322.0: 7.3513e-01,
              322.5: 7.1152e-01,
              323.0: 6.4900e-01,
              323.5: 6.8138e-01,
              324.0: 7.4050e-01,
              324.5: 7.8800e-01,
              325.0: 8.2918e-01,
              325.5: 9.1377e-01,
              326.0: 9.9835e-01,
              326.5: 1.0166E+00,
              327.0: 1.0047E+00,
              327.5: 9.8679e-01,
              328.0: 9.5681e-01,
              328.5: 9.3400e-01,
              329.0: 1.0046E+00,
              329.5: 1.0958E+00,
              330.0: 1.1098E+00,
              330.5: 1.0429E+00,
              331.0: 9.9100e-01,
              331.5: 9.9396e-01,
              332.0: 9.9277e-01,
              332.5: 9.8003e-01,
              333.0: 9.6394e-01,
              333.5: 9.3700e-01,
              334.0: 9.5557e-01,
              334.5: 9.8862e-01,
              335.0: 1.0097E+00,
              335.5: 9.7453e-01,
              336.0: 8.8979e-01,
              336.5: 8.2900e-01,
              337.0: 8.1800e-01,
              337.5: 8.7732e-01,
              338.0: 9.2529e-01,
              338.5: 9.5783e-01,
              339.0: 9.6863e-01,
              339.5: 9.8900e-01,
              340.0: 1.0544E+00,
              340.5: 1.0463E+00,
              341.0: 9.7100e-01,
              341.5: 9.5900e-01,
              342.0: 9.9570e-01,
              342.5: 1.0277E+00,
              343.0: 1.0417E+00,
              343.5: 9.8301e-01,
              344.0: 8.5416e-01,
              344.5: 8.1300e-01,
              345.0: 9.1583e-01,
              345.5: 9.7021e-01,
              346.0: 9.4269e-01,
              346.5: 9.5594e-01,
              347.0: 9.6954e-01,
              347.5: 9.3318e-01,
              348.0: 9.2472e-01,
              348.5: 9.3721e-01,
              349.0: 8.9900e-01,
              349.5: 9.1969e-01,
              350.0: 1.0122E+00,
              350.5: 1.0849E+00,
              351.0: 1.0534E+00,
              351.5: 1.0129E+00,
              352.0: 9.8383e-01,
              352.5: 9.2600e-01,
              353.0: 9.8012e-01,
              353.5: 1.0752E+00,
              354.0: 1.1346E+00,
              354.5: 1.1440E+00,
              355.0: 1.1406E+00,
              355.5: 1.0964E+00,
              356.0: 1.0250E+00,
              356.5: 9.5914e-01,
              357.0: 8.4200e-01,
              357.5: 8.5015e-01,
              358.0: 7.8916e-01,
              358.5: 7.3100e-01,
              359.0: 8.5805e-01,
              359.5: 1.0321E+00,
              360.0: 1.0890E+00,
              360.5: 1.0265E+00,
              361.0: 9.4150e-01,
              361.5: 9.1800e-01,
              362.0: 9.5800e-01,
              362.5: 1.0450E+00,
              363.0: 1.0710E+00,
              363.5: 1.0380E+00,
              364.0: 1.0716E+00,
              364.5: 1.0590E+00,
              365.0: 1.0970E+00,
              365.5: 1.2041E+00,
              366.0: 1.2860E+00,
              366.5: 1.2843E+00,
              367.0: 1.2577E+00,
              367.5: 1.2317E+00,
              368.0: 1.1570E+00,
              368.5: 1.1459E+00,
              369.0: 1.1944E+00,
              369.5: 1.2795E+00,
              370.0: 1.2934E+00,
              370.5: 1.1660E+00,
              371.0: 1.1811E+00,
              371.5: 1.2249E+00,
              372.0: 1.1444E+00,
              372.5: 1.0876E+00,
              373.0: 1.0445E+00,
              373.5: 9.3900e-01,
              374.0: 9.3442e-01,
              374.5: 9.2500e-01,
              375.0: 9.8500e-01,
              375.5: 1.0874E+00,
              376.0: 1.1240E+00,
              376.5: 1.1040E+00,
              377.0: 1.1813E+00,
              377.5: 1.3149E+00,
              378.0: 1.4144E+00,
              378.5: 1.3765E+00,
              379.0: 1.2260E+00,
              379.5: 1.0980E+00,
              380.0: 1.1520E+00,
              380.5: 1.2310E+00,
              381.0: 1.2490E+00,
              381.5: 1.1224E+00,
              382.0: 9.5426e-01,
              382.5: 8.2313e-01,
              383.0: 7.3603e-01,
              383.5: 7.1095e-01,
              384.0: 8.2100e-01,
              384.5: 9.8620e-01,
              385.0: 1.0802E+00,
              385.5: 1.0296E+00,
              386.0: 9.9113e-01,
              386.5: 1.0279E+00,
              387.0: 1.0354E+00,
              387.5: 1.0186E+00,
              388.0: 1.0067E+00,
              388.5: 9.9743e-01,
              389.0: 1.0810E+00,
              389.5: 1.1958E+00,
              390.0: 1.2519E+00,
              390.5: 1.2601E+00,
              391.0: 1.3322E+00,
              391.5: 1.3490E+00,
              392.0: 1.2400E+00,
              392.5: 1.0312E+00,
              393.0: 7.4500e-01,
              393.5: 5.9120e-01,
              394.0: 7.6675e-01,
              394.5: 1.0557E+00,
              395.0: 1.2450E+00,
              395.5: 1.3245E+00,
              396.0: 1.1626E+00,
              396.5: 8.4330e-01,
              397.0: 6.5200e-01,
              397.5: 9.6142e-01,
              398.0: 1.3010E+00,
              398.5: 1.5342E+00,
              399.0: 1.6245E+00,
              399.5: 1.6717E+00,
              400.0: 1.6885E+00,
              401.0: 1.7520E+00,
              402.0: 1.8140E+00,
              403.0: 1.7400E+00,
              404.0: 1.7630E+00,
              405.0: 1.7150E+00,
              406.0: 1.6660E+00,
              407.0: 1.6300E+00,
              408.0: 1.6990E+00,
              409.0: 1.8090E+00,
              410.0: 1.5370E+00,
              411.0: 1.7150E+00,
              412.0: 1.8160E+00,
              413.0: 1.7392E+00,
              414.0: 1.7144E+00,
              415.0: 1.7688E+00,
              416.0: 1.8150E+00,
              417.0: 1.7660E+00,
              418.0: 1.6850E+00,
              419.0: 1.7490E+00,
              420.0: 1.5990E+00,
              421.0: 1.8110E+00,
              422.0: 1.7820E+00,
              423.0: 1.7210E+00,
              424.0: 1.7080E+00,
              425.0: 1.7550E+00,
              426.0: 1.6990E+00,
              427.0: 1.6380E+00,
              428.0: 1.6510E+00,
              429.0: 1.5230E+00,
              430.0: 1.2120E+00,
              431.0: 1.0990E+00,
              432.0: 1.8220E+00,
              433.0: 1.6913E+00,
              434.0: 1.5600E+00,
              435.0: 1.7090E+00,
              436.0: 1.8680E+00,
              437.0: 1.9000E+00,
              438.0: 1.6630E+00,
              439.0: 1.6010E+00,
              440.0: 1.8300E+00,
              441.0: 1.7990E+00,
              442.0: 1.9220E+00,
              443.0: 1.9490E+00,
              444.0: 1.8941E+00,
              445.0: 1.9650E+00,
              446.0: 1.7557E+00,
              447.0: 1.9900E+00,
              448.0: 2.0140E+00,
              449.0: 2.0010E+00,
              450.0: 2.0690E+00,
              451.0: 2.1420E+00,
              452.0: 2.0470E+00,
              453.0: 1.8864E+00,
              454.0: 2.0180E+00,
              455.0: 2.0010E+00,
              456.0: 2.0630E+00,
              457.0: 2.0770E+00,
              458.0: 2.0320E+00,
              459.0: 2.0120E+00,
              460.0: 1.9973E+00,
              461.0: 2.0639E+00,
              462.0: 2.0780E+00,
              463.0: 2.0840E+00,
              464.0: 2.0150E+00,
              465.0: 1.9840E+00,
              466.0: 2.0210E+00,
              467.0: 1.9310E+00,
              468.0: 2.0120E+00,
              469.0: 2.0180E+00,
              470.0: 1.9390E+00,
              471.0: 1.9690E+00,
              472.0: 2.0700E+00,
              473.0: 1.9882E+00,
              474.0: 2.0120E+00,
              475.0: 2.0800E+00,
              476.0: 2.0120E+00,
              477.0: 2.0250E+00,
              478.0: 2.0860E+00,
              479.0: 2.0400E+00,
              480.0: 2.0680E+00,
              481.0: 2.0610E+00,
              482.0: 2.0623E+00,
              483.0: 2.0310E+00,
              484.0: 1.9890E+00,
              485.0: 1.9790E+00,
              486.0: 1.6010E+00,
              487.0: 1.7890E+00,
              488.0: 1.9350E+00,
              489.0: 1.8224E+00,
              490.0: 2.0320E+00,
              491.0: 1.9490E+00,
              492.0: 1.8560E+00,
              493.0: 1.9830E+00,
              494.0: 1.9339E+00,
              495.0: 2.0510E+00,
              496.0: 1.9490E+00,
              497.0: 1.9800E+00,
              498.0: 1.9240E+00,
              499.0: 1.9230E+00,
              500.0: 1.9160E+00,
              501.0: 1.8580E+00,
              502.0: 1.8600E+00,
              503.0: 1.9490E+00,
              504.0: 1.8330E+00,
              505.0: 1.9472E+00,
              506.0: 2.0250E+00,
              507.0: 1.9354E+00,
              508.0: 1.8800E+00,
              509.0: 1.9650E+00,
              510.0: 1.9100E+00,
              511.0: 1.9410E+00,
              512.0: 1.9890E+00,
              513.0: 1.8660E+00,
              514.0: 1.8240E+00,
              515.0: 1.8750E+00,
              516.0: 1.8910E+00,
              517.0: 1.5390E+00,
              518.0: 1.7590E+00,
              519.0: 1.7040E+00,
              520.0: 1.8600E+00,
              521.0: 1.8730E+00,
              522.0: 1.9150E+00,
              523.0: 1.8040E+00,
              524.0: 1.9410E+00,
              525.0: 1.9280E+00,
              526.0: 1.8740E+00,
              527.0: 1.6410E+00,
              528.0: 1.8800E+00,
              529.0: 1.9690E+00,
              530.0: 1.8920E+00,
              531.0: 1.9950E+00,
              532.0: 1.9580E+00,
              533.0: 1.7470E+00,
              534.0: 1.8690E+00,
              535.0: 1.8950E+00,
              536.0: 1.9740E+00,
              537.0: 1.8240E+00,
              538.0: 1.9130E+00,
              539.0: 1.8640E+00,
              540.0: 1.8000E+00,
              541.0: 1.7340E+00,
              542.0: 1.8880E+00,
              543.0: 1.8510E+00,
              544.0: 1.9190E+00,
              545.0: 1.8740E+00,
              546.0: 1.8609E+00,
              547.0: 1.8820E+00,
              548.0: 1.8260E+00,
              549.0: 1.8800E+00,
              550.0: 1.8630E+00,
              551.0: 1.8590E+00,
              552.0: 1.8960E+00,
              553.0: 1.8420E+00,
              554.0: 1.8780E+00,
              555.0: 1.8890E+00,
              556.0: 1.8570E+00,
              557.0: 1.8120E+00,
              558.0: 1.8530E+00,
              559.0: 1.7550E+00,
              560.0: 1.7860E+00,
              561.0: 1.8900E+00,
              562.0: 1.8010E+00,
              563.0: 1.8710E+00,
              564.0: 1.8360E+00,
              565.0: 1.8490E+00,
              566.0: 1.7500E+00,
              567.0: 1.8680E+00,
              568.0: 1.8590E+00,
              569.0: 1.8310E+00,
              570.0: 1.8280E+00,
              571.0: 1.7620E+00,
              572.0: 1.8720E+00,
              573.0: 1.8810E+00,
              574.0: 1.8730E+00,
              575.0: 1.8340E+00,
              576.0: 1.8180E+00,
              577.0: 1.8620E+00,
              578.0: 1.7990E+00,
              579.0: 1.8160E+00,
              580.0: 1.8340E+00,
              581.0: 1.8330E+00,
              582.0: 1.8520E+00,
              583.0: 1.8630E+00,
              584.0: 1.8540E+00,
              585.0: 1.8360E+00,
              586.0: 1.7920E+00,
              587.0: 1.8380E+00,
              588.0: 1.8210E+00,
              589.0: 1.6240E+00,
              590.0: 1.7218E+00,
              591.0: 1.8090E+00,
              592.0: 1.7880E+00,
              593.0: 1.7920E+00,
              594.0: 1.7890E+00,
              595.0: 1.7780E+00,
              596.0: 1.7960E+00,
              597.0: 1.8060E+00,
              598.0: 1.7720E+00,
              599.0: 1.7640E+00,
              600.0: 1.7700E+00,
              601.0: 1.7420E+00,
              602.0: 1.7150E+00,
              603.0: 1.7490E+00,
              604.0: 1.7790E+00,
              605.0: 1.7730E+00,
              606.0: 1.7580E+00,
              607.0: 1.7620E+00,
              608.0: 1.7510E+00,
              609.0: 1.7340E+00,
              610.0: 1.7240E+00,
              611.0: 1.7120E+00,
              612.0: 1.7360E+00,
              613.0: 1.7100E+00,
              614.0: 1.6550E+00,
              615.0: 1.7120E+00,
              616.0: 1.6640E+00,
              617.0: 1.6410E+00,
              618.0: 1.7020E+00,
              619.0: 1.7090E+00,
              620.0: 1.7110E+00,
              621.0: 1.7240E+00,
              622.0: 1.6784E+00,
              623.0: 1.6820E+00,
              624.0: 1.6670E+00,
              625.0: 1.6440E+00,
              626.0: 1.6400E+00,
              627.0: 1.6930E+00,
              628.0: 1.6930E+00,
              629.0: 1.6870E+00,
              630.0: 1.6650E+00,
              631.0: 1.6590E+00,
              632.0: 1.5901E+00,
              633.0: 1.6740E+00,
              634.0: 1.6370E+00,
              635.0: 1.6520E+00,
              636.0: 1.6093E+00,
              637.0: 1.6610E+00,
              638.0: 1.6650E+00,
              639.0: 1.6530E+00,
              640.0: 1.6130E+00,
              641.0: 1.6100E+00,
              642.0: 1.6090E+00,
              643.0: 1.6250E+00,
              644.0: 1.6140E+00,
              645.0: 1.6270E+00,
              646.0: 1.5910E+00,
              647.0: 1.6060E+00,
              648.0: 1.6020E+00,
              649.0: 1.5510E+00,
              650.0: 1.5260E+00,
              651.0: 1.6130E+00,
              652.0: 1.5910E+00,
              653.0: 1.5980E+00,
              654.0: 1.5750E+00,
              655.0: 1.5230E+00,
              656.0: 1.3233E+00,
              657.0: 1.3840E+00,
              658.0: 1.5390E+00,
              659.0: 1.5420E+00,
              660.0: 1.5580E+00,
              661.0: 1.5660E+00,
              662.0: 1.5710E+00,
              663.0: 1.5630E+00,
              664.0: 1.5540E+00,
              665.0: 1.5670E+00,
              666.0: 1.5550E+00,
              667.0: 1.5354E+00,
              668.0: 1.5348E+00,
              669.0: 1.5580E+00,
              670.0: 1.5340E+00,
              671.0: 1.5290E+00,
              672.0: 1.5060E+00,
              673.0: 1.5170E+00,
              674.0: 1.5130E+00,
              675.0: 1.4990E+00,
              676.0: 1.5150E+00,
              677.0: 1.5000E+00,
              678.0: 1.5070E+00,
              679.0: 1.4930E+00,
              680.0: 1.4940E+00,
              681.0: 1.4870E+00,
              682.0: 1.4930E+00,
              683.0: 1.4760E+00,
              684.0: 1.4660E+00,
              685.0: 1.4650E+00,
              686.0: 1.4330E+00,
              687.0: 1.4720E+00,
              688.0: 1.4760E+00,
              689.0: 1.4780E+00,
              690.0: 1.4790E+00,
              691.0: 1.4680E+00,
              692.0: 1.4540E+00,
              693.0: 1.4580E+00,
              694.0: 1.4570E+00,
              695.0: 1.4350E+00,
              696.0: 1.4420E+00,
              697.0: 1.4380E+00,
              698.0: 1.4170E+00,
              699.0: 1.4340E+00,
              700.0: 1.4220E+00,
              701.0: 1.4131E+00,
              702.0: 1.3987E+00,
              703.0: 1.4095E+00,
              704.0: 1.4187E+00,
              705.0: 1.4330E+00,
              706.0: 1.4138E+00,
              707.0: 1.4040E+00,
              708.0: 1.3990E+00,
              709.0: 1.3900E+00,
              710.0: 1.4040E+00,
              711.0: 1.3970E+00,
              712.0: 1.3818E+00,
              713.0: 1.3702E+00,
              714.0: 1.3819E+00,
              715.0: 1.3502E+00,
              716.0: 1.3694E+00,
              717.0: 1.3650E+00,
              718.0: 1.3570E+00,
              719.0: 1.3010E+00,
              720.0: 1.3487E+00,
              721.0: 1.3480E+00,
              722.0: 1.3600E+00,
              723.0: 1.3510E+00,
              724.0: 1.3607E+00,
              725.0: 1.3465E+00,
              726.0: 1.3429E+00,
              727.0: 1.3444E+00,
              728.0: 1.3370E+00,
              729.0: 1.2796E+00,
              730.0: 1.3357E+00,
              731.0: 1.3104E+00,
              732.0: 1.3333E+00,
              733.0: 1.3108E+00,
              734.0: 1.3390E+00,
              735.0: 1.3271E+00,
              736.0: 1.3100E+00,
              737.0: 1.3120E+00,
              738.0: 1.3000E+00,
              739.0: 1.2646E+00,
              740.0: 1.2830E+00,
              741.0: 1.2707E+00,
              742.0: 1.2649E+00,
              743.0: 1.2892E+00,
              744.0: 1.2955E+00,
              745.0: 1.2920E+00,
              746.0: 1.2892E+00,
              747.0: 1.2890E+00,
              748.0: 1.2808E+00,
              749.0: 1.2760E+00,
              750.0: 1.2740E+00,
              751.0: 1.2680E+00,
              752.0: 1.2720E+00,
              753.0: 1.2650E+00,
              754.0: 1.2809E+00,
              755.0: 1.2771E+00,
              756.0: 1.2610E+00,
              757.0: 1.2598E+00,
              758.0: 1.2680E+00,
              759.0: 1.2500E+00,
              760.0: 1.2590E+00,
              761.0: 1.2487E+00,
              762.0: 1.2460E+00,
              763.0: 1.2543E+00,
              764.0: 1.2566E+00,
              765.0: 1.2452E+00,
              766.0: 1.2000E+00,
              767.0: 1.2120E+00,
              768.0: 1.2310E+00,
              769.0: 1.2142E+00,
              770.0: 1.2146E+00,
              771.0: 1.2073E+00,
              772.0: 1.2120E+00,
              773.0: 1.2100E+00,
              774.0: 1.2090E+00,
              775.0: 1.2080E+00,
              776.0: 1.2102E+00,
              777.0: 1.2028E+00,
              778.0: 1.2014E+00,
              779.0: 1.2067E+00,
              780.0: 1.1930E+00,
              781.0: 1.1898E+00,
              782.0: 1.1950E+00,
              783.0: 1.1896E+00,
              784.0: 1.1810E+00,
              785.0: 1.1874E+00,
              786.0: 1.1890E+00,
              787.0: 1.1877E+00,
              788.0: 1.1800E+00,
              789.0: 1.1812E+00,
              790.0: 1.1704E+00,
              791.0: 1.1632E+00,
              792.0: 1.1500E+00,
              793.0: 1.1449E+00,
              794.0: 1.1260E+00,
              795.0: 1.1280E+00,
              796.0: 1.1427E+00,
              797.0: 1.1580E+00,
              798.0: 1.1501E+00,
              799.0: 1.1350E+00,
              800.0: 1.1248E+00,
              801.0: 1.1427E+00,
              802.0: 1.1422E+00,
              803.0: 1.1241E+00,
              804.0: 1.1342E+00,
              805.0: 1.1001E+00,
              806.0: 1.1309E+00,
              807.0: 1.1230E+00,
              808.0: 1.1162E+00,
              809.0: 1.0980E+00,
              810.0: 1.1100E+00,
              811.0: 1.1143E+00,
              812.0: 1.1160E+00,
              813.0: 1.1180E+00,
              814.0: 1.1174E+00,
              815.0: 1.1091E+00,
              816.0: 1.1050E+00,
              817.0: 1.1083E+00,
              818.0: 1.0861E+00,
              819.0: 1.0738E+00,
              820.0: 1.0740E+00,
              821.0: 1.0760E+00,
              822.0: 1.0560E+00,
              823.0: 1.0760E+00,
              824.0: 1.0715E+00,
              825.0: 1.0699E+00,
              826.0: 1.0759E+00,
              827.0: 1.0762E+00,
              828.0: 1.0749E+00,
              829.0: 1.0620E+00,
              830.0: 1.0563E+00,
              831.0: 1.0560E+00,
              832.0: 1.0550E+00,
              833.0: 1.0327E+00,
              834.0: 1.0459E+00,
              835.0: 1.0476E+00,
              836.0: 1.0597E+00,
              837.0: 1.0511E+00,
              838.0: 1.0549E+00,
              839.0: 1.0400E+00,
              840.0: 1.0500E+00,
              841.0: 1.0520E+00,
              842.0: 1.0287E+00,
              843.0: 1.0271E+00,
              844.0: 1.0140E+00,
              845.0: 1.0360E+00,
              846.0: 1.0370E+00,
              847.0: 1.0060E+00,
              848.0: 1.0110E+00,
              849.0: 1.0050E+00,
              850.0: 9.1000e-01,
              851.0: 9.9800e-01,
              852.0: 9.9000e-01,
              853.0: 9.8100e-01,
              854.0: 8.6900e-01,
              855.0: 9.2700e-01,
              856.0: 9.9017e-01,
              857.0: 1.0056E+00,
              858.0: 1.0070E+00,
              859.0: 1.0050E+00,
              860.0: 1.0000E+00,
              861.0: 9.9900e-01,
              862.0: 1.0060E+00,
              863.0: 1.0120E+00,
              864.0: 9.9006e-01,
              865.0: 9.7354e-01,
              866.0: 8.5800e-01,
              867.0: 9.2500e-01,
              868.0: 9.6900e-01,
              869.0: 9.5900e-01,
              870.0: 9.7700e-01,
              871.0: 9.6317e-01,
              872.0: 9.7600e-01,
              873.0: 9.6614e-01,
              874.0: 9.4913e-01,
              875.0: 9.3562e-01,
              876.0: 9.6200e-01,
              877.0: 9.6500e-01,
              878.0: 9.6200e-01,
              879.0: 9.4571e-01,
              880.0: 9.4856e-01,
              881.0: 9.1963e-01,
              882.0: 9.4408e-01,
              883.0: 9.3930e-01,
              884.0: 9.4352e-01,
              885.0: 9.5469e-01,
              886.0: 9.2558e-01,
              887.0: 9.2300e-01,
              888.0: 9.4066e-01,
              889.0: 9.4367e-01,
              890.0: 9.4236e-01,
              891.0: 9.3928e-01,
              892.0: 9.3259e-01,
              893.0: 9.1800e-01,
              894.0: 9.2400e-01,
              895.0: 9.2600e-01,
              896.0: 9.3425e-01,
              897.0: 9.2686e-01,
              898.0: 9.2378e-01,
              899.0: 9.1396e-01,
              900.0: 9.1378e-01,
              901.0: 8.9834e-01,
              902.0: 8.7720e-01,
              903.0: 9.2233e-01,
              904.0: 9.2100e-01,
              905.0: 9.1800e-01,
              906.0: 9.0780e-01,
              907.0: 9.1450e-01,
              908.0: 9.0143e-01,
              909.0: 8.8552e-01,
              910.0: 8.9496e-01,
              911.0: 8.9730e-01,
              912.0: 8.9016e-01,
              913.0: 9.0020e-01,
              914.0: 8.9820e-01,
              915.0: 8.8830e-01,
              916.0: 8.9470e-01,
              917.0: 8.8860e-01,
              918.0: 8.8810e-01,
              919.0: 8.9210e-01,
              920.0: 8.8540e-01,
              921.0: 8.7030e-01,
              922.0: 8.6280e-01,
              923.0: 8.3335e-01,
              924.0: 8.6957e-01,
              925.0: 8.7642e-01,
              926.0: 8.4561e-01,
              927.0: 8.7809e-01,
              928.0: 8.7110e-01,
              929.0: 8.6954e-01,
              930.0: 8.6866e-01,
              931.0: 8.7078e-01,
              932.0: 8.6514e-01,
              933.0: 8.6958e-01,
              934.0: 8.6709e-01,
              935.0: 8.5601e-01,
              936.0: 8.5222e-01,
              937.0: 8.5368e-01,
              938.0: 8.5936e-01,
              939.0: 8.5676e-01,
              940.0: 8.4000e-01,
              941.0: 8.3153e-01,
              942.0: 8.0616e-01,
              943.0: 8.3814e-01,
              944.0: 8.1854e-01,
              945.0: 8.2441e-01,
              946.0: 8.3138e-01,
              947.0: 8.2809e-01,
              948.0: 8.3137e-01,
              949.0: 8.3303e-01,
              950.0: 8.2867e-01,
              951.0: 8.2019e-01,
              952.0: 8.2402e-01,
              953.0: 8.2050e-01,
              954.0: 8.1200e-01,
              955.0: 7.6900e-01,
              956.0: 8.0200e-01,
              957.0: 8.0583e-01,
              958.0: 8.1147e-01,
              959.0: 8.1483e-01,
              960.0: 8.0627e-01,
              961.0: 7.9982e-01,
              962.0: 8.0173e-01,
              963.0: 7.9761e-01,
              964.0: 7.8808e-01,
              965.0: 7.9579e-01,
              966.0: 7.8598e-01,
              967.0: 7.9503e-01,
              968.0: 7.8526e-01,
              969.0: 7.8789e-01,
              970.0: 7.9140e-01,
              971.0: 7.8186e-01,
              972.0: 7.8785e-01,
              973.0: 7.8149e-01,
              974.0: 7.6882e-01,
              975.0: 7.7348e-01,
              976.0: 7.8054e-01,
              977.0: 7.7208e-01,
              978.0: 7.7800e-01,
              979.0: 7.6650e-01,
              980.0: 7.7512e-01,
              981.0: 7.7300e-01,
              982.0: 7.7879e-01,
              983.0: 7.7541e-01,
              984.0: 7.6881e-01,
              985.0: 7.7081e-01,
              986.0: 7.6620e-01,
              987.0: 7.6606e-01,
              988.0: 7.6368e-01,
              989.0: 7.5500e-01,
              990.0: 7.5694e-01,
              991.0: 7.6161e-01,
              992.0: 7.5714e-01,
              993.0: 7.5639e-01,
              994.0: 7.5851e-01,
              995.0: 7.5392e-01,
              996.0: 7.5347e-01,
              997.0: 7.4300e-01,
              998.0: 7.4300e-01,
              999.0: 7.4200e-01,
              1000.0: 7.4255e-01,
              1001.0: 7.4741e-01,
              1002.0: 7.4677e-01,
              1003.0: 7.3836e-01,
              1004.0: 7.3133e-01,
              1005.0: 6.8365e-01,
              1006.0: 7.1770e-01,
              1007.0: 7.3093e-01,
              1008.0: 7.3339e-01,
              1009.0: 7.2934e-01,
              1010.0: 7.3055e-01,
              1011.0: 7.2562e-01,
              1012.0: 7.2496e-01,
              1013.0: 7.1960e-01,
              1014.0: 7.2265e-01,
              1015.0: 7.1062e-01,
              1016.0: 7.1419e-01,
              1017.0: 7.0659e-01,
              1018.0: 7.1566e-01,
              1019.0: 6.9606e-01,
              1020.0: 7.0127e-01,
              1021.0: 7.0300e-01,
              1022.0: 6.9300e-01,
              1023.0: 6.9900e-01,
              1024.0: 7.0057e-01,
              1025.0: 6.9918e-01,
              1026.0: 6.9943e-01,
              1027.0: 6.9594e-01,
              1028.0: 6.9722e-01,
              1029.0: 6.9000e-01,
              1030.0: 6.9208e-01,
              1031.0: 6.8985e-01,
              1032.0: 6.8960e-01,
              1033.0: 6.7800e-01,
              1034.0: 6.8181e-01,
              1035.0: 6.8400e-01,
              1036.0: 6.8431e-01,
              1037.0: 6.7731e-01,
              1038.0: 6.7400e-01,
              1039.0: 6.7900e-01,
              1040.0: 6.7440e-01,
              1041.0: 6.7490e-01,
              1042.0: 6.7516e-01,
              1043.0: 6.6860e-01,
              1044.0: 6.7199e-01,
              1045.0: 6.6846e-01,
              1046.0: 6.5131e-01,
              1047.0: 6.6155e-01,
              1048.0: 6.6781e-01,
              1049.0: 6.6491e-01,
              1050.0: 6.6117e-01,
              1051.0: 6.6182e-01,
              1052.0: 6.5848e-01,
              1053.0: 6.5698e-01,
              1054.0: 6.5474e-01,
              1055.0: 6.5753e-01,
              1056.0: 6.5629e-01,
              1057.0: 6.5540e-01,
              1058.0: 6.4937e-01,
              1059.0: 6.3046e-01,
              1060.0: 6.4831e-01,
              1061.0: 6.3387e-01,
              1062.0: 6.4651e-01,
              1063.0: 6.3669e-01,
              1064.0: 6.4621e-01,
              1065.0: 6.4533e-01,
              1066.0: 6.3139e-01,
              1067.0: 6.3517e-01,
              1068.0: 6.3591e-01,
              1069.0: 6.0084e-01,
              1070.0: 6.2165e-01,
              1071.0: 6.2821e-01,
              1072.0: 6.3131e-01,
              1073.0: 6.1796e-01,
              1074.0: 6.3063e-01,
              1075.0: 6.1768e-01,
              1076.0: 6.2468e-01,
              1077.0: 6.2426e-01,
              1078.0: 6.2531e-01,
              1079.0: 6.1203e-01,
              1080.0: 6.2250e-01,
              1081.0: 6.0726e-01,
              1082.0: 6.0249e-01,
              1083.0: 6.0900e-01,
              1084.0: 6.1392e-01,
              1085.0: 6.1181e-01,
              1086.0: 6.1570e-01,
              1087.0: 5.9145e-01,
              1088.0: 6.0728e-01,
              1089.0: 6.0908e-01,
              1090.0: 6.0442e-01,
              1091.0: 6.0799e-01,
              1092.0: 5.9692e-01,
              1093.0: 5.9221e-01,
              1094.0: 5.6065e-01,
              1095.0: 5.8252e-01,
              1096.0: 5.8600e-01,
              1097.0: 5.9100e-01,
              1098.0: 5.8700e-01,
              1099.0: 5.8216e-01,
              1100.0: 6.0000e-01,
              1101.0: 5.9900e-01,
              1102.0: 5.8202e-01,
              1103.0: 5.9500e-01,
              1104.0: 5.9300e-01,
              1105.0: 5.9100e-01,
              1106.0: 5.9000e-01,
              1107.0: 5.8600e-01,
              1108.0: 5.8650e-01,
              1109.0: 5.8550e-01,
              1110.0: 5.8700e-01,
              1111.0: 5.8250e-01,
              1112.0: 5.8342e-01,
              1113.0: 5.7800e-01,
              1114.0: 5.8400e-01,
              1115.0: 5.8200e-01,
              1116.0: 5.7500e-01,
              1117.0: 5.8300e-01,
              1118.0: 5.7222e-01,
              1119.0: 5.7151e-01,
              1120.0: 5.6850e-01,
              1121.0: 5.7200e-01,
              1122.0: 5.7500e-01,
              1123.0: 5.7400e-01,
              1124.0: 5.7250e-01,
              1125.0: 5.7012e-01,
              1126.0: 5.5300e-01,
              1127.0: 5.6179e-01,
              1128.0: 5.6829e-01,
              1129.0: 5.6837e-01,
              1130.0: 5.6401e-01,
              1131.0: 5.6800e-01,
              1132.0: 5.7000e-01,
              1133.0: 5.6186e-01,
              1134.0: 5.6346e-01,
              1135.0: 5.6250e-01,
              1136.0: 5.6481e-01,
              1137.0: 5.5197e-01,
              1138.0: 5.4400e-01,
              1139.0: 5.5300e-01,
              1140.0: 5.5573e-01,
              1141.0: 5.4303e-01,
              1142.0: 5.5531e-01,
              1143.0: 5.4960e-01,
              1144.0: 5.4501e-01,
              1145.0: 5.5001e-01,
              1146.0: 5.5407e-01,
              1147.0: 5.5408e-01,
              1148.0: 5.5032e-01,
              1149.0: 5.4788e-01,
              1150.0: 5.4623e-01,
              1151.0: 5.4531e-01,
              1152.0: 5.4170e-01,
              1153.0: 5.4308e-01,
              1154.0: 5.4651e-01,
              1155.0: 5.4731e-01,
              1156.0: 5.4022e-01,
              1157.0: 5.4370e-01,
              1158.0: 5.4334e-01,
              1159.0: 5.3548e-01,
              1160.0: 5.2928e-01,
              1161.0: 5.1899e-01,
              1162.0: 5.3336e-01,
              1163.0: 5.3708e-01,
              1164.0: 5.2684e-01,
              1165.0: 5.3482e-01,
              1166.0: 5.2227e-01,
              1167.0: 5.2984e-01,
              1168.0: 5.3415e-01,
              1169.0: 5.1640e-01,
              1170.0: 5.2875e-01,
              1171.0: 5.3074e-01,
              1172.0: 5.3063e-01,
              1173.0: 5.2686e-01,
              1174.0: 5.2760e-01,
              1175.0: 5.1852e-01,
              1176.0: 5.1036e-01,
              1177.0: 5.2296e-01,
              1178.0: 5.1703e-01,
              1179.0: 5.1504e-01,
              1180.0: 5.2199e-01,
              1181.0: 5.1843e-01,
              1182.0: 5.1743e-01,
              1183.0: 4.8821e-01,
              1184.0: 5.1098e-01,
              1185.0: 5.1605e-01,
              1186.0: 5.1519e-01,
              1187.0: 5.1455e-01,
              1188.0: 5.1012e-01,
              1189.0: 4.8402e-01,
              1190.0: 5.0583e-01,
              1191.0: 5.1354e-01,
              1192.0: 5.1175e-01,
              1193.0: 5.1064e-01,
              1194.0: 5.1127e-01,
              1195.0: 5.0592e-01,
              1196.0: 5.0561e-01,
              1197.0: 5.0286e-01,
              1198.0: 4.9217e-01,
              1199.0: 4.7434e-01,
              1200.0: 5.0005e-01,
              1201.0: 5.0579e-01,
              1202.0: 5.0705e-01,
              1203.0: 4.8836e-01,
              1204.0: 4.8696e-01,
              1205.0: 5.0195e-01,
              1206.0: 4.9869e-01,
              1207.0: 5.0143e-01,
              1208.0: 4.8962e-01,
              1209.0: 4.6636e-01,
              1210.0: 4.9225e-01,
              1211.0: 4.7909e-01,
              1212.0: 4.9064e-01,
              1213.0: 4.9190e-01,
              1214.0: 4.8932e-01,
              1215.0: 4.9044e-01,
              1216.0: 4.8878e-01,
              1217.0: 4.8764e-01,
              1218.0: 4.8599e-01,
              1219.0: 4.8234e-01,
              1220.0: 4.8433e-01,
              1221.0: 4.8343e-01,
              1222.0: 4.8133e-01,
              1223.0: 4.7600e-01,
              1224.0: 4.8224e-01,
              1225.0: 4.7925e-01,
              1226.0: 4.8026e-01,
              1227.0: 4.6841e-01,
              1228.0: 4.7443e-01,
              1229.0: 4.7725e-01,
              1230.0: 4.7502e-01,
              1231.0: 4.7425e-01,
              1232.0: 4.6896e-01,
              1233.0: 4.7118e-01,
              1234.0: 4.6946e-01,
              1235.0: 4.6644e-01,
              1236.0: 4.6863e-01,
              1237.0: 4.6790e-01,
              1238.0: 4.6748e-01,
              1239.0: 4.6235e-01,
              1240.0: 4.5968e-01,
              1241.0: 4.6439e-01,
              1242.0: 4.6169e-01,
              1243.0: 4.5655e-01,
              1244.0: 4.5535e-01,
              1245.0: 4.5600e-01,
              1246.0: 4.5875e-01,
              1247.0: 4.5775e-01,
              1248.0: 4.6076e-01,
              1249.0: 4.6013e-01,
              1250.0: 4.5860e-01,
              1251.0: 4.5638e-01,
              1252.0: 4.5195e-01,
              1253.0: 4.5371e-01,
              1254.0: 4.5358e-01,
              1255.0: 4.5304e-01,
              1256.0: 4.5192e-01,
              1257.0: 4.4880e-01,
              1258.0: 4.5116e-01,
              1259.0: 4.4797e-01,
              1260.0: 4.5018e-01,
              1261.0: 4.4925e-01,
              1262.0: 4.4252e-01,
              1263.0: 4.4825e-01,
              1264.0: 4.4198e-01,
              1265.0: 4.4267e-01,
              1266.0: 4.4615e-01,
              1267.0: 4.3912e-01,
              1268.0: 4.3594e-01,
              1269.0: 4.4042e-01,
              1270.0: 4.4212e-01,
              1271.0: 4.4290e-01,
              1272.0: 4.4130e-01,
              1273.0: 4.4084e-01,
              1274.0: 4.3807e-01,
              1275.0: 4.4072e-01,
              1276.0: 4.4192e-01,
              1277.0: 4.4188e-01,
              1278.0: 4.4074e-01,
              1279.0: 4.3795e-01,
              1280.0: 4.3543e-01,
              1281.0: 4.2267e-01,
              1282.0: 3.7589e-01,
              1283.0: 4.1159e-01,
              1284.0: 4.2667e-01,
              1285.0: 4.2909e-01,
              1286.0: 4.2878e-01,
              1287.0: 4.3036e-01,
              1288.0: 4.2684e-01,
              1289.0: 4.2755e-01,
              1290.0: 4.1851e-01,
              1291.0: 4.2159e-01,
              1292.0: 4.2278e-01,
              1293.0: 4.2261e-01,
              1294.0: 4.1615e-01,
              1295.0: 4.1944e-01,
              1296.0: 4.1979e-01,
              1297.0: 4.1718e-01,
              1298.0: 4.1263e-01,
              1299.0: 4.1701e-01,
              1300.0: 4.1537e-01,
              1301.0: 4.1404e-01,
              1302.0: 4.0955e-01,
              1303.0: 4.0768e-01,
              1304.0: 4.0632e-01,
              1305.0: 4.1028e-01,
              1306.0: 4.1039e-01,
              1307.0: 4.0958e-01,
              1308.0: 4.0873e-01,
              1309.0: 4.0907e-01,
              1310.0: 4.0714e-01,
              1311.0: 4.0611e-01,
              1312.0: 4.0315e-01,
              1313.0: 3.9084e-01,
              1314.0: 4.0271e-01,
              1315.0: 3.9162e-01,
              1316.0: 3.9421e-01,
              1317.0: 4.0116e-01,
              1318.0: 3.9547e-01,
              1319.0: 4.0093e-01,
              1320.0: 3.9875e-01,
              1321.0: 3.9545e-01,
              1322.0: 3.9538e-01,
              1323.0: 3.9502e-01,
              1324.0: 3.9516e-01,
              1325.0: 3.9369e-01,
              1326.0: 3.9229e-01,
              1327.0: 3.8877e-01,
              1328.0: 3.9062e-01,
              1329.0: 3.7950e-01,
              1330.0: 3.8548e-01,
              1331.0: 3.8765e-01,
              1332.0: 3.7850e-01,
              1333.0: 3.8699e-01,
              1334.0: 3.8684e-01,
              1335.0: 3.8563e-01,
              1336.0: 3.8231e-01,
              1337.0: 3.8285e-01,
              1338.0: 3.8094e-01,
              1339.0: 3.7570e-01,
              1340.0: 3.7439e-01,
              1341.0: 3.8168e-01,
              1342.0: 3.8020e-01,
              1343.0: 3.7977e-01,
              1344.0: 3.7986e-01,
              1345.0: 3.7953e-01,
              1346.0: 3.7554e-01,
              1347.0: 3.7760e-01,
              1348.0: 3.7664e-01,
              1349.0: 3.7550e-01,
              1350.0: 3.7081e-01,
              1351.0: 3.7167e-01,
              1352.0: 3.7533e-01,
              1353.0: 3.7401e-01,
              1354.0: 3.7294e-01,
              1355.0: 3.6827e-01,
              1356.0: 3.6876e-01,
              1357.0: 3.6439e-01,
              1358.0: 3.6997e-01,
              1359.0: 3.6903e-01,
              1360.0: 3.6464e-01,
              1361.0: 3.7027e-01,
              1362.0: 3.6936e-01,
              1363.0: 3.6185e-01,
              1364.0: 3.6683e-01,
              1365.0: 3.6629e-01,
              1366.0: 3.6592e-01,
              1367.0: 3.6242e-01,
              1368.0: 3.6557e-01,
              1369.0: 3.5886e-01,
              1370.0: 3.5912e-01,
              1371.0: 3.5949e-01,
              1372.0: 3.6313e-01,
              1373.0: 3.6294e-01,
              1374.0: 3.6291e-01,
              1375.0: 3.5749e-01,
              1376.0: 3.5350e-01,
              1377.0: 3.5790e-01,
              1378.0: 3.5652e-01,
              1379.0: 3.6035e-01,
              1380.0: 3.5682e-01,
              1381.0: 3.5784e-01,
              1382.0: 3.5274e-01,
              1383.0: 3.5376e-01,
              1384.0: 3.5594e-01,
              1385.0: 3.5242e-01,
              1386.0: 3.5493e-01,
              1387.0: 3.5136e-01,
              1388.0: 3.5277e-01,
              1389.0: 3.5406e-01,
              1390.0: 3.4736e-01,
              1391.0: 3.5242e-01,
              1392.0: 3.5179e-01,
              1393.0: 3.5008e-01,
              1394.0: 3.4760e-01,
              1395.0: 3.4754e-01,
              1396.0: 3.4943e-01,
              1397.0: 3.4932e-01,
              1398.0: 3.4952e-01,
              1399.0: 3.4886e-01,
              1400.0: 3.3896e-01,
              1401.0: 3.3929e-01,
              1402.0: 3.4411e-01,
              1403.0: 3.4014e-01,
              1404.0: 3.4106e-01,
              1405.0: 3.4012e-01,
              1406.0: 3.4164e-01,
              1407.0: 3.4085e-01,
              1408.0: 3.4007e-01,
              1409.0: 3.4118e-01,
              1410.0: 3.4046e-01,
              1411.0: 3.3767e-01,
              1412.0: 3.2920e-01,
              1413.0: 3.3290e-01,
              1414.0: 3.3704e-01,
              1415.0: 3.3707e-01,
              1416.0: 3.3503e-01,
              1417.0: 3.3748e-01,
              1418.0: 3.3773e-01,
              1419.0: 3.3391e-01,
              1420.0: 3.3270e-01,
              1421.0: 3.3594e-01,
              1422.0: 3.2875e-01,
              1423.0: 3.2884e-01,
              1424.0: 3.2772e-01,
              1425.0: 3.3256e-01,
              1426.0: 3.2306e-01,
              1427.0: 3.2745e-01,
              1428.0: 3.3036e-01,
              1429.0: 3.2347e-01,
              1430.0: 3.2277e-01,
              1431.0: 3.2421e-01,
              1432.0: 3.3114e-01,
              1433.0: 3.3067e-01,
              1434.0: 3.2509e-01,
              1435.0: 3.2731e-01,
              1436.0: 3.2662e-01,
              1437.0: 3.2786e-01,
              1438.0: 3.2601e-01,
              1439.0: 3.2856e-01,
              1440.0: 3.1269e-01,
              1441.0: 3.1695e-01,
              1442.0: 3.1584e-01,
              1443.0: 3.1780e-01,
              1444.0: 3.1911e-01,
              1445.0: 3.1929e-01,
              1446.0: 3.2135e-01,
              1447.0: 3.1760e-01,
              1448.0: 3.2096e-01,
              1449.0: 3.2026e-01,
              1450.0: 3.1774e-01,
              1451.0: 3.1367e-01,
              1452.0: 3.1355e-01,
              1453.0: 3.1539e-01,
              1454.0: 3.1282e-01,
              1455.0: 3.1210e-01,
              1456.0: 3.0956e-01,
              1457.0: 3.1241e-01,
              1458.0: 3.1612e-01,
              1459.0: 3.1746e-01,
              1460.0: 3.1367e-01,
              1461.0: 3.1308e-01,
              1462.0: 3.1018e-01,
              1463.0: 3.0823e-01,
              1464.0: 3.0820e-01,
              1465.0: 3.1158e-01,
              1466.0: 3.0696e-01,
              1467.0: 3.1322e-01,
              1468.0: 3.0750e-01,
              1469.0: 3.1201e-01,
              1470.0: 3.1010e-01,
              1471.0: 3.0414e-01,
              1472.0: 3.0581e-01,
              1473.0: 3.0552e-01,
              1474.0: 2.9705e-01,
              1475.0: 3.0349e-01,
              1476.0: 3.0369e-01,
              1477.0: 3.0250e-01,
              1478.0: 3.0228e-01,
              1479.0: 3.0461e-01,
              1480.0: 3.0626e-01,
              1481.0: 3.0179e-01,
              1482.0: 3.0462e-01,
              1483.0: 2.9813e-01,
              1484.0: 3.0172e-01,
              1485.0: 3.0607e-01,
              1486.0: 3.0168e-01,
              1487.0: 2.9989e-01,
              1488.0: 2.6525e-01,
              1489.0: 2.9910e-01,
              1490.0: 3.0093e-01,
              1491.0: 3.0175e-01,
              1492.0: 2.9773e-01,
              1493.0: 3.0200e-01,
              1494.0: 3.0190e-01,
              1495.0: 3.0074e-01,
              1496.0: 2.9140e-01,
              1497.0: 2.9684e-01,
              1498.0: 2.9946e-01,
              1499.0: 2.9528e-01,
              1500.0: 3.0077e-01,
              1501.0: 2.9990e-01,
              1502.0: 2.9743e-01,
              1503.0: 2.6920e-01,
              1504.0: 2.8319e-01,
              1505.0: 2.7149e-01,
              1506.0: 2.8367e-01,
              1507.0: 2.9376e-01,
              1508.0: 2.9028e-01,
              1509.0: 2.8805e-01,
              1510.0: 2.9394e-01,
              1511.0: 2.9446e-01,
              1512.0: 2.8845e-01,
              1513.0: 2.8729e-01,
              1514.0: 2.8500e-01,
              1515.0: 2.8807e-01,
              1516.0: 2.8720e-01,
              1517.0: 2.8526e-01,
              1518.0: 2.8970e-01,
              1519.0: 2.8495e-01,
              1520.0: 2.8786e-01,
              1521.0: 2.8083e-01,
              1522.0: 2.8031e-01,
              1523.0: 2.8448e-01,
              1524.0: 2.8460e-01,
              1525.0: 2.7846e-01,
              1526.0: 2.8596e-01,
              1527.0: 2.8297e-01,
              1528.0: 2.8490e-01,
              1529.0: 2.8290e-01,
              1530.0: 2.6731e-01,
              1531.0: 2.8272e-01,
              1532.0: 2.8263e-01,
              1533.0: 2.8041e-01,
              1534.0: 2.7543e-01,
              1535.0: 2.7754e-01,
              1536.0: 2.8058e-01,
              1537.0: 2.8096e-01,
              1538.0: 2.7338e-01,
              1539.0: 2.7808e-01,
              1540.0: 2.6857e-01,
              1541.0: 2.7297e-01,
              1542.0: 2.7536e-01,
              1543.0: 2.7499e-01,
              1544.0: 2.7716e-01,
              1545.0: 2.7768e-01,
              1546.0: 2.7722e-01,
              1547.0: 2.7354e-01,
              1548.0: 2.6698e-01,
              1549.0: 2.7267e-01,
              1550.0: 2.6961e-01,
              1551.0: 2.7075e-01,
              1552.0: 2.7400e-01,
              1553.0: 2.7090e-01,
              1554.0: 2.6495e-01,
              1555.0: 2.6712e-01,
              1556.0: 2.6292e-01,
              1557.0: 2.7013e-01,
              1558.0: 2.6805e-01,
              1559.0: 2.6780e-01,
              1560.0: 2.6546e-01,
              1561.0: 2.6968e-01,
              1562.0: 2.6833e-01,
              1563.0: 2.6688e-01,
              1564.0: 2.6307e-01,
              1565.0: 2.6682e-01,
              1566.0: 2.6262e-01,
              1567.0: 2.6352e-01,
              1568.0: 2.6042e-01,
              1569.0: 2.6275e-01,
              1570.0: 2.6068e-01,
              1571.0: 2.6668e-01,
              1572.0: 2.6694e-01,
              1573.0: 2.6344e-01,
              1574.0: 2.6221e-01,
              1575.0: 2.4738e-01,
              1576.0: 2.6031e-01,
              1577.0: 2.3690e-01,
              1578.0: 2.5723e-01,
              1579.0: 2.6167e-01,
              1580.0: 2.6381e-01,
              1581.0: 2.6483e-01,
              1582.0: 2.5412e-01,
              1583.0: 2.5771e-01,
              1584.0: 2.5449e-01,
              1585.0: 2.6198e-01,
              1586.0: 2.5817e-01,
              1587.0: 2.5492e-01,
              1588.0: 2.5048e-01,
              1589.0: 2.3152e-01,
              1590.0: 2.4154e-01,
              1591.0: 2.4132e-01,
              1592.0: 2.5286e-01,
              1593.0: 2.5751e-01,
              1594.0: 2.5643e-01,
              1595.0: 2.5765e-01,
              1596.0: 2.4452e-01,
              1597.0: 2.4854e-01,
              1598.0: 2.5629e-01,
              1599.0: 2.4977e-01,
              1600.0: 2.5259e-01,
              1601.0: 2.4494e-01,
              1602.0: 2.4857e-01,
              1603.0: 2.4901e-01,
              1604.0: 2.5002e-01,
              1605.0: 2.4703e-01,
              1606.0: 2.4748e-01,
              1607.0: 2.4544e-01,
              1608.0: 2.4999e-01,
              1609.0: 2.4858e-01,
              1610.0: 2.3312e-01,
              1611.0: 2.4078e-01,
              1612.0: 2.4125e-01,
              1613.0: 2.4581e-01,
              1614.0: 2.4350e-01,
              1615.0: 2.4434e-01,
              1616.0: 2.3248e-01,
              1617.0: 2.3644e-01,
              1618.0: 2.4415e-01,
              1619.0: 2.4224e-01,
              1620.0: 2.3407e-01,
              1621.0: 2.3386e-01,
              1622.0: 2.3702e-01,
              1623.0: 2.4166e-01,
              1624.0: 2.4240e-01,
              1625.0: 2.3746e-01,
              1626.0: 2.3958e-01,
              1627.0: 2.4037e-01,
              1628.0: 2.4149e-01,
              1629.0: 2.4111e-01,
              1630.0: 2.3655e-01,
              1631.0: 2.3932e-01,
              1632.0: 2.3884e-01,
              1633.0: 2.3585e-01,
              1634.0: 2.3276e-01,
              1635.0: 2.3735e-01,
              1636.0: 2.3676e-01,
              1637.0: 2.2747e-01,
              1638.0: 2.2708e-01,
              1639.0: 2.2205e-01,
              1640.0: 2.2571e-01,
              1641.0: 2.2393e-01,
              1642.0: 2.2402e-01,
              1643.0: 2.2879e-01,
              1644.0: 2.2572e-01,
              1645.0: 2.2512e-01,
              1646.0: 2.2899e-01,
              1647.0: 2.2910e-01,
              1648.0: 2.2898e-01,
              1649.0: 2.2354e-01,
              1650.0: 2.2840e-01,
              1651.0: 2.2369e-01,
              1652.0: 2.2802e-01,
              1653.0: 2.2533e-01,
              1654.0: 2.2654e-01,
              1655.0: 2.2390e-01,
              1656.0: 2.2607e-01,
              1657.0: 2.2564e-01,
              1658.0: 2.2532e-01,
              1659.0: 2.2221e-01,
              1660.0: 2.2412e-01,
              1661.0: 2.2543e-01,
              1662.0: 2.2421e-01,
              1663.0: 2.2265e-01,
              1664.0: 2.2252e-01,
              1665.0: 2.2137e-01,
              1666.0: 2.1853e-01,
              1667.0: 2.2016e-01,
              1668.0: 2.1617e-01,
              1669.0: 2.1809e-01,
              1670.0: 2.2208e-01,
              1671.0: 2.2096e-01,
              1672.0: 2.1210e-01,
              1673.0: 2.1606e-01,
              1674.0: 2.1907e-01,
              1675.0: 2.1621e-01,
              1676.0: 2.1319e-01,
              1677.0: 2.1548e-01,
              1678.0: 2.1585e-01,
              1679.0: 2.1295e-01,
              1680.0: 2.0641e-01,
              1681.0: 1.9848e-01,
              1682.0: 2.0341e-01,
              1683.0: 2.0893e-01,
              1684.0: 2.1198e-01,
              1685.0: 2.1306e-01,
              1686.0: 2.1026e-01,
              1687.0: 2.1123e-01,
              1688.0: 2.1081e-01,
              1689.0: 2.0805e-01,
              1690.0: 2.0845e-01,
              1691.0: 2.1203e-01,
              1692.0: 2.1211e-01,
              1693.0: 2.1122e-01,
              1694.0: 2.1068e-01,
              1695.0: 2.1055e-01,
              1696.0: 2.0962e-01,
              1697.0: 2.0850e-01,
              1698.0: 2.0879e-01,
              1699.0: 2.0804e-01,
              1700.0: 2.0539e-01,
              1702.0: 2.0520e-01,
              1705.0: 2.0428e-01,
              1710.0: 1.9894e-01,
              1715.0: 2.0189e-01,
              1720.0: 1.9799e-01,
              1725.0: 1.9688e-01,
              1730.0: 1.9309e-01,
              1735.0: 1.8200e-01,
              1740.0: 1.9001e-01,
              1745.0: 1.8624e-01,
              1750.0: 1.8518e-01,
              1755.0: 1.8509e-01,
              1760.0: 1.8243e-01,
              1765.0: 1.8143e-01,
              1770.0: 1.7969e-01,
              1775.0: 1.7600e-01,
              1780.0: 1.7553e-01,
              1785.0: 1.7411e-01,
              1790.0: 1.7371e-01,
              1795.0: 1.7102e-01,
              1800.0: 1.6800e-01,
              1805.0: 1.6884e-01,
              1810.0: 1.6859e-01,
              1815.0: 1.5500e-01,
              1820.0: 1.6000e-01,
              1825.0: 1.6257e-01,
              1830.0: 1.5921e-01,
              1835.0: 1.5760e-01,
              1840.0: 1.5552e-01,
              1845.0: 1.5288e-01,
              1850.0: 1.5337e-01,
              1855.0: 1.5126e-01,
              1860.0: 1.4933e-01,
              1865.0: 1.4710e-01,
              1870.0: 1.4565e-01,
              1875.0: 1.3231e-01,
              1880.0: 1.4667e-01,
              1885.0: 1.4573e-01,
              1890.0: 1.4041e-01,
              1895.0: 1.3842e-01,
              1900.0: 1.4041e-01,
              1905.0: 1.3904e-01,
              1910.0: 1.3654e-01,
              1915.0: 1.3611e-01,
              1920.0: 1.3463e-01,
              1925.0: 1.3447e-01,
              1930.0: 1.3145e-01,
              1935.0: 1.3243e-01,
              1940.0: 1.2950e-01,
              1945.0: 1.1950e-01,
              1950.0: 1.2627e-01,
              1955.0: 1.2833e-01,
              1960.0: 1.2610e-01,
              1965.0: 1.2291e-01,
              1970.0: 1.2375e-01,
              1975.0: 1.2150e-01,
              1980.0: 1.1968e-01,
              1985.0: 1.1902e-01,
              1990.0: 1.1977e-01,
              1995.0: 1.1684e-01,
              2000.0: 1.1673e-01,
              2005.0: 1.1501e-01,
              2010.0: 1.1512e-01,
              2015.0: 1.1399e-01,
              2020.0: 1.1192e-01,
              2025.0: 1.1176e-01,
              2030.0: 1.0969e-01,
              2035.0: 1.0915e-01,
              2040.0: 1.0720e-01,
              2045.0: 1.0756e-01,
              2050.0: 1.0592e-01,
              2055.0: 1.0471e-01,
              2060.0: 1.0320e-01,
              2065.0: 1.0182e-01,
              2070.0: 1.0095e-01,
              2075.0: 1.0095e-01,
              2080.0: 9.9330e-02,
              2085.0: 9.8300e-02,
              2090.0: 9.7540e-02,
              2095.0: 9.5990e-02,
              2100.0: 9.6240e-02,
              2105.0: 9.5750e-02,
              2110.0: 9.4630e-02,
              2115.0: 9.3850e-02,
              2120.0: 9.3140e-02,
              2125.0: 9.2180e-02,
              2130.0: 9.2380e-02,
              2135.0: 9.1120e-02,
              2140.0: 9.1050e-02,
              2145.0: 8.9940e-02,
              2150.0: 8.9710e-02,
              2155.0: 8.8880e-02,
              2160.0: 8.7890e-02,
              2165.0: 8.2000e-02,
              2170.0: 8.5370e-02,
              2175.0: 8.5760e-02,
              2180.0: 8.4640e-02,
              2185.0: 8.4680e-02,
              2190.0: 8.3140e-02,
              2195.0: 8.3470e-02,
              2200.0: 8.2790e-02,
              2205.0: 8.0900e-02,
              2210.0: 8.0810e-02,
              2215.0: 8.0410e-02,
              2220.0: 7.9990e-02,
              2225.0: 7.8840e-02,
              2230.0: 7.8400e-02,
              2235.0: 7.7930e-02,
              2240.0: 7.6510e-02,
              2245.0: 7.6250e-02,
              2250.0: 7.5370e-02,
              2255.0: 7.4340e-02,
              2260.0: 7.4090e-02,
              2265.0: 7.3280e-02,
              2270.0: 7.3100e-02,
              2275.0: 7.2630e-02,
              2280.0: 7.1390e-02,
              2285.0: 7.1380e-02,
              2290.0: 7.1190e-02,
              2295.0: 6.9300e-02,
              2300.0: 6.9640e-02,
              2305.0: 6.9400e-02,
              2310.0: 6.8900e-02,
              2315.0: 6.8150e-02,
              2320.0: 6.7630e-02,
              2325.0: 6.5560e-02,
              2330.0: 6.6220e-02,
              2335.0: 6.5720e-02,
              2340.0: 6.5220e-02,
              2345.0: 6.5100e-02,
              2350.0: 6.4340e-02,
              2355.0: 6.2600e-02,
              2360.0: 6.3070e-02,
              2365.0: 6.3010e-02,
              2370.0: 6.2390e-02,
              2375.0: 6.1410e-02,
              2380.0: 6.1810e-02,
              2385.0: 5.9130e-02,
              2390.0: 6.0360e-02,
              2395.0: 6.0070e-02,
              2400.0: 5.9740e-02,
              2405.0: 5.9440e-02,
              2410.0: 5.9150e-02,
              2415.0: 5.6980e-02,
              2420.0: 5.7820e-02,
              2425.0: 5.7330e-02,
              2430.0: 5.7190e-02,
              2435.0: 5.6470e-02,
              2440.0: 5.6270e-02,
              2445.0: 5.5560e-02,
              2450.0: 5.4590e-02,
              2455.0: 5.4160e-02,
              2460.0: 5.4510e-02,
              2465.0: 5.4300e-02,
              2470.0: 5.3400e-02,
              2475.0: 5.3370e-02,
              2480.0: 5.2100e-02,
              2485.0: 5.1770e-02,
              2490.0: 5.2220e-02,
              2495.0: 4.8890e-02,
              2500.0: 5.1380e-02,
              2505.0: 5.0820e-02,
              2510.0: 4.9600e-02,
              2515.0: 4.8760e-02,
              2520.0: 4.8140e-02,
              2525.0: 4.7650e-02,
              2530.0: 4.7700e-02,
              2535.0: 4.7120e-02,
              2540.0: 4.6770e-02,
              2545.0: 4.6610e-02,
              2550.0: 4.6260e-02,
              2555.0: 4.5930e-02,
              2560.0: 4.5400e-02,
              2565.0: 4.5160e-02,
              2570.0: 4.4850e-02,
              2575.0: 4.4480e-02,
              2580.0: 4.4100e-02,
              2585.0: 4.3270e-02,
              2590.0: 4.3410e-02,
              2595.0: 4.3040e-02,
              2600.0: 4.2910e-02,
              2605.0: 4.2800e-02,
              2610.0: 4.2500e-02,
              2615.0: 4.2280e-02,
              2620.0: 4.1900e-02,
              2625.0: 4.0580e-02,
              2630.0: 4.1160e-02,
              2635.0: 4.0960e-02,
              2640.0: 4.0220e-02,
              2645.0: 4.0180e-02,
              2650.0: 3.9960e-02,
              2655.0: 3.9590e-02,
              2660.0: 3.9480e-02,
              2665.0: 3.8950e-02,
              2670.0: 3.9100e-02,
              2675.0: 3.8610e-02,
              2680.0: 3.8430e-02,
              2685.0: 3.8160e-02,
              2690.0: 3.7790e-02,
              2695.0: 3.7460e-02,
              2700.0: 3.7120e-02,
              2705.0: 3.6930e-02,
              2710.0: 3.6680e-02,
              2715.0: 3.6560e-02,
              2720.0: 3.6380e-02,
              2725.0: 3.6130e-02,
              2730.0: 3.6010e-02,
              2735.0: 3.5740e-02,
              2740.0: 3.5530e-02,
              2745.0: 3.5380e-02,
              2750.0: 3.5180e-02,
              2755.0: 3.4860e-02,
              2760.0: 3.4550e-02,
              2765.0: 3.4400e-02,
              2770.0: 3.4260e-02,
              2775.0: 3.3930e-02,
              2780.0: 3.3640e-02,
              2785.0: 3.3420e-02,
              2790.0: 3.3200e-02,
              2795.0: 3.2930e-02,
              2800.0: 3.2760e-02,
              2805.0: 3.2530e-02,
              2810.0: 3.2420e-02,
              2815.0: 3.2260e-02,
              2820.0: 3.2060e-02,
              2825.0: 3.1840e-02,
              2830.0: 3.1700e-02,
              2835.0: 3.1450e-02,
              2840.0: 3.1180e-02,
              2845.0: 3.0950e-02,
              2850.0: 3.0680e-02,
              2855.0: 3.0430e-02,
              2860.0: 3.0160e-02,
              2865.0: 2.9960e-02,
              2870.0: 2.9600e-02,
              2875.0: 2.9400e-02,
              2880.0: 2.9400e-02,
              2885.0: 2.9220e-02,
              2890.0: 2.9070e-02,
              2895.0: 2.8960e-02,
              2900.0: 2.8810e-02,
              2905.0: 2.8670e-02,
              2910.0: 2.8500e-02,
              2915.0: 2.8220e-02,
              2920.0: 2.8040e-02,
              2925.0: 2.7780e-02,
              2930.0: 2.7570e-02,
              2935.0: 2.7330e-02,
              2940.0: 2.7140e-02,
              2945.0: 2.6840e-02,
              2950.0: 2.6680e-02,
              2955.0: 2.6560e-02,
              2960.0: 2.6380e-02,
              2965.0: 2.6260e-02,
              2970.0: 2.6120e-02,
              2975.0: 2.6010e-02,
              2980.0: 2.5800e-02,
              2985.0: 2.5690e-02,
              2990.0: 2.5540e-02,
              2995.0: 2.5400e-02,
              3000.0: 2.5250e-02,
              3005.0: 2.5120e-02,
              3010.0: 2.4980e-02,
              3015.0: 2.4860e-02,
              3020.0: 2.4660e-02,
              3025.0: 2.4540e-02,
              3030.0: 2.4420e-02,
              3035.0: 2.4210e-02,
              3040.0: 2.3930e-02,
              3045.0: 2.4020e-02,
              3050.0: 2.3950e-02,
              3055.0: 2.3860e-02,
              3060.0: 2.3760e-02,
              3065.0: 2.3540e-02,
              3070.0: 2.3440e-02,
              3075.0: 2.3350e-02,
              3080.0: 2.3150e-02,
              3085.0: 2.2950e-02,
              3090.0: 2.2800e-02,
              3095.0: 2.2630e-02,
              3100.0: 2.2430e-02,
              3105.0: 2.2280e-02,
              3110.0: 2.2110e-02,
              3115.0: 2.1770e-02,
              3120.0: 2.1740e-02,
              3125.0: 2.1580e-02,
              3130.0: 2.1450e-02,
              3135.0: 2.1360e-02,
              3140.0: 2.1180e-02,
              3145.0: 2.1070e-02,
              3150.0: 2.0880e-02,
              3155.0: 2.0830e-02,
              3160.0: 2.0730e-02,
              3165.0: 2.0660e-02,
              3170.0: 2.0520e-02,
              3175.0: 2.0470e-02,
              3180.0: 2.0320e-02,
              3185.0: 2.0240e-02,
              3190.0: 2.0190e-02,
              3195.0: 2.0080e-02,
              3200.0: 1.9940e-02,
              3205.0: 1.9860e-02,
              3210.0: 1.9760e-02,
              3215.0: 1.9620e-02,
              3220.0: 1.9540e-02,
              3225.0: 1.9420e-02,
              3230.0: 1.9380e-02,
              3235.0: 1.9260e-02,
              3240.0: 1.9210e-02,
              3245.0: 1.9100e-02,
              3250.0: 1.9090e-02,
              3255.0: 1.8900e-02,
              3260.0: 1.8800e-02,
              3265.0: 1.8700e-02,
              3270.0: 1.8600e-02,
              3275.0: 1.8480e-02,
              3280.0: 1.8370e-02,
              3285.0: 1.8320e-02,
              3290.0: 1.8260e-02,
              3295.0: 1.8030e-02,
              3300.0: 1.7920e-02,
              3305.0: 1.7860e-02,
              3310.0: 1.7700e-02,
              3315.0: 1.7600e-02,
              3320.0: 1.7300e-02,
              3325.0: 1.7280e-02,
              3330.0: 1.7130e-02,
              3335.0: 1.7030e-02,
              3340.0: 1.6850e-02,
              3345.0: 1.6780e-02,
              3350.0: 1.6600e-02,
              3355.0: 1.6500e-02,
              3360.0: 1.6360e-02,
              3365.0: 1.6290e-02,
              3370.0: 1.6180e-02,
              3375.0: 1.6110e-02,
              3380.0: 1.6010e-02,
              3385.0: 1.5980e-02,
              3390.0: 1.5900e-02,
              3395.0: 1.5800e-02,
              3400.0: 1.5640e-02,
              3405.0: 1.5650e-02,
              3410.0: 1.5540e-02,
              3415.0: 1.5470e-02,
              3420.0: 1.5430e-02,
              3425.0: 1.5320e-02,
              3430.0: 1.5220e-02,
              3435.0: 1.5150e-02,
              3440.0: 1.5050e-02,
              3445.0: 1.5020e-02,
              3450.0: 1.4920e-02,
              3455.0: 1.4880e-02,
              3460.0: 1.4800e-02,
              3465.0: 1.4760e-02,
              3470.0: 1.4660e-02,
              3475.0: 1.4560e-02,
              3480.0: 1.4480e-02,
              3485.0: 1.4450e-02,
              3490.0: 1.4360e-02,
              3495.0: 1.4320e-02,
              3500.0: 1.4230e-02,
              3505.0: 1.4150e-02,
              3510.0: 1.4050e-02,
              3515.0: 1.4020e-02,
              3520.0: 1.3930e-02,
              3525.0: 1.3890e-02,
              3530.0: 1.3800e-02,
              3535.0: 1.3760e-02,
              3540.0: 1.3670e-02,
              3545.0: 1.3660e-02,
              3550.0: 1.3570e-02,
              3555.0: 1.3450e-02,
              3560.0: 1.3360e-02,
              3565.0: 1.3360e-02,
              3570.0: 1.3250e-02,
              3575.0: 1.3180e-02,
              3580.0: 1.3080e-02,
              3585.0: 1.3050e-02,
              3590.0: 1.2980e-02,
              3595.0: 1.2940e-02,
              3600.0: 1.2850e-02,
              3605.0: 1.2830e-02,
              3610.0: 1.2740e-02,
              3615.0: 1.2720e-02,
              3620.0: 1.2640e-02,
              3625.0: 1.2610e-02,
              3630.0: 1.2510e-02,
              3635.0: 1.2490e-02,
              3640.0: 1.2400e-02,
              3645.0: 1.2360e-02,
              3650.0: 1.2280e-02,
              3655.0: 1.2220e-02,
              3660.0: 1.2150e-02,
              3665.0: 1.2150e-02,
              3670.0: 1.2040e-02,
              3675.0: 1.2030e-02,
              3680.0: 1.1890e-02,
              3685.0: 1.1830e-02,
              3690.0: 1.1770e-02,
              3695.0: 1.1720e-02,
              3700.0: 1.1550e-02,
              3705.0: 1.1600e-02,
              3710.0: 1.1510e-02,
              3715.0: 1.1500e-02,
              3720.0: 1.1440e-02,
              3725.0: 1.1430e-02,
              3730.0: 1.1340e-02,
              3735.0: 1.1280e-02,
              3740.0: 1.1080e-02,
              3745.0: 1.1080e-02,
              3750.0: 1.1040e-02,
              3755.0: 1.1000e-02,
              3760.0: 1.0940e-02,
              3765.0: 1.0930e-02,
              3770.0: 1.0860e-02,
              3775.0: 1.0800e-02,
              3780.0: 1.0740e-02,
              3785.0: 1.0730e-02,
              3790.0: 1.0630e-02,
              3795.0: 1.0620e-02,
              3800.0: 1.0530e-02,
              3805.0: 1.0530e-02,
              3810.0: 1.0430e-02,
              3815.0: 1.0410e-02,
              3820.0: 1.0340e-02,
              3825.0: 1.0330e-02,
              3830.0: 1.0250e-02,
              3835.0: 1.0210e-02,
              3840.0: 1.0150e-02,
              3845.0: 1.0130e-02,
              3850.0: 1.0050e-02,
              3855.0: 1.0030e-02,
              3860.0: 9.9400e-03,
              3865.0: 9.9100e-03,
              3870.0: 9.8300e-03,
              3875.0: 9.8300e-03,
              3880.0: 9.7500e-03,
              3885.0: 9.7000e-03,
              3890.0: 9.7100e-03,
              3895.0: 9.6200e-03,
              3900.0: 9.6000e-03,
              3905.0: 9.5900e-03,
              3910.0: 9.5200e-03,
              3915.0: 9.5000e-03,
              3920.0: 9.4100e-03,
              3925.0: 9.4000e-03,
              3930.0: 9.3200e-03,
              3935.0: 9.3000e-03,
              3940.0: 9.2300e-03,
              3945.0: 9.2000e-03,
              3950.0: 9.1100e-03,
              3955.0: 9.0800e-03,
              3960.0: 9.0200e-03,
              3965.0: 9.0100e-03,
              3970.0: 8.9300e-03,
              3975.0: 8.9100e-03,
              3980.0: 8.8400e-03,
              3985.0: 8.8000e-03,
              3990.0: 8.7800e-03,
              3995.0: 8.7000e-03,
              4000.0: 8.6800e-03}

# ASTM_G_173 scaling factor calculated with :def:`color.spectral.transformations.spectral_to_XYZ` definition.
for key, value in ASTM_G_173.iteritems():
    ASTM_G_173[key] = value * 1.37905559e+13

ASTM_G_173_spd = color.SpectralPowerDistribution(name="ASTM G-173", spd=ASTM_G_173)
ASTM_G_173_spd.resample(steps=1)

blackbody_spd = color.blackbody_spectral_power_distribution(5778, *ASTM_G_173_spd.shape)
blackbody_spd.name = "The Sun - 5778K"

multi_spectral_power_distribution_plot([ASTM_G_173_spd, blackbody_spd],
                                       y_label=u"W / (sr m²) / m",
                                       legend_location="upper right")

# Plotting blackbody colors.
blackbody_colors_plot()
