#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
**spd.py**

**Platform:**
    Windows, Linux, Mac Os X.

**Description:**
    Defines **Color** package *illuminants* spectral power distributions.

**Others:**

"""

from __future__ import unicode_literals

import numpy

import color.exceptions
import color.verbose

from color.spectral.spd import SpectralPowerDistribution

__author__ = "Thomas Mansencal"
__copyright__ = "Copyright (C) 2013 - 2014 - Thomas Mansencal"
__license__ = "GPL V3.0 - http://www.gnu.org/licenses/"
__maintainer__ = "Thomas Mansencal"
__email__ = "thomas.mansencal@gmail.com"
__status__ = "Production"

__all__ = ["LOGGER",
           "ILLUMINANTS_RELATIVE_SPD_DATA",
           "ILLUMINANTS_RELATIVE_SPD",
           "D_ILLUMINANTS_S_DISTRIBUTIONS",
           "D_illuminant_relative_spectral_power_distribution"]

LOGGER = color.verbose.install_logger()

# Relative spectral power distribution for various illuminants in at 5 nm steps.
# http://files.cie.co.at/204.xls
# https://law.resource.org/pub/us/cfr/ibr/003/cie.15.2004.tables.xls
ILLUMINANTS_RELATIVE_SPD_DATA = {
    "A": {300: 0.930483,
          305: 1.128210,
          310: 1.357690,
          315: 1.622190,
          320: 1.925080,
          325: 2.269800,
          330: 2.659810,
          335: 3.098610,
          340: 3.589680,
          345: 4.136480,
          350: 4.742380,
          355: 5.410700,
          360: 6.144620,
          365: 6.947200,
          370: 7.821350,
          375: 8.769800,
          380: 9.795100,
          385: 10.899600,
          390: 12.085300,
          395: 13.354300,
          400: 14.708000,
          405: 16.148000,
          410: 17.675300,
          415: 19.290700,
          420: 20.995000,
          425: 22.788300,
          430: 24.670900,
          435: 26.642500,
          440: 28.702700,
          445: 30.850800,
          450: 33.085900,
          455: 35.406800,
          460: 37.812100,
          465: 40.300200,
          470: 42.869300,
          475: 45.517400,
          480: 48.242300,
          485: 51.041800,
          490: 53.913200,
          495: 56.853900,
          500: 59.861100,
          505: 62.932000,
          510: 66.063500,
          515: 69.252500,
          520: 72.495900,
          525: 75.790300,
          530: 79.132600,
          535: 82.519300,
          540: 85.947000,
          545: 89.412400,
          550: 92.912000,
          555: 96.442300,
          560: 100.000000,
          565: 103.582000,
          570: 107.184000,
          575: 110.803000,
          580: 114.436000,
          585: 118.080000,
          590: 121.731000,
          595: 125.386000,
          600: 129.043000,
          605: 132.697000,
          610: 136.346000,
          615: 139.988000,
          620: 143.618000,
          625: 147.235000,
          630: 150.836000,
          635: 154.418000,
          640: 157.979000,
          645: 161.516000,
          650: 165.028000,
          655: 168.510000,
          660: 171.963000,
          665: 175.383000,
          670: 178.769000,
          675: 182.118000,
          680: 185.429000,
          685: 188.701000,
          690: 191.931000,
          695: 195.118000,
          700: 198.261000,
          705: 201.359000,
          710: 204.409000,
          715: 207.411000,
          720: 210.365000,
          725: 213.268000,
          730: 216.120000,
          735: 218.920000,
          740: 221.667000,
          745: 224.361000,
          750: 227.000000,
          755: 229.585000,
          760: 232.115000,
          765: 234.589000,
          770: 237.008000,
          775: 239.370000,
          780: 241.675000},
    "D65": {300: 0.034100,
            305: 1.664300,
            310: 3.294500,
            315: 11.765200,
            320: 20.236000,
            325: 28.644700,
            330: 37.053500,
            335: 38.501100,
            340: 39.948800,
            345: 42.430200,
            350: 44.911700,
            355: 45.775000,
            360: 46.638300,
            365: 49.363700,
            370: 52.089100,
            375: 51.032300,
            380: 49.975500,
            385: 52.311800,
            390: 54.648200,
            395: 68.701500,
            400: 82.754900,
            405: 87.120400,
            410: 91.486000,
            415: 92.458900,
            420: 93.431800,
            425: 90.057000,
            430: 86.682300,
            435: 95.773600,
            440: 104.865000,
            445: 110.936000,
            450: 117.008000,
            455: 117.410000,
            460: 117.812000,
            465: 116.336000,
            470: 114.861000,
            475: 115.392000,
            480: 115.923000,
            485: 112.367000,
            490: 108.811000,
            495: 109.082000,
            500: 109.354000,
            505: 108.578000,
            510: 107.802000,
            515: 106.296000,
            520: 104.790000,
            525: 106.239000,
            530: 107.689000,
            535: 106.047000,
            540: 104.405000,
            545: 104.225000,
            550: 104.046000,
            555: 102.023000,
            560: 100.000000,
            565: 98.167100,
            570: 96.334200,
            575: 96.061100,
            580: 95.788000,
            585: 92.236800,
            590: 88.685600,
            595: 89.345900,
            600: 90.006200,
            605: 89.802600,
            610: 89.599100,
            615: 88.648900,
            620: 87.698700,
            625: 85.493600,
            630: 83.288600,
            635: 83.493900,
            640: 83.699200,
            645: 81.863000,
            650: 80.026800,
            655: 80.120700,
            660: 80.214600,
            665: 81.246200,
            670: 82.277800,
            675: 80.281000,
            680: 78.284200,
            685: 74.002700,
            690: 69.721300,
            695: 70.665200,
            700: 71.609100,
            705: 72.979000,
            710: 74.349000,
            715: 67.976500,
            720: 61.604000,
            725: 65.744800,
            730: 69.885600,
            735: 72.486300,
            740: 75.087000,
            745: 69.339800,
            750: 63.592700,
            755: 55.005400,
            760: 46.418200,
            765: 56.611800,
            770: 66.805400,
            775: 65.094100,
            780: 63.382800},
    "C": {300: 0.00,
          305: 0.00,
          310: 0.00,
          315: 0.00,
          320: 0.01,
          325: 0.20,
          330: 0.40,
          335: 1.55,
          340: 2.70,
          345: 4.85,
          350: 7.00,
          355: 9.95,
          360: 12.90,
          365: 17.20,
          370: 21.40,
          375: 27.50,
          380: 33.00,
          385: 39.92,
          390: 47.40,
          395: 55.17,
          400: 63.30,
          405: 71.81,
          410: 80.60,
          415: 89.53,
          420: 98.10,
          425: 105.80,
          430: 112.40,
          435: 117.75,
          440: 121.50,
          445: 123.45,
          450: 124.00,
          455: 123.60,
          460: 123.10,
          465: 123.30,
          470: 123.80,
          475: 124.09,
          480: 123.90,
          485: 122.92,
          490: 120.70,
          495: 116.90,
          500: 112.10,
          505: 106.98,
          510: 102.30,
          515: 98.81,
          520: 96.90,
          525: 96.78,
          530: 98.00,
          535: 99.94,
          540: 102.10,
          545: 103.95,
          550: 105.20,
          555: 105.67,
          560: 105.30,
          565: 104.11,
          570: 102.30,
          575: 100.15,
          580: 97.80,
          585: 95.43,
          590: 93.20,
          595: 91.22,
          600: 89.70,
          605: 88.83,
          610: 88.40,
          615: 88.19,
          620: 88.10,
          625: 88.06,
          630: 88.00,
          635: 87.86,
          640: 87.80,
          645: 87.99,
          650: 88.20,
          655: 88.20,
          660: 87.90,
          665: 87.22,
          670: 86.30,
          675: 85.30,
          680: 84.00,
          685: 82.21,
          690: 80.20,
          695: 78.24,
          700: 76.30,
          705: 74.36,
          710: 72.40,
          715: 70.40,
          720: 68.30,
          725: 66.30,
          730: 64.40,
          735: 62.80,
          740: 61.50,
          745: 60.20,
          750: 59.20,
          755: 58.50,
          760: 58.10,
          765: 58.00,
          770: 58.20,
          775: 58.50,
          780: 59.10},
    "D50": {300: 0.019,
            305: 1.035,
            310: 2.051,
            315: 4.914,
            320: 7.778,
            325: 11.263,
            330: 14.748,
            335: 16.348,
            340: 17.948,
            345: 19.479,
            350: 21.010,
            355: 22.476,
            360: 23.942,
            365: 25.451,
            370: 26.961,
            375: 25.724,
            380: 24.488,
            385: 27.179,
            390: 29.871,
            395: 39.589,
            400: 49.308,
            405: 52.910,
            410: 56.513,
            415: 58.273,
            420: 60.034,
            425: 58.926,
            430: 57.818,
            435: 66.321,
            440: 74.825,
            445: 81.036,
            450: 87.247,
            455: 88.930,
            460: 90.612,
            465: 90.990,
            470: 91.368,
            475: 93.238,
            480: 95.109,
            485: 93.536,
            490: 91.963,
            495: 93.843,
            500: 95.724,
            505: 96.169,
            510: 96.613,
            515: 96.871,
            520: 97.129,
            525: 99.614,
            530: 102.099,
            535: 101.427,
            540: 100.755,
            545: 101.536,
            550: 102.317,
            555: 101.159,
            560: 100.000,
            565: 98.868,
            570: 97.735,
            575: 98.327,
            580: 98.918,
            585: 96.208,
            590: 93.499,
            595: 95.593,
            600: 97.688,
            605: 98.478,
            610: 99.269,
            615: 99.155,
            620: 99.042,
            625: 97.382,
            630: 95.722,
            635: 97.290,
            640: 98.857,
            645: 97.262,
            650: 95.667,
            655: 96.929,
            660: 98.190,
            665: 100.597,
            670: 103.003,
            675: 101.068,
            680: 99.133,
            685: 93.257,
            690: 87.381,
            695: 89.492,
            700: 91.604,
            705: 92.246,
            710: 92.889,
            715: 84.872,
            720: 76.854,
            725: 81.683,
            730: 86.511,
            735: 89.546,
            740: 92.580,
            745: 85.405,
            750: 78.230,
            755: 67.961,
            760: 57.692,
            765: 70.307,
            770: 82.923,
            775: 80.599,
            780: 78.274},
    "D55": {300: 0.024,
            305: 1.048,
            310: 2.072,
            315: 6.648,
            320: 11.224,
            325: 15.936,
            330: 20.647,
            335: 22.266,
            340: 23.885,
            345: 25.851,
            350: 27.817,
            355: 29.219,
            360: 30.621,
            365: 32.464,
            370: 34.308,
            375: 33.446,
            380: 32.584,
            385: 35.335,
            390: 38.087,
            395: 49.518,
            400: 60.949,
            405: 64.751,
            410: 68.554,
            415: 70.065,
            420: 71.577,
            425: 69.746,
            430: 67.914,
            435: 76.760,
            440: 85.605,
            445: 91.799,
            450: 97.993,
            455: 99.228,
            460: 100.463,
            465: 100.188,
            470: 99.913,
            475: 101.326,
            480: 102.739,
            485: 100.409,
            490: 98.078,
            495: 99.379,
            500: 100.680,
            505: 100.688,
            510: 100.695,
            515: 100.341,
            520: 99.987,
            525: 102.098,
            530: 104.210,
            535: 103.156,
            540: 102.102,
            545: 102.535,
            550: 102.968,
            555: 101.484,
            560: 100.000,
            565: 98.608,
            570: 97.216,
            575: 97.482,
            580: 97.749,
            585: 94.590,
            590: 91.432,
            595: 92.926,
            600: 94.419,
            605: 94.780,
            610: 95.140,
            615: 94.680,
            620: 94.220,
            625: 92.334,
            630: 90.448,
            635: 91.389,
            640: 92.330,
            645: 90.592,
            650: 88.854,
            655: 89.586,
            660: 90.317,
            665: 92.133,
            670: 93.950,
            675: 91.953,
            680: 89.956,
            685: 84.817,
            690: 79.677,
            695: 81.258,
            700: 82.840,
            705: 83.842,
            710: 84.844,
            715: 77.539,
            720: 70.235,
            725: 74.768,
            730: 79.301,
            735: 82.147,
            740: 84.993,
            745: 78.437,
            750: 71.880,
            755: 62.337,
            760: 52.793,
            765: 64.360,
            770: 75.927,
            775: 73.872,
            780: 71.818},
    "D75": {300: 0.043,
            305: 2.588,
            310: 5.133,
            315: 17.470,
            320: 29.808,
            325: 42.369,
            330: 54.930,
            335: 56.095,
            340: 57.259,
            345: 60.000,
            350: 62.740,
            355: 62.861,
            360: 62.982,
            365: 66.647,
            370: 70.312,
            375: 68.507,
            380: 66.703,
            385: 68.333,
            390: 69.963,
            395: 85.946,
            400: 101.929,
            405: 106.911,
            410: 111.894,
            415: 112.346,
            420: 112.798,
            425: 107.945,
            430: 103.092,
            435: 112.145,
            440: 121.198,
            445: 127.104,
            450: 133.010,
            455: 132.682,
            460: 132.355,
            465: 129.838,
            470: 127.322,
            475: 127.061,
            480: 126.800,
            485: 122.291,
            490: 117.783,
            495: 117.186,
            500: 116.589,
            505: 115.146,
            510: 113.702,
            515: 111.181,
            520: 108.659,
            525: 109.552,
            530: 110.445,
            535: 108.367,
            540: 106.289,
            545: 105.596,
            550: 104.904,
            555: 102.452,
            560: 100.000,
            565: 97.808,
            570: 95.616,
            575: 94.914,
            580: 94.213,
            585: 90.605,
            590: 86.997,
            595: 87.112,
            600: 87.227,
            605: 86.684,
            610: 86.140,
            615: 84.861,
            620: 83.581,
            625: 81.164,
            630: 78.747,
            635: 78.587,
            640: 78.428,
            645: 76.614,
            650: 74.801,
            655: 74.562,
            660: 74.324,
            665: 74.873,
            670: 75.422,
            675: 73.499,
            680: 71.576,
            685: 67.714,
            690: 63.852,
            695: 64.464,
            700: 65.076,
            705: 66.573,
            710: 68.070,
            715: 62.256,
            720: 56.443,
            725: 60.343,
            730: 64.242,
            735: 66.697,
            740: 69.151,
            745: 63.890,
            750: 58.629,
            755: 50.623,
            760: 42.617,
            765: 51.985,
            770: 61.352,
            775: 59.838,
            780: 58.324},
    "F1": {380: 1.87,
           385: 2.36,
           390: 2.94,
           395: 3.47,
           400: 5.17,
           405: 19.49,
           410: 6.13,
           415: 6.24,
           420: 7.01,
           425: 7.79,
           430: 8.56,
           435: 43.67,
           440: 16.94,
           445: 10.72,
           450: 11.35,
           455: 11.89,
           460: 12.37,
           465: 12.75,
           470: 13,
           475: 13.15,
           480: 13.23,
           485: 13.17,
           490: 13.13,
           495: 12.85,
           500: 12.52,
           505: 12.2,
           510: 11.83,
           515: 11.5,
           520: 11.22,
           525: 11.05,
           530: 11.03,
           535: 11.18,
           540: 11.53,
           545: 27.74,
           550: 17.05,
           555: 13.55,
           560: 14.33,
           565: 15.01,
           570: 15.52,
           575: 18.29,
           580: 19.55,
           585: 15.48,
           590: 14.91,
           595: 14.15,
           600: 13.22,
           605: 12.19,
           610: 11.12,
           615: 10.03,
           620: 8.95,
           625: 7.96,
           630: 7.02,
           635: 6.2,
           640: 5.42,
           645: 4.73,
           650: 4.15,
           655: 3.64,
           660: 3.2,
           665: 2.81,
           670: 2.47,
           675: 2.18,
           680: 1.93,
           685: 1.72,
           690: 1.67,
           695: 1.43,
           700: 1.29,
           705: 1.19,
           710: 1.08,
           715: 0.96,
           720: 0.88,
           725: 0.81,
           730: 0.77,
           735: 0.75,
           740: 0.73,
           745: 0.68,
           750: 0.69,
           755: 0.64,
           760: 0.68,
           765: 0.69,
           770: 0.61,
           775: 0.52,
           780: 0.43},
    "F2": {380: 1.18,
           385: 1.48,
           390: 1.84,
           395: 2.15,
           400: 3.44,
           405: 15.69,
           410: 3.85,
           415: 3.74,
           420: 4.19,
           425: 4.62,
           430: 5.06,
           435: 34.98,
           440: 11.81,
           445: 6.27,
           450: 6.63,
           455: 6.93,
           460: 7.19,
           465: 7.4,
           470: 7.54,
           475: 7.62,
           480: 7.65,
           485: 7.62,
           490: 7.62,
           495: 7.45,
           500: 7.28,
           505: 7.15,
           510: 7.05,
           515: 7.04,
           520: 7.16,
           525: 7.47,
           530: 8.04,
           535: 8.88,
           540: 10.01,
           545: 24.88,
           550: 16.64,
           555: 14.59,
           560: 16.16,
           565: 17.56,
           570: 18.62,
           575: 21.47,
           580: 22.79,
           585: 19.29,
           590: 18.66,
           595: 17.73,
           600: 16.54,
           605: 15.21,
           610: 13.8,
           615: 12.36,
           620: 10.95,
           625: 9.65,
           630: 8.4,
           635: 7.32,
           640: 6.31,
           645: 5.43,
           650: 4.68,
           655: 4.02,
           660: 3.45,
           665: 2.96,
           670: 2.55,
           675: 2.19,
           680: 1.89,
           685: 1.64,
           690: 1.53,
           695: 1.27,
           700: 1.1,
           705: 0.99,
           710: 0.88,
           715: 0.76,
           720: 0.68,
           725: 0.61,
           730: 0.56,
           735: 0.54,
           740: 0.51,
           745: 0.47,
           750: 0.47,
           755: 0.43,
           760: 0.46,
           765: 0.47,
           770: 0.4,
           775: 0.33,
           780: 0.27},
    "F3": {380: 0.82,
           385: 1.02,
           390: 1.26,
           395: 1.44,
           400: 2.57,
           405: 14.36,
           410: 2.7,
           415: 2.45,
           420: 2.73,
           425: 3,
           430: 3.28,
           435: 31.85,
           440: 9.47,
           445: 4.02,
           450: 4.25,
           455: 4.44,
           460: 4.59,
           465: 4.72,
           470: 4.8,
           475: 4.86,
           480: 4.87,
           485: 4.85,
           490: 4.88,
           495: 4.77,
           500: 4.67,
           505: 4.62,
           510: 4.62,
           515: 4.73,
           520: 4.99,
           525: 5.48,
           530: 6.25,
           535: 7.34,
           540: 8.78,
           545: 23.82,
           550: 16.14,
           555: 14.59,
           560: 16.63,
           565: 18.49,
           570: 19.95,
           575: 23.11,
           580: 24.69,
           585: 21.41,
           590: 20.85,
           595: 19.93,
           600: 18.67,
           605: 17.22,
           610: 15.65,
           615: 14.04,
           620: 12.45,
           625: 10.95,
           630: 9.51,
           635: 8.27,
           640: 7.11,
           645: 6.09,
           650: 5.22,
           655: 4.45,
           660: 3.8,
           665: 3.23,
           670: 2.75,
           675: 2.33,
           680: 1.99,
           685: 1.7,
           690: 1.55,
           695: 1.27,
           700: 1.09,
           705: 0.96,
           710: 0.83,
           715: 0.71,
           720: 0.62,
           725: 0.54,
           730: 0.49,
           735: 0.46,
           740: 0.43,
           745: 0.39,
           750: 0.39,
           755: 0.35,
           760: 0.38,
           765: 0.39,
           770: 0.33,
           775: 0.28,
           780: 0.21},
    "F4": {380: 0.57,
           385: 0.7,
           390: 0.87,
           395: 0.98,
           400: 2.01,
           405: 13.75,
           410: 1.95,
           415: 1.59,
           420: 1.76,
           425: 1.93,
           430: 2.1,
           435: 30.28,
           440: 8.03,
           445: 2.55,
           450: 2.7,
           455: 2.82,
           460: 2.91,
           465: 2.99,
           470: 3.04,
           475: 3.08,
           480: 3.09,
           485: 3.09,
           490: 3.14,
           495: 3.06,
           500: 3,
           505: 2.98,
           510: 3.01,
           515: 3.14,
           520: 3.41,
           525: 3.9,
           530: 4.69,
           535: 5.81,
           540: 7.32,
           545: 22.59,
           550: 15.11,
           555: 13.88,
           560: 16.33,
           565: 18.68,
           570: 20.64,
           575: 24.28,
           580: 26.26,
           585: 23.28,
           590: 22.94,
           595: 22.14,
           600: 20.91,
           605: 19.43,
           610: 17.74,
           615: 16,
           620: 14.42,
           625: 12.56,
           630: 10.93,
           635: 9.52,
           640: 8.18,
           645: 7.01,
           650: 6,
           655: 5.11,
           660: 4.36,
           665: 3.69,
           670: 3.13,
           675: 2.64,
           680: 2.24,
           685: 1.91,
           690: 1.7,
           695: 1.39,
           700: 1.18,
           705: 1.03,
           710: 0.88,
           715: 0.74,
           720: 0.64,
           725: 0.54,
           730: 0.49,
           735: 0.46,
           740: 0.42,
           745: 0.37,
           750: 0.37,
           755: 0.33,
           760: 0.35,
           765: 0.36,
           770: 0.31,
           775: 0.26,
           780: 0.19},
    "F5": {380: 1.87,
           385: 2.35,
           390: 2.92,
           395: 3.45,
           400: 5.1,
           405: 18.91,
           410: 6,
           415: 6.11,
           420: 6.85,
           425: 7.58,
           430: 8.31,
           435: 40.76,
           440: 16.06,
           445: 10.32,
           450: 10.91,
           455: 11.4,
           460: 11.83,
           465: 12.17,
           470: 12.4,
           475: 12.54,
           480: 12.58,
           485: 12.52,
           490: 12.47,
           495: 12.2,
           500: 11.89,
           505: 11.61,
           510: 11.33,
           515: 11.1,
           520: 10.96,
           525: 10.97,
           530: 11.16,
           535: 11.54,
           540: 12.12,
           545: 27.78,
           550: 17.73,
           555: 14.47,
           560: 15.2,
           565: 15.77,
           570: 16.1,
           575: 18.54,
           580: 19.5,
           585: 15.39,
           590: 14.64,
           595: 13.72,
           600: 12.69,
           605: 11.57,
           610: 10.45,
           615: 9.35,
           620: 8.29,
           625: 7.32,
           630: 6.41,
           635: 5.63,
           640: 4.9,
           645: 4.26,
           650: 3.72,
           655: 3.25,
           660: 2.83,
           665: 2.49,
           670: 2.19,
           675: 1.93,
           680: 1.71,
           685: 1.52,
           690: 1.48,
           695: 1.26,
           700: 1.13,
           705: 1.05,
           710: 0.96,
           715: 0.85,
           720: 0.78,
           725: 0.72,
           730: 0.68,
           735: 0.67,
           740: 0.65,
           745: 0.61,
           750: 0.62,
           755: 0.59,
           760: 0.62,
           765: 0.64,
           770: 0.55,
           775: 0.47,
           780: 0.4},
    "F6": {380: 1.05,
           385: 1.31,
           390: 1.63,
           395: 1.9,
           400: 3.11,
           405: 14.8,
           410: 3.43,
           415: 3.3,
           420: 3.68,
           425: 4.07,
           430: 4.45,
           435: 32.61,
           440: 10.74,
           445: 5.48,
           450: 5.78,
           455: 6.03,
           460: 6.25,
           465: 6.41,
           470: 6.52,
           475: 6.58,
           480: 6.59,
           485: 6.56,
           490: 6.56,
           495: 6.42,
           500: 6.28,
           505: 6.2,
           510: 6.19,
           515: 6.3,
           520: 6.6,
           525: 7.12,
           530: 7.94,
           535: 9.07,
           540: 10.49,
           545: 25.22,
           550: 17.46,
           555: 15.63,
           560: 17.22,
           565: 18.53,
           570: 19.43,
           575: 21.97,
           580: 23.01,
           585: 19.41,
           590: 18.56,
           595: 17.42,
           600: 16.09,
           605: 14.64,
           610: 13.15,
           615: 11.68,
           620: 10.25,
           625: 8.95,
           630: 7.74,
           635: 6.69,
           640: 5.71,
           645: 4.87,
           650: 4.16,
           655: 3.55,
           660: 3.02,
           665: 2.57,
           670: 2.2,
           675: 1.87,
           680: 1.6,
           685: 1.37,
           690: 1.29,
           695: 1.05,
           700: 0.91,
           705: 0.81,
           710: 0.71,
           715: 0.61,
           720: 0.54,
           725: 0.48,
           730: 0.44,
           735: 0.43,
           740: 0.4,
           745: 0.37,
           750: 0.38,
           755: 0.35,
           760: 0.39,
           765: 0.41,
           770: 0.33,
           775: 0.26,
           780: 0.21},
    "F7": {380: 2.56,
           385: 3.18,
           390: 3.84,
           395: 4.53,
           400: 6.15,
           405: 19.37,
           410: 7.37,
           415: 7.05,
           420: 7.71,
           425: 8.41,
           430: 9.15,
           435: 44.14,
           440: 17.52,
           445: 11.35,
           450: 12,
           455: 12.58,
           460: 13.08,
           465: 13.45,
           470: 13.71,
           475: 13.88,
           480: 13.95,
           485: 13.93,
           490: 13.82,
           495: 13.64,
           500: 13.43,
           505: 13.25,
           510: 13.08,
           515: 12.93,
           520: 12.78,
           525: 12.6,
           530: 12.44,
           535: 12.33,
           540: 12.26,
           545: 29.52,
           550: 17.05,
           555: 12.44,
           560: 12.58,
           565: 12.72,
           570: 12.83,
           575: 15.46,
           580: 16.75,
           585: 12.83,
           590: 12.67,
           595: 12.45,
           600: 12.19,
           605: 11.89,
           610: 11.6,
           615: 11.35,
           620: 11.12,
           625: 10.95,
           630: 10.76,
           635: 10.42,
           640: 10.11,
           645: 10.04,
           650: 10.02,
           655: 10.11,
           660: 9.87,
           665: 8.65,
           670: 7.27,
           675: 6.44,
           680: 5.83,
           685: 5.41,
           690: 5.04,
           695: 4.57,
           700: 4.12,
           705: 3.77,
           710: 3.46,
           715: 3.08,
           720: 2.73,
           725: 2.47,
           730: 2.25,
           735: 2.06,
           740: 1.9,
           745: 1.75,
           750: 1.62,
           755: 1.54,
           760: 1.45,
           765: 1.32,
           770: 1.17,
           775: 0.99,
           780: 0.81},
    "F8": {380: 1.21,
           385: 1.5,
           390: 1.81,
           395: 2.13,
           400: 3.17,
           405: 13.08,
           410: 3.83,
           415: 3.45,
           420: 3.86,
           425: 4.42,
           430: 5.09,
           435: 34.1,
           440: 12.42,
           445: 7.68,
           450: 8.6,
           455: 9.46,
           460: 10.24,
           465: 10.84,
           470: 11.33,
           475: 11.71,
           480: 11.98,
           485: 12.17,
           490: 12.28,
           495: 12.32,
           500: 12.35,
           505: 12.44,
           510: 12.55,
           515: 12.68,
           520: 12.77,
           525: 12.72,
           530: 12.6,
           535: 12.43,
           540: 12.22,
           545: 28.96,
           550: 16.51,
           555: 11.79,
           560: 11.76,
           565: 11.77,
           570: 11.84,
           575: 14.61,
           580: 16.11,
           585: 12.34,
           590: 12.53,
           595: 12.72,
           600: 12.92,
           605: 13.12,
           610: 13.34,
           615: 13.61,
           620: 13.87,
           625: 14.07,
           630: 14.2,
           635: 14.16,
           640: 14.13,
           645: 14.34,
           650: 14.5,
           655: 14.46,
           660: 14,
           665: 12.58,
           670: 10.99,
           675: 9.98,
           680: 9.22,
           685: 8.62,
           690: 8.07,
           695: 7.39,
           700: 6.71,
           705: 6.16,
           710: 5.63,
           715: 5.03,
           720: 4.46,
           725: 4.02,
           730: 3.66,
           735: 3.36,
           740: 3.09,
           745: 2.85,
           750: 2.65,
           755: 2.51,
           760: 2.37,
           765: 2.15,
           770: 1.89,
           775: 1.61,
           780: 1.32},
    "F9": {380: 0.9,
           385: 1.12,
           390: 1.36,
           395: 1.6,
           400: 2.59,
           405: 12.8,
           410: 3.05,
           415: 2.56,
           420: 2.86,
           425: 3.3,
           430: 3.82,
           435: 32.62,
           440: 10.77,
           445: 5.84,
           450: 6.57,
           455: 7.25,
           460: 7.86,
           465: 8.35,
           470: 8.75,
           475: 9.06,
           480: 9.31,
           485: 9.48,
           490: 9.61,
           495: 9.68,
           500: 9.74,
           505: 9.88,
           510: 10.04,
           515: 10.26,
           520: 10.48,
           525: 10.63,
           530: 10.78,
           535: 10.96,
           540: 11.18,
           545: 27.71,
           550: 16.29,
           555: 12.28,
           560: 12.74,
           565: 13.21,
           570: 13.65,
           575: 16.57,
           580: 18.14,
           585: 14.55,
           590: 14.65,
           595: 14.66,
           600: 14.61,
           605: 14.5,
           610: 14.39,
           615: 14.4,
           620: 14.47,
           625: 14.62,
           630: 14.72,
           635: 14.55,
           640: 14.4,
           645: 14.58,
           650: 14.88,
           655: 15.51,
           660: 15.47,
           665: 13.2,
           670: 10.57,
           675: 9.18,
           680: 8.25,
           685: 7.57,
           690: 7.03,
           695: 6.35,
           700: 5.72,
           705: 5.25,
           710: 4.8,
           715: 4.29,
           720: 3.8,
           725: 3.43,
           730: 3.12,
           735: 2.86,
           740: 2.64,
           745: 2.43,
           750: 2.26,
           755: 2.14,
           760: 2.02,
           765: 1.83,
           770: 1.61,
           775: 1.38,
           780: 1.12},
    "F10": {380: 1.11,
            385: 0.8,
            390: 0.62,
            395: 0.57,
            400: 1.48,
            405: 12.16,
            410: 2.12,
            415: 2.7,
            420: 3.74,
            425: 5.14,
            430: 6.75,
            435: 34.39,
            440: 14.86,
            445: 10.4,
            450: 10.76,
            455: 10.67,
            460: 10.11,
            465: 9.27,
            470: 8.29,
            475: 7.29,
            480: 7.91,
            485: 16.64,
            490: 16.73,
            495: 10.44,
            500: 5.94,
            505: 3.34,
            510: 2.35,
            515: 1.88,
            520: 1.59,
            525: 1.47,
            530: 1.8,
            535: 5.71,
            540: 40.98,
            545: 73.69,
            550: 33.61,
            555: 8.24,
            560: 3.38,
            565: 2.47,
            570: 2.14,
            575: 4.86,
            580: 11.45,
            585: 14.79,
            590: 12.16,
            595: 8.97,
            600: 6.52,
            605: 8.31,
            610: 44.12,
            615: 34.55,
            620: 12.09,
            625: 12.15,
            630: 10.52,
            635: 4.43,
            640: 1.95,
            645: 2.19,
            650: 3.19,
            655: 2.77,
            660: 2.29,
            665: 2,
            670: 1.52,
            675: 1.35,
            680: 1.47,
            685: 1.79,
            690: 1.74,
            695: 1.02,
            700: 1.14,
            705: 3.32,
            710: 4.49,
            715: 2.05,
            720: 0.49,
            725: 0.24,
            730: 0.21,
            735: 0.21,
            740: 0.24,
            745: 0.24,
            750: 0.21,
            755: 0.17,
            760: 0.21,
            765: 0.22,
            770: 0.17,
            775: 0.12,
            780: 0.09},
    "F11": {380: 0.91,
            385: 0.63,
            390: 0.46,
            395: 0.37,
            400: 1.29,
            405: 12.68,
            410: 1.59,
            415: 1.79,
            420: 2.46,
            425: 3.33,
            430: 4.49,
            435: 33.94,
            440: 12.13,
            445: 6.95,
            450: 7.19,
            455: 7.12,
            460: 6.72,
            465: 6.13,
            470: 5.46,
            475: 4.79,
            480: 5.66,
            485: 14.29,
            490: 14.96,
            495: 8.97,
            500: 4.72,
            505: 2.33,
            510: 1.47,
            515: 1.1,
            520: 0.89,
            525: 0.83,
            530: 1.18,
            535: 4.9,
            540: 39.59,
            545: 72.84,
            550: 32.61,
            555: 7.52,
            560: 2.83,
            565: 1.96,
            570: 1.67,
            575: 4.43,
            580: 11.28,
            585: 14.76,
            590: 12.73,
            595: 9.74,
            600: 7.33,
            605: 9.72,
            610: 55.27,
            615: 42.58,
            620: 13.18,
            625: 13.16,
            630: 12.26,
            635: 5.11,
            640: 2.07,
            645: 2.34,
            650: 3.58,
            655: 3.01,
            660: 2.48,
            665: 2.14,
            670: 1.54,
            675: 1.33,
            680: 1.46,
            685: 1.94,
            690: 2,
            695: 1.2,
            700: 1.35,
            705: 4.1,
            710: 5.58,
            715: 2.51,
            720: 0.57,
            725: 0.27,
            730: 0.23,
            735: 0.21,
            740: 0.24,
            745: 0.24,
            750: 0.2,
            755: 0.24,
            760: 0.32,
            765: 0.26,
            770: 0.16,
            775: 0.12,
            780: 0.09},
    "F12": {380: 0.96,
            385: 0.64,
            390: 0.4,
            395: 0.33,
            400: 1.19,
            405: 12.48,
            410: 1.12,
            415: 0.94,
            420: 1.08,
            425: 1.37,
            430: 1.78,
            435: 29.05,
            440: 7.9,
            445: 2.65,
            450: 2.71,
            455: 2.65,
            460: 2.49,
            465: 2.33,
            470: 2.1,
            475: 1.91,
            480: 3.01,
            485: 10.83,
            490: 11.88,
            495: 6.88,
            500: 3.43,
            505: 1.49,
            510: 0.92,
            515: 0.71,
            520: 0.6,
            525: 0.63,
            530: 1.1,
            535: 4.56,
            540: 34.4,
            545: 65.4,
            550: 29.48,
            555: 7.16,
            560: 3.08,
            565: 2.47,
            570: 2.27,
            575: 5.09,
            580: 11.96,
            585: 15.32,
            590: 14.27,
            595: 11.86,
            600: 9.28,
            605: 12.31,
            610: 68.53,
            615: 53.02,
            620: 14.67,
            625: 14.38,
            630: 14.71,
            635: 6.46,
            640: 2.57,
            645: 2.75,
            650: 4.18,
            655: 3.44,
            660: 2.81,
            665: 2.42,
            670: 1.64,
            675: 1.36,
            680: 1.49,
            685: 2.14,
            690: 2.34,
            695: 1.42,
            700: 1.61,
            705: 5.04,
            710: 6.98,
            715: 3.19,
            720: 0.71,
            725: 0.3,
            730: 0.26,
            735: 0.23,
            740: 0.28,
            745: 0.28,
            750: 0.21,
            755: 0.17,
            760: 0.21,
            765: 0.19,
            770: 0.15,
            775: 0.1,
            780: 0.05}
}

ILLUMINANTS_RELATIVE_SPD = {
    "A": SpectralPowerDistribution("A", ILLUMINANTS_RELATIVE_SPD_DATA.get("A")),
    "D65": SpectralPowerDistribution("D65", ILLUMINANTS_RELATIVE_SPD_DATA.get("D65")),
    "C": SpectralPowerDistribution("C", ILLUMINANTS_RELATIVE_SPD_DATA.get("C")),
    "D50": SpectralPowerDistribution("D50", ILLUMINANTS_RELATIVE_SPD_DATA.get("D50")),
    "D55": SpectralPowerDistribution("D55", ILLUMINANTS_RELATIVE_SPD_DATA.get("D55")),
    "D75": SpectralPowerDistribution("D75", ILLUMINANTS_RELATIVE_SPD_DATA.get("D75")),
    "F1": SpectralPowerDistribution("F1", ILLUMINANTS_RELATIVE_SPD_DATA.get("F1")),
    "F2": SpectralPowerDistribution("F2", ILLUMINANTS_RELATIVE_SPD_DATA.get("F2")),
    "F3": SpectralPowerDistribution("F3", ILLUMINANTS_RELATIVE_SPD_DATA.get("F3")),
    "F4": SpectralPowerDistribution("F4", ILLUMINANTS_RELATIVE_SPD_DATA.get("F4")),
    "F5": SpectralPowerDistribution("F5", ILLUMINANTS_RELATIVE_SPD_DATA.get("F5")),
    "F6": SpectralPowerDistribution("F6", ILLUMINANTS_RELATIVE_SPD_DATA.get("F6")),
    "F7": SpectralPowerDistribution("F7", ILLUMINANTS_RELATIVE_SPD_DATA.get("F7")),
    "F8": SpectralPowerDistribution("F8", ILLUMINANTS_RELATIVE_SPD_DATA.get("F8")),
    "F9": SpectralPowerDistribution("F9", ILLUMINANTS_RELATIVE_SPD_DATA.get("F9")),
    "F10": SpectralPowerDistribution("F10", ILLUMINANTS_RELATIVE_SPD_DATA.get("F10")),
    "F11": SpectralPowerDistribution("F11", ILLUMINANTS_RELATIVE_SPD_DATA.get("F11")),
    "F12": SpectralPowerDistribution("F12", ILLUMINANTS_RELATIVE_SPD_DATA.get("F12"))}

# *CIE D-Illuminant* Sn(λ) distributions.
# http://www.brucelindbloom.com/index.html?Eqn_DIlluminant.html
D_ILLUMINANTS_S_DISTRIBUTIONS = {
    "S0": {
        300: 0.04,
        310: 6.00,
        320: 29.60,
        330: 55.30,
        340: 57.30,
        350: 61.80,
        360: 61.50,
        370: 68.80,
        380: 63.40,
        390: 65.80,
        400: 94.80,
        410: 104.80,
        420: 105.90,
        430: 96.80,
        440: 113.90,
        450: 125.60,
        460: 125.50,
        470: 121.30,
        480: 121.30,
        490: 113.50,
        500: 113.10,
        510: 110.80,
        520: 106.50,
        530: 108.80,
        540: 105.30,
        550: 104.40,
        560: 100.00,
        570: 96.00,
        580: 95.10,
        590: 89.10,
        600: 90.50,
        610: 90.30,
        620: 88.40,
        630: 84.00,
        640: 85.10,
        650: 81.90,
        660: 82.60,
        670: 84.90,
        680: 81.30,
        690: 71.90,
        700: 74.30,
        710: 76.40,
        720: 63.30,
        730: 71.70,
        740: 77.00,
        750: 65.20,
        760: 47.70,
        770: 68.60,
        780: 65.00,
        790: 66.00,
        800: 61.00,
        810: 53.30,
        820: 58.90,
        830: 61.90
    },
    "S1": {
        300: 0.02,
        310: 4.50,
        320: 22.40,
        330: 42.00,
        340: 40.60,
        350: 41.60,
        360: 38.00,
        370: 43.40,
        380: 38.50,
        390: 35.00,
        400: 43.40,
        410: 46.30,
        420: 43.90,
        430: 37.10,
        440: 36.70,
        450: 35.90,
        460: 32.60,
        470: 27.90,
        480: 24.30,
        490: 20.10,
        500: 16.20,
        510: 13.20,
        520: 8.60,
        530: 6.10,
        540: 4.20,
        550: 1.90,
        560: 0.00,
        570: -1.60,
        580: -3.50,
        590: -3.50,
        600: -5.80,
        610: -7.20,
        620: -8.60,
        630: -9.50,
        640: -10.90,
        650: -10.70,
        660: -12.00,
        670: -14.00,
        680: -13.60,
        690: -12.00,
        700: -13.30,
        710: -12.90,
        720: -10.60,
        730: -11.60,
        740: -12.20,
        750: -10.20,
        760: -7.80,
        770: -11.20,
        780: -10.40,
        790: -10.60,
        800: -9.70,
        810: -8.30,
        820: -9.30,
        830: -9.80
    },
    "S2": {
        300: 0.0,
        310: 2.0,
        320: 4.0,
        330: 8.5,
        340: 7.8,
        350: 6.7,
        360: 5.3,
        370: 6.1,
        380: 3.0,
        390: 1.2,
        400: -1.1,
        410: -0.5,
        420: -0.7,
        430: -1.2,
        440: -2.6,
        450: -2.9,
        460: -2.8,
        470: -2.6,
        480: -2.6,
        490: -1.8,
        500: -1.5,
        510: -1.3,
        520: -1.2,
        530: -1.0,
        540: -0.5,
        550: -0.3,
        560: 0.0,
        570: 0.2,
        580: 0.5,
        590: 2.1,
        600: 3.2,
        610: 4.1,
        620: 4.7,
        630: 5.1,
        640: 6.7,
        650: 7.3,
        660: 8.6,
        670: 9.8,
        680: 10.2,
        690: 8.3,
        700: 9.6,
        710: 8.5,
        720: 7.0,
        730: 7.6,
        740: 8.0,
        750: 6.7,
        760: 5.2,
        770: 7.4,
        780: 6.8,
        790: 7.0,
        800: 6.4,
        810: 5.5,
        820: 6.1,
        830: 6.5}}


def D_illuminant_relative_spectral_power_distribution(xy, name=None):
    """
    Returns the relative spectral power distribution of given *CIE D Illuminant* *xy* chromaticity coordinates.

    Reference: http://www.brucelindbloom.com/Eqn_DIlluminant.html

    Usage::

        >>> D_illuminant_relative_spectral_power_distribution((0.34567, 0.35850))
        <color.spectral.SpectralPowerDistribution object at 0x101023590>

    :param xy: *xy* chromaticity coordinate.
    :type xy: tuple
    :param name: *CIE D Illuminant* name.
    :type name: str or unicode
    :return: *CIE D Illuminant* relative spectral power distribution.
    :rtype: SpectralPowerDistribution
    """

    M = 0.0241 + 0.2562 * xy[0] - 0.7341 * xy[1]
    M1 = (-1.3515 - 1.7703 * xy[0] + 5.9114 * xy[1]) / M
    M2 = (0.0300 - 31.4424 * xy[0] + 30.0717 * xy[1]) / M

    distribution = {}
    start, end, steps = SpectralPowerDistribution("Spd", D_ILLUMINANTS_S_DISTRIBUTIONS.get("S0")).shape
    for i in numpy.arange(start, end + steps, steps):
        S0 = D_ILLUMINANTS_S_DISTRIBUTIONS.get("S0").get(i)
        S1 = D_ILLUMINANTS_S_DISTRIBUTIONS.get("S1").get(i)
        S2 = D_ILLUMINANTS_S_DISTRIBUTIONS.get("S2").get(i)
        distribution[i] = S0 + M1 * S1 + M2 * S2

    return SpectralPowerDistribution(name, distribution)

# Add calculated *CIE D60 Illuminant* relative spectral power distribution.
# xy = color.illuminants.ILLUMINANTS.get("Standard CIE 1931 2 Degree Observer").get("D60")
ILLUMINANTS_RELATIVE_SPD["D60"] = D_illuminant_relative_spectral_power_distribution((0.32168, 0.33767), "D60")
