## bindata

This tool converts any file into managable Go source code. Useful for embedding
binary data into a go program. The file data is optionally gzip compressed
before being converted to a raw byte slice.

### Usage

The simplest invocation is to pass it only the input file name.
The output file and code settings are inferred from this automatically.

    $ go-bindata testdata/gophercolor.png
    [w] No output file specified. Using 'testdata/gophercolor.png.go'.
    [w] No package name specified. Using 'main'.
    [w] No function name specified. Using 'testdata_gophercolor_png'.

This creates the `testdata/gophercolor.png.go` file which has a package
declaration with name `main` and one function named `testdata_gophercolor_png` with
the following signature:

```go
func testdata_gophercolor_png() []byte
```

You can now simply include the new .go file in your program and call
`testdata_gophercolor_png()` to get the (uncompressed) image data. The function panics
if something went wrong during decompression. See the testdata directory for
example input and output files for various modes.

Aternatively, you can pipe the input file data into stdin. `go-bindata` will
then spit out the generated Go code to stdout. This does require explicitly
naming the desired function name, as it can not be inferred from the
input data. The package name will still default to 'main'.

     $ cat testdata/gophercolor.png | go-bindata -f gophercolor_png | gofmt

Invoke the program with the `-h` flag for more options.

In order to strip off a part of the generated function name, we can use the `-prefix` flag.
In the above example, the input file `testdata/gophercolor.png` yields a function named
`testdata_gophercolor_png`. If we want the `testdata` component to be left out, we invoke
the program as follows:

    $ go-bindata -prefix "testdata/" testdata/gophercolor.png


### Lower memory footprint

Using the `-nomemcopy` flag, will alter the way the output file is generated.
It will employ a hack that allows us to read the file data directly from
the compiled program's `.rodata` section. This ensures that when we call
call our generated function, we omit unnecessary memcopies.

The downside of this, is that it requires dependencies on the `reflect` and
`unsafe` packages. These may be restricted on platforms like AppEngine and
thus prevent you from using this mode.

Another disadvantage is that the byte slice we create, is strictly read-only.
For most use-cases this is not a problem, but if you ever try to alter the
returned byte slice, a runtime panic is thrown. Use this mode only on target
platforms where memory constraints are an issue.

The default behaviour is to use the old code generation method. This
prevents the two previously mentioned issues, but will employ at least one
extra memcopy and thus increase memory requirements.

For instance, consider the following two examples:

This would be the default mode, using an extra memcopy but gives a safe
implementation without dependencies on `reflect` and `unsafe`:

```go
func myfile() []byte {
    return []byte{0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a}
}
```

Here is the same functionality, but uses the `.rodata` hack.
The byte slice returned from this example can not be written to without
generating a runtime error.

```go
var _myfile = "\x89\x50\x4e\x47\x0d\x0a\x1a"

func myfile() []byte {
    var empty [0]byte
    sx := (*reflect.StringHeader)(unsafe.Pointer(&_myfile))
    b := empty[:]
    bx := (*reflect.SliceHeader)(unsafe.Pointer(&b))
    bx.Data = sx.Data
    bx.Len = len(_myfile)
    bx.Cap = bx.Len
    return b
}
```


### Optional compression

When the `-uncompressed` flag is given, the supplied resource is *not* GZIP compressed
before being turned into Go code. The data should still be accessed through
a function call, so nothing changes in the usage of the generated file.

This feature is useful if you do not care for compression, or the supplied
resource is already compressed. Doing it again would not add any value and may
even increase the size of the data.

The default behaviour of the program is to use compression.


### Table of Contents

With the `-toc` flag, we can have `go-bindata` create a table of contents for all the files
which have been generated by the tool. It does this by first generating a new file named
`bindata-toc.go`. This contains a global map of type `map[string] func() []byte`. It uses the
input filename as the key and the data function as the value. We can use this
to fetch all data for our files, matching a given pattern.

It then appands an `init` function to each generated file, which simply makes the data
function append itself to the global `bindata` map.

Once you have compiled your program with all these new files and run it, the map will
be populated by all generated data files.

**Note**: The `bindata-toc.go` file will not be created when we run in `pipe` mode.
The reason being, that the tool does not write any files at all, as it has no idea
where to save them. The data file is written to `stdout` instead after all.


#### Table of Contents keys

The keys used in the `go_bindata` map, are the same as the input file name passed to `go-bindata`.
This includes the fully qualified (absolute) path. In most cases, this is not desireable, as it
puts potentially sensitive information in your code base. For this purpose, the tool supplies
another command line flag `-prefix`. This accepts a portion of a path name, which should be
stripped off from the map keys and function names.

For example, running without the `-prefix` flag, we get:

	$ go-bindata /path/to/templates/foo.html
    
	go_bindata["/path/to/templates/foo.html"] = path_to_templates_foo_html

Running with the `-prefix` flag, we get:

	$ go-bindata -prefix "/path/to/" /path/to/templates/foo.html
    
	go_bindata["templates/foo.html"] = templates_foo_html


#### bindata-toc.go

The `bindata-toc.go` file is very simple and looks as follows:

```go
package $PACKAGENAME

// Global Table of Contents map. Generated by go-bindata.
// After startup of the program, all generated data files will
// put themselves in this map. The key is the full filename, as
// supplied to go-bindata.
var go_bindata = make(map[string] func() []byte)
```

#### Build tags

With the optional -tags flag, you can specify any go build tags that
must be fulfilled for the output file to be included in a build. This
is useful for including binary data in multiple formats, where the desired
format is specified at build time with the appropriate tag(s).

The tags are appended to a `// +build` line in the beginning of the output file
and must follow the build tags syntax specified by the go tool.
