/*
   Copyright 2020 Docker Compose CLI authors

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

package local

import (
	"bufio"
	"context"
	"io"
	"strings"
	"time"

	"github.com/docker/docker/api/types"
	"github.com/docker/docker/api/types/container"
	"github.com/docker/docker/api/types/mount"
	"github.com/docker/docker/api/types/network"
	"github.com/docker/docker/client"
	"github.com/docker/docker/pkg/stdcopy"
	"github.com/docker/docker/pkg/stringid"
	specs "github.com/opencontainers/image-spec/specs-go/v1"
	"github.com/pkg/errors"

	"github.com/docker/compose-cli/api/containers"
	"github.com/docker/compose-cli/api/errdefs"
	"github.com/docker/compose-cli/local/moby"
)

type containerService struct {
	apiClient client.APIClient
}

func (cs *containerService) Inspect(ctx context.Context, id string) (containers.Container, error) {

	c, err := cs.apiClient.ContainerInspect(ctx, id)
	if err != nil {
		return containers.Container{}, err
	}

	status := ""
	if c.State != nil {
		status = c.State.Status
	}

	command := ""
	if c.Config != nil &&
		c.Config.Cmd != nil {
		command = strings.Join(c.Config.Cmd, " ")
	}

	rc := moby.ToRuntimeConfig(&c)
	hc := moby.ToHostConfig(&c)

	return containers.Container{
		ID:         stringid.TruncateID(c.ID),
		Status:     status,
		Image:      c.Image,
		Command:    command,
		Platform:   c.Platform,
		Config:     rc,
		HostConfig: hc,
	}, nil
}

func (cs *containerService) List(ctx context.Context, all bool) ([]containers.Container, error) {
	css, err := cs.apiClient.ContainerList(ctx, types.ContainerListOptions{
		All: all,
	})

	if err != nil {
		return []containers.Container{}, err
	}

	var result []containers.Container
	for _, container := range css {
		result = append(result, containers.Container{
			ID:    stringid.TruncateID(container.ID),
			Image: container.Image,
			// TODO: `Status` is a human readable string ("Up 24 minutes"),
			// we need to return the `State` instead but first we need to
			// define an enum on the proto side with all the possible container
			// statuses. We also need to add a `Created` property on the gRPC side.
			Status:  container.Status,
			Command: container.Command,
			Ports:   moby.ToPorts(container.Ports),
		})
	}

	return result, nil
}

func (cs *containerService) Run(ctx context.Context, r containers.ContainerConfig) error {
	exposedPorts, hostBindings, err := moby.FromPorts(r.Ports)
	if err != nil {
		return err
	}

	var mounts []mount.Mount
	for _, v := range r.Volumes {
		tokens := strings.Split(v, ":")
		if len(tokens) != 2 {
			return errors.Wrapf(errdefs.ErrParsingFailed, "volume %q has invalid format", v)
		}
		src := tokens[0]
		tgt := tokens[1]
		mounts = append(mounts, mount.Mount{Type: "volume", Source: src, Target: tgt})
	}

	containerConfig := &container.Config{
		Image:        r.Image,
		Cmd:          r.Command,
		Labels:       r.Labels,
		Env:          r.Environment,
		ExposedPorts: exposedPorts,
	}
	hostConfig := &container.HostConfig{
		PortBindings:  hostBindings,
		Mounts:        mounts,
		AutoRemove:    r.AutoRemove,
		RestartPolicy: moby.ToRestartPolicy(r.RestartPolicyCondition),
		Resources: container.Resources{
			NanoCPUs: int64(r.CPULimit * 1e9),
			Memory:   int64(r.MemLimit),
		},
	}

	id, err := cs.create(ctx, containerConfig, hostConfig, nil, r.Platform, r.ID)
	if err != nil {
		return err
	}
	return cs.apiClient.ContainerStart(ctx, id, types.ContainerStartOptions{})
}

func (cs *containerService) create(ctx context.Context,
	containerConfig *container.Config,
	hostConfig *container.HostConfig,
	networkingConfig *network.NetworkingConfig,
	platform *specs.Platform, name string) (string, error) {
	created, err := cs.apiClient.ContainerCreate(ctx, containerConfig, hostConfig, networkingConfig, platform, name)

	if err != nil {
		if client.IsErrNotFound(err) {
			io, err := cs.apiClient.ImagePull(ctx, containerConfig.Image, types.ImagePullOptions{})
			if err != nil {
				return "", err
			}
			scanner := bufio.NewScanner(io)

			// Read the whole body, otherwise the pulling stops
			for scanner.Scan() {
			}

			if err = scanner.Err(); err != nil {
				return "", err
			}
			if err = io.Close(); err != nil {
				return "", err
			}
			created, err = cs.apiClient.ContainerCreate(ctx, containerConfig, hostConfig, networkingConfig, platform, name)
			if err != nil {
				return "", err
			}
		} else {
			return "", err
		}
	}
	return created.ID, nil
}

func (cs *containerService) Start(ctx context.Context, containerID string) error {
	return cs.apiClient.ContainerStart(ctx, containerID, types.ContainerStartOptions{})
}

func (cs *containerService) Stop(ctx context.Context, containerID string, timeout *uint32) error {
	var t *time.Duration
	if timeout != nil {
		timeoutValue := time.Duration(*timeout) * time.Second
		t = &timeoutValue
	}
	return cs.apiClient.ContainerStop(ctx, containerID, t)
}

func (cs *containerService) Kill(ctx context.Context, containerID string, signal string) error {
	return cs.apiClient.ContainerKill(ctx, containerID, signal)
}

func (cs *containerService) Exec(ctx context.Context, name string, request containers.ExecRequest) error {
	cec, err := cs.apiClient.ContainerExecCreate(ctx, name, types.ExecConfig{
		Cmd:          []string{request.Command},
		Tty:          true,
		AttachStdin:  true,
		AttachStdout: true,
		AttachStderr: true,
	})
	if err != nil {
		return err
	}
	resp, err := cs.apiClient.ContainerExecAttach(ctx, cec.ID, types.ExecStartCheck{
		Tty: true,
	})
	if err != nil {
		return err
	}
	defer resp.Close()

	readChannel := make(chan error, 10)
	writeChannel := make(chan error, 10)

	go func() {
		_, err := io.Copy(request.Stdout, resp.Reader)
		readChannel <- err
	}()

	go func() {
		_, err := io.Copy(resp.Conn, request.Stdin)
		writeChannel <- err
	}()

	for {
		select {
		case err := <-readChannel:
			return err
		case err := <-writeChannel:
			return err
		}
	}
}

func (cs *containerService) Logs(ctx context.Context, containerName string, request containers.LogsRequest) error {
	c, err := cs.apiClient.ContainerInspect(ctx, containerName)
	if err != nil {
		return err
	}

	r, err := cs.apiClient.ContainerLogs(ctx, containerName, types.ContainerLogsOptions{
		ShowStdout: true,
		ShowStderr: true,
		Follow:     request.Follow,
	})

	if err != nil {
		return err
	}

	// nolint errcheck
	defer r.Close()

	if c.Config.Tty {
		_, err = io.Copy(request.Writer, r)
	} else {
		_, err = stdcopy.StdCopy(request.Writer, request.Writer, r)
	}

	return err
}

func (cs *containerService) Delete(ctx context.Context, containerID string, request containers.DeleteRequest) error {
	err := cs.apiClient.ContainerRemove(ctx, containerID, types.ContainerRemoveOptions{
		Force: request.Force,
	})
	if client.IsErrNotFound(err) {
		return errors.Wrapf(errdefs.ErrNotFound, "container %q", containerID)
	}
	return err
}
