/*******************************************************************************
 * Copyright (c) 2019 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v20.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
import mocha from "mocha";
import { expect } from "chai";
import * as _ from "lodash";

import * as genericLib from "../../../lib/generic";
import { logMsg } from "../../../lib/utils";
import { fail } from "assert";

export default class LoggingTest {
    testName: string;

    constructor(testName: string) {
        this.testName = testName;
    }

    run(runOnly?: boolean): void {
        (runOnly ? describe.only : describe)(this.testName, () => {
            const logLevels: Array<string> = ["info", "warn", "error", "trace", "debug", "invalid"];
            const invalidLevel = _.cloneDeep(logLevels).pop();
            for (const level of logLevels) {
                it(`set logging level to ${level}`, async () => {
                    logMsg(this.testName, "it", `Beginning logging level test to set level=${level}`);

                    try {
                        await genericLib.setLoggingLevel(level);
                    } catch (err) {
                        if (level === invalidLevel) {
                            expect(err);
                            expect(err.message).to.equal(`${level} is not a valid value, the following log levels are available {"error":"error","warn":"warn","info":"info","debug":"debug","trace":"trace"}`);
                        } else {
                            fail(`setting log level test failed when setting it to ${level}`);
                        }
                    }

                    logMsg(this.testName, "it", `Ending logging level test to set level=${level}`);
                });
            }
            this.afterAllHook();
        });
    }

    private afterAllHook(): void {
        after(`reset log level to ${process.env.DEFAULT_LOG_LEVEL}`, async () => {
            logMsg(this.testName, "after", `Resetting log level to ${process.env.DEFAULT_LOG_LEVEL}`);
            await genericLib.setLoggingLevel(process.env.DEFAULT_LOG_LEVEL);
        });
    }
}
