/*******************************************************************************
 * Copyright (c) 2019 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v20.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
// Functions that can be used anywhere in the codebase
const http = require('follow-redirects').http;
const https = require('follow-redirects').https;
const fs = require('fs-extra');
const request = require('request');

// This if statement allows us to only include one utils function, exporting either
//      the Docker of K8s one depending on which environment we're in
module.exports = require((global.codewind.RUNNING_IN_K8S ? './kubernetesFunctions' : './dockerFunctions'));

// variable to do a async timeout
module.exports.timeout = ms => new Promise(resolve => setTimeout(resolve, ms));

/**
 * Function check if a file exists (is accessable)
 * Essentially an async version of fs.exists
 * @param file, the file
 * @return true if file is accessable, false if not
 */
module.exports.fileExists = async function fileExists(file) {
  try {
    await fs.access(file);
    return true;
  } catch (err) {
    return false;
  }
}

/**
 * Function to make a HTTP request using a promise instead of callback
 * @param options, the HTTP request options
 * @param body, the HTTP request body (optional)
 * @param secure, true for https requests, false for http requests (optional, default is http)
 * @return res, the response from the HTTP request
 */

module.exports.asyncHttpRequest = function asyncHttpRequest(options, body, secure = false) {
  return new Promise(function (resolve, reject) {
    let protocol = secure ? https : http;
    let req = protocol.request(options, function(res) {
      res.body = '';
      res.on('error', function(err) {
        return reject(err);
      });
      res.on('data', function (data) {
        res.body += data
      });
      res.on('end', function() {
        return resolve(res);
      });
    }).on('error', function(err) {
      return reject(err);
    });
    if (body) {
      req.write(JSON.stringify(body));
    }
    req.end();
  });
}

/**
 * Function to make a HTTP file download request using a promise
 * @param url, the URL target for the download
 * @param destination, destination file descriptor
 * @return res, the response from the HTTP request
 * @return promise, resolved if the download completed ok
 */
module.exports.asyncHttpDownload = function asyncHttpDownload(url, destination) {
  return new Promise(function (resolve, reject) {
    let stream = request.get({followAllRedirects: true, url: url})
      .on('error', function(err) {
        return reject(err);
      })
      .pipe(destination);
    stream.on('finish', function () {
      return resolve();
    });
  });
}

/**
 * Function which takes two Javascript Objects and updates the first
 *  with the fields in the second
 * @param objectToUpdate, an object which should be updated with the new fields/values
 * @param fieldsToAddToObject, an object which contains fields/values to add to the object
 * @return the updated object
 */
module.exports.updateObject = function updateObject(objectToUpdate, fieldsToAddToObject) {
  for (let key in fieldsToAddToObject) {
    objectToUpdate[key] = fieldsToAddToObject[key];
  }
  return objectToUpdate;
}


/** C:\helloThere -> /c/helloThere */
module.exports.convertFromWindowsDriveLetter = function convertFromWindowsDriveLetter(absolutePath) {

  if (!isWindowsAbsolutePath(absolutePath)) {
    return absolutePath;
  }
  let temp;
  // Replace \ with /
  temp = convertBackSlashesToForwardSlashes(absolutePath);
  const char0 = temp.charAt(0);
  // Strip first two characters
  temp = temp.substring(2);
  temp = "/" + char0.toLowerCase() + temp;
  return temp;

}

function convertBackSlashesToForwardSlashes(str) {
  return str.split("\\").join("/");
}

function isWindowsAbsolutePath(absolutePath) {
  if (absolutePath.length < 2) {
    return false;
  }
  const char0 = absolutePath.charAt(0);
  if (!isLetter(char0)) {
    return false;
  }
  if (absolutePath.charAt(1) !== ":") {
    return false;
  }
  return true;
}

function isLetter(currentChar) {
  return ("a" <= currentChar && currentChar <= "z")
      || ("A" <= currentChar && currentChar <= "Z");
}
