/*******************************************************************************
 * Copyright 2019 Dell Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 *******************************************************************************/

// notifications provides a client for integrating with the support-notifications service.
package notifications

import (
	"context"

	"github.com/edgexfoundry/go-mod-core-contracts/clients"
	"github.com/edgexfoundry/go-mod-core-contracts/clients/interfaces"
)

type CategoryEnum string

const (
	SECURITY  CategoryEnum = "SECURITY"
	HW_HEALTH CategoryEnum = "HW_HEALTH"
	SW_HEALTH CategoryEnum = "SW_HEALTH"
)

type SeverityEnum string

const (
	CRITICAL SeverityEnum = "CRITICAL"
	NORMAL   SeverityEnum = "NORMAL"
)

type StatusEnum string

const (
	NEW       StatusEnum = "NEW"
	PROCESSED StatusEnum = "PROCESSED"
	ESCALATED StatusEnum = "ESCALATED"
)

// NotificationsClient defines the interface for interactions with the EdgeX Foundry support-notifications service.
type NotificationsClient interface {
	// SendNotification sends a notification.
	SendNotification(ctx context.Context, n Notification) error
}

// Type struct for REST-specific implementation of the NotificationsClient interface
type notificationsRestClient struct {
	urlClient interfaces.URLClient
}

// Notification defines the structure of data being sent.
type Notification struct {
	Id          string       `json:"id,omitempty"`          // Generated by the system, users can ignore
	Slug        string       `json:"slug"`                  // A meaningful identifier provided by client
	Sender      string       `json:"sender"`                // Indicates the sender of a notification -- possibly a service name or email address
	Category    CategoryEnum `json:"category"`              // Allows for categorization of notifications
	Severity    SeverityEnum `json:"severity"`              // Denotes the severity of a notification
	Content     string       `json:"content"`               // Content contains the body of the notification
	Description string       `json:"description,omitempty"` // Description content for the notification
	Status      StatusEnum   `json:"status,omitempty"`      // Status reflects a simple workflow assignment for the notification
	Labels      []string     `json:"labels,omitempty"`      // Labels allows the notification to be further described/classified
	Created     int          `json:"created,omitempty"`     // The creation timestamp
	Modified    int          `json:"modified,omitempty"`    // The last modification timestamp
}

// NewNotificationsClient creates an instance of NotificationsClient
func NewNotificationsClient(urlClient interfaces.URLClient) NotificationsClient {
	return &notificationsRestClient{
		urlClient: urlClient,
	}
}

func (nc *notificationsRestClient) SendNotification(ctx context.Context, n Notification) error {
	_, err := clients.PostJSONRequest(ctx, "", n, nc.urlClient)
	return err
}
