//
// Copyright (C) 2020-2021 IOTech Ltd
//
// SPDX-License-Identifier: Apache-2.0

package models

import (
	"encoding/json"

	"github.com/edgexfoundry/go-mod-core-contracts/v2/errors"
)

// Subscription and its properties are defined in the APIv2 specification:
// https://app.swaggerhub.com/apis-docs/EdgeXFoundry1/support-notifications/2.x#/Subscription
// Model fields are same as the DTOs documented by this swagger. Exceptions, if any, are noted below.
type Subscription struct {
	DBTimestamp
	Categories     []string
	Labels         []string
	Channels       []Address
	Description    string
	Id             string
	Receiver       string
	Name           string
	ResendLimit    int64
	ResendInterval string
}

// ChannelType controls the range of values which constitute valid delivery types for channels
type ChannelType string

func (subscription *Subscription) UnmarshalJSON(b []byte) error {
	var alias struct {
		DBTimestamp
		Categories     []string
		Labels         []string
		Channels       []interface{}
		Description    string
		Id             string
		Receiver       string
		Name           string
		ResendLimit    int64
		ResendInterval string
	}
	if err := json.Unmarshal(b, &alias); err != nil {
		return errors.NewCommonEdgeX(errors.KindContractInvalid, "Failed to unmarshal intervalAction.", err)
	}
	channels := make([]Address, len(alias.Channels))
	for i, c := range alias.Channels {
		address, err := instantiateAddress(c)
		if err != nil {
			return errors.NewCommonEdgeXWrapper(err)
		}
		channels[i] = address
	}

	*subscription = Subscription{
		DBTimestamp:    alias.DBTimestamp,
		Categories:     alias.Categories,
		Labels:         alias.Labels,
		Description:    alias.Description,
		Id:             alias.Id,
		Receiver:       alias.Receiver,
		Name:           alias.Name,
		ResendLimit:    alias.ResendLimit,
		ResendInterval: alias.ResendInterval,
		Channels:       channels,
	}
	return nil
}
