## Internationalization

The default language of Element is Chinese. If you wish to use another language, you'll need to do some i18n configuration. In your entry file, if you are importing Element entirely:

```javascript
import Vue from 'vue'
import ElementUI from 'element-ui'
import locale from 'element-ui/lib/locale/lang/en'

Vue.use(ElementUI, { locale })
```

Or if you are importing Element on demand:

```javascript
import Vue from 'vue'
import { Button, Select } from 'element-ui'
import lang from 'element-ui/lib/locale/lang/en'
import locale from 'element-ui/lib/locale'

// configure language
locale.use(lang)

// import components
Vue.component(Button.name, Button)
Vue.component(Select.name, Select)
```

The Chinese language pack is imported by default, even if you're using another language. But with `NormalModuleReplacementPlugin` provided by webpack you can replace default locale:

webpack.config.js
```javascript
{
  plugins: [
    new webpack.NormalModuleReplacementPlugin(/element-ui[\/\\]lib[\/\\]locale[\/\\]lang[\/\\]zh-CN/, 'element-ui/lib/locale/lang/en')
  ]
}
```

## Compatible with `vue-i18n`

Element is compatible with [vue-i18n](https://github.com/kazupon/vue-i18n), which makes multilingual switching even easier.

```javascript
import Vue from 'vue'
import VueI18n from 'vue-i18n'
import Element from 'element-ui'
import enLocale from 'element-ui/lib/locale/lang/en'
import zhLocale from 'element-ui/lib/locale/lang/zh-CN'

Vue.use(VueI18n)
Vue.use(Element)

Vue.config.lang = 'zh-cn'
Vue.locale('zh-cn', zhLocale)
Vue.locale('en', enLocale)
```

## Compatibility with other i18n plugins
Element may not be compatible with i18n plugins other than vue-i18n, but you can customize how Element processes i18n.

```javascript
import Vue from 'vue'
import Element from 'element-ui'
import enLocale from 'element-ui/lib/locale/lang/en'
import zhLocale from 'element-ui/lib/locale/lang/zh-CN'

Vue.use(Element, {
  i18n: function (path, options) {
    // ...
  }
})
```

## Import via CDN

```html
<script src="//unpkg.com/vue"></script>
<script src="//unpkg.com/element-ui"></script>
<script src="//unpkg.com/element-ui/lib/umd/locale/en.js"></script>

<script>
  ELEMENT.locale(ELEMENT.lang.en)
</script>
```

Compatible with `vue-i18n`

```html
<script src="//unpkg.com/vue"></script>
<script src="//unpkg.com/vue-i18n/dist/vue-i18n.js"></script>
<script src="//unpkg.com/element-ui"></script>
<script src="//unpkg.com/element-ui/lib/umd/locale/zh-CN.js"></script>
<script src="//unpkg.com/element-ui/lib/umd/locale/en.js"></script>

<script>
  Vue.locale('en', ELEMENT.lang.en)
  Vue.locale('zh-cn', ELEMENT.lang.zhCN)
</script>
```

Currently Element ships with the following languages:
<ul class="language-list">
  <li>Simplified Chinese (zh-CN)</li>
  <li>English (en)</li>
  <li>German (de)</li>
  <li>Portuguese (pt)</li>
  <li>Spanish (es)</li>
  <li>Danish (da)</li>
  <li>French (fr)</li>
  <li>Norwegian (nb-NO)</li>
  <li>Traditional Chinese (zh-TW)</li>
  <li>Italian (it)</li>
  <li>Korean (ko)</li>
  <li>Japanese (ja)</li>
  <li>Dutch (nl)</li>
  <li>Vietnamese (vi)</li>
  <li>Russian (ru-RU)</li>
  <li>Turkish (tr-TR)</li>
  <li>Brazilian Portuguese (pt-br)</li>
  <li>Farsi (fa)</li>
  <li>Thai (th)</li>
  <li>Indonesian (id)</li>
  <li>Bulgarian (bg)</li>
  <li>Polish (pl)</li>
  <li>Finnish (fi)</li>
  <li>Swedish (sv-SE)</li>
  <li>Greek (el)</li>
  <li>Slovak (sk)</li>
</ul>

If your target language is not included, you are more than welcome to contribute: just add another language config [here](https://github.com/ElemeFE/element/tree/master/src/locale/lang) and create a pull request.
