/* global */
const EventEmitter = require('events')
const $ = require('jquery')
const yo = require('yo-yo')
const copy = require('copy-text-to-clipboard')
var QueryParams = require('../../lib/query-params')
const TreeView = require('../ui/TreeView')
const modalDialog = require('../ui/modaldialog')
const copyToClipboard = require('../ui/copy-to-clipboard')
const modalDialogCustom = require('../ui/modal-dialog-custom')
const parseContracts = require('./compileTab/contractParser')
const addTooltip = require('../ui/tooltip')
const globalRegistry = require('../../global/registry')

var css = require('./styles/compile-tab-styles')

const CompileTabLogic = require('./compileTab/compileTab.js')
const CompilerContainer = require('./compileTab/compilerContainer.js')

import { ViewPlugin } from '@remixproject/engine'
import * as packageJson from '../../../package.json'
import publishToStorage from '../../publishToStorage'

const profile = {
  name: 'solidity',
  displayName: 'Solidity compiler',
  icon: 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAB4AAAAeCAYAAAA7MK6iAAAACXBIWXMAAASdAAAEnQF8NGuhAAAAB3RJTUUH4wQMDx84DVYryQAAAjBJREFUSMe1102ITmEUB/DfO4aZRuM7otQgqSlCFmIhSRaSJClWs9BY2RAba1tLZTGxmbJWImFsWBhhZONzpJSEaJhhPl6bM/V2ux/vnTtz6untPvc857z/8/E/z61pXnZjHeoFejcwUWSs1qTTFjzHOP7l2PqCAVxuxmAzcgbLsSQH8SQ+YxM6igzOa8LpMvSH4dYI43iK3nDs17AND6oiPt+Qs3qgTso4fjU8r0Z3Fcfd6E0505nYe52olyn0VAn1FaxM2W/HSETgN76l6HREet6URbwPe3LeLw5k73OK7UDZdmrBIJYWROse3hak8gluJ1+0ZhyYwlNsLyCMHjOUvFCfij+Q19vmwjFcy9D5gUdVHDdDmY8xP3HmULDUnCGGswmnL3Oc7sBxsygDUeVDaMvR68fDnKItPSROBNo+/M3QOYwtwWq9s4n6ajBWltyJqAziVQbjlZpO03IzZ8CfDpab7vmJGKP3q14E8mQR7qaAaMdJvKiS4zw5lxG5MVyoWlxZshFHI8RpazP2lgl1DRdjAmVxdR070VVAUIM4Uqa4PuFg6LSlrFVRQJ3hIG2NY1fZUH/Asxy0a+L3a07H9M20nYZjmE8mnK5omNWTWJgCYhTHZup4BAuwPjHDNyT0/iTuYbXo7XdVqvpWA/fWI7dpF4exhufvwWSVmGv66ro10HdlVPpokEkd+/FzNobEQKBY23AuuWpx4xzCxyKDrSXI4nrkPO+DrA2XmjH2H8KUd4MWwdIJAAAAAElFTkSuQmCC',
  description: 'Compile solidity contracts',
  kind: 'compiler',
  permission: true,
  location: 'sidePanel',
  documentation: 'https://remix-ide.readthedocs.io/en/latest/solidity_editor.html',
  version: packageJson.version,
  methods: ['getCompilationResult', 'compile']

}

// EditorApi:
// - events: ['compilationFinished'],
// - methods: ['getCompilationResult']

class CompileTab extends ViewPlugin {

  constructor (editor, config, renderer, fileProvider, fileManager) {
    super(profile)
    this.events = new EventEmitter()
    this._view = {
      el: null,
      warnCompilationSlow: null,
      errorContainer: null,
      contractEl: null
    }
    this.queryParams = new QueryParams()
    this.fileProvider = fileProvider
    // dependencies
    this.editor = editor
    this.config = config
    this.renderer = renderer
    this.fileManager = fileManager

    this.data = {
      contractsDetails: {},
      eventHandlers: {}
    }
  }

  onActivationInternal () {
    this.compileTabLogic = new CompileTabLogic(this.queryParams, this.fileManager, this.editor, this.config, this.fileProvider)
    this.compiler = this.compileTabLogic.compiler
    this.compileTabLogic.init()

    this.compilerContainer = new CompilerContainer(
      this.compileTabLogic,
      this.editor,
      this.config,
      this.queryParams
    )
  }

  /************
   * EVENTS
   */

  listenToEvents () {
    this.data.eventHandlers.onContentChanged = () => {
      this.emit('statusChanged', {key: 'edited', title: 'the content has changed, needs recompilation', type: 'info'})
    }
    this.editor.event.register('contentChanged', this.data.eventHandlers.onContentChanged)

    this.data.eventHandlers.onLoadingCompiler = () => {
      this.emit('statusChanged', {key: 'loading', title: 'loading compiler...', type: 'info'})
    }
    this.compiler.event.register('loadingCompiler', this.data.eventHandlers.onLoadingCompiler)

    this.data.eventHandlers.onCompilerLoaded = () => {
      this.emit('statusChanged', {key: 'none'})
    }
    this.compiler.event.register('compilerLoaded', this.data.eventHandlers.onCompilerLoaded)

    this.data.eventHandlers.onStartingCompilation = () => {
      if (this._view.errorContainer) {
        this._view.errorContainer.innerHTML = ''
      }
      this.emit('statusChanged', {key: 'loading', title: 'compiling...', type: 'info'})
    }
    this.compileTabLogic.event.on('startingCompilation', this.data.eventHandlers.onStartingCompilation)

    this.data.eventHandlers.onCurrentFileChanged = (name) => {
      this.compilerContainer.currentFile = name
    }
    this.fileManager.events.on('currentFileChanged', this.data.eventHandlers.onCurrentFileChanged)

    this.data.eventHandlers.onNoFileSelected = () => {
      this.compilerContainer.currentFile = ''
    }
    this.fileManager.events.on('noFileSelected', this.data.eventHandlers.onNoFileSelected)

    this.data.eventHandlers.onCompilationFinished = (success, data, source) => {
      if (success) {
        // forwarding the event to the appManager infra
        this.emit('compilationFinished', source.target, source, 'soljson', data)
        if (data.errors && data.errors.length > 0) {
          this.emit('statusChanged', {
            key: data.errors.length,
            title: `compilation finished successful with warning${data.errors.length > 1 ? 's' : ''}`,
            type: 'warning'
          })
        } else this.emit('statusChanged', {key: 'succeed', title: 'compilation successful', type: 'success'})
        // Store the contracts
        this.data.contractsDetails = {}
        this.compiler.visitContracts((contract) => {
          this.data.contractsDetails[contract.name] = parseContracts(
            contract.name,
            contract.object,
            this.compiler.getSource(contract.file)
          )
        })
      } else {
        const count = (data.errors ? data.errors.filter(error => error.severity === 'error').length : 0 + data.error ? 1 : 0)
        this.emit('statusChanged', {key: count, title: `compilation failed with ${count} error${count.length > 1 ? 's' : ''}`, type: 'error'})
      }
      // Update contract Selection
      let contractMap = {}
      if (success) this.compiler.visitContracts((contract) => { contractMap[contract.name] = contract })
      let contractSelection = this.contractSelection(contractMap)
      yo.update(this._view.contractSelection, contractSelection)

      if (data['error']) {
        this.renderer.error(
          data['error'].formattedMessage || data['error'],
          this._view.errorContainer,
          {type: data['error'].severity || 'error'}
        )
        if (data['error'].mode === 'panic') {
          return modalDialogCustom.alert(yo`
            <div><i class="fas fa-exclamation-circle ${css.panicError}" aria-hidden="true"></i>
            The compiler returned with the following internal error: <br> <b>${data['error'].formattedMessage}.<br>
            The compiler might be in a non-sane state, please be careful and do not use further compilation data to deploy to mainnet.
            It is heavily recommended to use another browser not affected by this issue (Firefox is known to not be affected).</b><br>
            Please join <a href="https://gitter.im/ethereum/remix" target="blank" >remix gitter channel</a> for more information.</div>`)
        }
      }
      if (data.errors && data.errors.length) {
        data.errors.forEach((err) => {
          if (this.config.get('hideWarnings')) {
            if (err.severity !== 'warning') {
              this.renderer.error(err.formattedMessage, this._view.errorContainer, {type: err.severity})
            }
          } else {
            this.renderer.error(err.formattedMessage, this._view.errorContainer, {type: err.severity})
          }
        })
      }
    }
    this.compiler.event.register('compilationFinished', this.data.eventHandlers.onCompilationFinished)

    this.data.eventHandlers.onThemeChanged = (theme) => {
      const invert = theme.quality === 'dark' ? 1 : 0
      const img = document.getElementById('swarmLogo')
      if (img) {
        img.style.filter = `invert(${invert})`
      }
    }
    globalRegistry.get('themeModule').api.events.on('themeChanged', this.data.eventHandlers.onThemeChanged)

    // Run the compiler instead of trying to save the website
    $(window).keydown((e) => {
      // ctrl+s or command+s
      if ((e.metaKey || e.ctrlKey) && e.keyCode === 83) {
        e.preventDefault()
        this.compileTabLogic.runCompiler()
      }
    })
  }

  getCompilationResult () {
    return this.compileTabLogic.compiler.state.lastCompilationResult
  }

  // This function is used by remix-plugin
  compile (fileName) {
    return this.compileTabLogic.compileFile(fileName)
  }

  // This function is used for passing the compiler remix-tests
  getCurrentVersion () {
    return this.compilerContainer.data.selectedVersion
  }

  // This function is used for passing the compiler configuration to 'remix-tests'
  getCurrentCompilerConfig () {
    return {
      currentVersion: this.compilerContainer.data.selectedVersion,
      evmVersion: this.compileTabLogic.evmVersion,
      optimize: this.compileTabLogic.optimize
    }
  }

  /*********
   * SUB-COMPONENTS
   */

  /**
   * Section to select the compiled contract
   * @param {string[]} contractList Names of the compiled contracts
   */
  contractSelection (contractMap) {
    // Return the file name of a path: ex "browser/ballot.sol" -> "ballot.sol"
    const getFileName = (path) => {
      const part = path.split('/')
      return part[part.length - 1]
    }
    const contractList = contractMap ? Object.keys(contractMap).map((key) => ({
      name: key,
      file: getFileName(contractMap[key].file)
    })) : []
    let selectEl = yo`
      <select
        onchange="${e => this.selectContract(e.target.value)}"
        data-id="compiledContracts" id="compiledContracts" class="custom-select"
      >
        ${contractList.map(({name, file}) => yo`<option value="${name}">${name} (${file})</option>`)}
      </select>
    `
    // define swarm logo
    const swarmImg = 'data:image/png;base64,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'

    const ipfsImg = 'data:image/png;base64,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'

    let result = contractList.length
    ? yo`<section class="${css.compilerSection} pt-3">
      <!-- Select Compiler Version -->
      <div class="mb-3">
        <label class="${css.compilerLabel} form-check-label" for="compiledContracts">Contract</label>
        ${selectEl}
      </div>
      <article class="mt-2 pb-0">
        <button id="publishOnSwarm" class="btn btn-secondary btn-block" title="Publish on Swarm" onclick="${() => { publishToStorage('swarm', this.fileProvider, this.fileManager, this.data.contractsDetails[this.selectedContract]) }}">
          <span>Publish on Swarm</span>
          <img id="swarmLogo" class="${css.storageLogo} ml-2" src="${swarmImg}">
        </button>
        <button id="publishOnIpfs" class="btn btn-secondary btn-block" title="Publish on Ipfs" onclick="${() => { publishToStorage('ipfs', this.fileProvider, this.fileManager, this.data.contractsDetails[this.selectedContract]) }}">
        <span>Publish on Ipfs</span>
        <img id="ipfsLogo" class="${css.storageLogo} ml-2" src="${ipfsImg}">
      </button>
        <button class="btn btn-secondary btn-block" title="Display Contract Details" onclick="${() => { this.details() }}">
          Compilation Details
        </button>
        <!-- Copy to Clipboard -->
        <div class="${css.contractHelperButtons}">
          <div class="input-group">
            <div class="btn-group" role="group" aria-label="Copy to Clipboard">
              <button class="btn ${css.copyButton}" title="Copy ABI to clipboard" onclick="${() => { this.copyABI() }}">
                <i class="${css.copyIcon} far fa-copy" aria-hidden="true"></i>
                <span>ABI</span>
              </button>
              <button class="btn ${css.copyButton}" title="Copy Bytecode to clipboard" onclick="${() => { this.copyBytecode() }}">
                <i class="${css.copyIcon} far fa-copy" aria-hidden="true"></i>
                <span>Bytecode</span>
              </button>
            </div>
          </div>
        </div>
      </div>
    </section>`
    : yo`<section class="${css.container} clearfix"><article class="px-2 mt-2 pb-0 d-flex">
      <span class="mt-2 mx-3 w-100 alert alert-warning" role="alert">No Contract Compiled Yet</span>
    </article></section>`

    if (contractList.length) {
      this.selectedContract = selectEl.value
    } else {
      delete this.selectedContract
    }
    return result
  }

  // TODO : Add success alert when compilation succeed
  contractCompiledSuccess () {
    return yo`<div></div>`
  }
  // TODO : Add error alert when compilation failed
  contractCompiledError () {
    return yo`<div></div>`
  }

  /************
   * METHODS
   */

  selectContract (contractName) {
    this.selectedContract = contractName
  }

  details () {
    const help = {
      'Assembly': 'Assembly opcodes describing the contract including corresponding solidity source code',
      'Opcodes': 'Assembly opcodes describing the contract',
      'Runtime Bytecode': 'Bytecode storing the state and being executed during normal contract call',
      'bytecode': 'Bytecode being executed during contract creation',
      'functionHashes': 'List of declared function and their corresponding hash',
      'gasEstimates': 'Gas estimation for each function call',
      'metadata': 'Contains all informations related to the compilation',
      'metadataHash': 'Hash representing all metadata information',
      'abi': 'ABI: describing all the functions (input/output params, scope, ...)',
      'name': 'Name of the compiled contract',
      'swarmLocation': 'Swarm url where all metadata information can be found (contract needs to be published first)',
      'web3Deploy': 'Copy/paste this code to any JavaScript/Web3 console to deploy this contract'
    }
    if (!this.selectedContract) throw new Error('No contract compiled yet')
    const contractProperties = this.data.contractsDetails[this.selectedContract]
    const log = yo`<div class="${css.detailsJSON}"></div>`
    Object.keys(contractProperties).map(propertyName => {
      const copyDetails = yo`<span class="${css.copyDetails}">${copyToClipboard(() => contractProperties[propertyName])}</span>`
      const questionMark = yo`<span class="${css.questionMark}"><i title="${help[propertyName]}" class="fas fa-question-circle" aria-hidden="true"></i></span>`
      log.appendChild(yo`<div class=${css.log}>
        <div class="${css.key}">${propertyName} ${copyDetails} ${questionMark}</div>
        ${this.insertValue(contractProperties, propertyName)}
      </div>`)
    })
    modalDialog(this.selectedContract, log, { label: '' }, { label: 'Close' })
  }

  insertValue (details, propertyName) {
    var node
    if (propertyName === 'web3Deploy' || propertyName === 'name' || propertyName === 'Assembly') {
      node = yo`<pre>${details[propertyName]}</pre>`
    } else if (propertyName === 'abi' || propertyName === 'metadata') {
      const treeView = new TreeView({
        extractData: function (item, parent, key) {
          var ret = {}
          if (item instanceof Array) {
            ret.children = item.map((item, index) => ({ key: index, value: item }))
            ret.self = ''
          } else if (item instanceof Object) {
            ret.children = Object.keys(item).map((key) => ({key: key, value: item[key]}))
            ret.self = ''
          } else {
            ret.self = item
            ret.children = []
          }
          return ret
        }
      })
      if (details[propertyName] !== '') {
        try {
          node = yo`
          <div>
            ${treeView.render(typeof details[propertyName] === 'object' ? details[propertyName] : JSON.parse(details[propertyName]))}
          </div>` // catch in case the parsing fails.
        } catch (e) {
          node = yo`<div>Unable to display "${propertyName}": ${e.message}</div>`
        }
      } else {
        node = yo`<div> - </div>`
      }
    } else {
      node = yo`<div>${JSON.stringify(details[propertyName], null, 4)}</div>`
    }
    return yo`<pre class="${css.value}">${node || ''}</pre>`
  }

  getContractProperty (property) {
    if (!this.selectedContract) throw new Error('No contract compiled yet')
    const contractProperties = this.data.contractsDetails[this.selectedContract]
    return contractProperties[property] || null
  }

  copyContractProperty (property) {
    let content = this.getContractProperty(property)
    if (!content) {
      addTooltip('No content available for ' + property)
      return
    }

    try {
      if (typeof content !== 'string') {
        content = JSON.stringify(content, null, '\t')
      }
    } catch (e) {}

    copy(content)
    addTooltip('Copied value to clipboard')
  }

  copyABI () {
    this.copyContractProperty('abi')
  }

  copyBytecode () {
    this.copyContractProperty('bytecode')
  }

  render () {
    if (this._view.el) return this._view.el
    this.onActivationInternal()
    this._view.errorContainer = yo`<div class="${css.errorBlobs} p-4"></div>`
    this._view.contractSelection = this.contractSelection()
    this._view.compilerContainer = this.compilerContainer.render()
    this.compilerContainer.activate()
    this._view.el = yo`
      <div id="compileTabView">
        ${this._view.compilerContainer}
        ${this._view.contractSelection}
        ${this._view.errorContainer}
      </div>`
    return this._view.el
  }

  onActivation () {
    this.listenToEvents()
  }

  onDeactivation () {
    this.compilerContainer.deactivate()
    this.editor.event.unregister('contentChanged', this.data.eventHandlers.onContentChanged)
    this.compiler.event.unregister('loadingCompiler', this.data.eventHandlers.onLoadingCompiler)
    this.compiler.event.unregister('compilerLoaded', this.data.eventHandlers.onCompilerLoaded)
    this.compileTabLogic.event.removeListener('startingCompilation', this.data.eventHandlers.onStartingCompilation)
    this.fileManager.events.removeListener('currentFileChanged', this.data.eventHandlers.onCurrentFileChanged)
    this.fileManager.events.removeListener('noFileSelected', this.data.eventHandlers.onNoFileSelected)
    this.compiler.event.unregister('compilationFinished', this.data.eventHandlers.onCompilationFinished)
    globalRegistry.get('themeModule').api.events.removeListener('themeChanged', this.data.eventHandlers.onThemeChanged)
  }
}

module.exports = CompileTab
