/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.arch.persistence.room.migration;

import android.arch.persistence.db.SupportSQLiteDatabase;

/**
 * Base class for a database migration.
 * <p>
 * Each migration can move between 2 versions that are defined by {@link #startVersion} and
 * {@link #endVersion}.
 * <p>
 * A migration can handle more than 1 version (e.g. if you have a faster path to choose when
 * going version 3 to 5 without going to version 4). If Room opens a database at version
 * 3 and latest version is &gt;= 5, Room will use the migration object that can migrate from
 * 3 to 5 instead of 3 to 4 and 4 to 5.
 * <p>
 * If there are not enough migrations provided to move from the current version to the latest
 * version, Room will clear the database and recreate so even if you have no changes between 2
 * versions, you should still provide a Migration object to the builder.
 */
public abstract class Migration {
    public final int startVersion;
    public final int endVersion;

    /**
     * Creates a new migration between {@code startVersion} and {@code endVersion}.
     *
     * @param startVersion The start version of the database.
     * @param endVersion The end version of the database after this migration is applied.
     */
    public Migration(int startVersion, int endVersion) {
        this.startVersion = startVersion;
        this.endVersion = endVersion;
    }

    /**
     * Should run the necessary migrations.
     * <p>
     * This class cannot access any generated Dao in this method.
     * <p>
     * This method is already called inside a transaction and that transaction might actually be a
     * composite transaction of all necessary {@code Migration}s.
     *
     * @param database The database instance
     */
    public abstract void migrate(SupportSQLiteDatabase database);
}
