/*
 * Decompiled with CFR 0.152.
 */
package androidx.camera.core;

import android.graphics.Bitmap;
import android.media.Image;
import android.media.ImageWriter;
import android.os.Build;
import android.util.Log;
import android.view.Surface;
import androidx.annotation.IntRange;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.annotation.RestrictTo;
import androidx.camera.core.AndroidImageProxy;
import androidx.camera.core.ImageProxy;
import androidx.camera.core.Logger;
import androidx.camera.core.SingleCloseImageProxy;
import androidx.camera.core.impl.ImageReaderProxy;
import androidx.camera.core.internal.compat.ImageWriterCompat;
import androidx.camera.core.internal.utils.ImageUtil;
import androidx.core.util.Preconditions;
import java.nio.ByteBuffer;
import java.util.Locale;

@RestrictTo(value={RestrictTo.Scope.LIBRARY_GROUP})
public final class ImageProcessingUtil {
    private static final String TAG = "ImageProcessingUtil";
    public static final String JNI_LIB_NAME = "image_processing_util_jni";
    private static int sImageCount = 0;

    private ImageProcessingUtil() {
    }

    @Nullable
    public static ImageProxy convertJpegBytesToImage(@NonNull ImageReaderProxy jpegImageReaderProxy, @NonNull byte[] jpegBytes) {
        Preconditions.checkArgument((jpegImageReaderProxy.getImageFormat() == 256 ? 1 : 0) != 0);
        Preconditions.checkNotNull((Object)jpegBytes);
        Surface surface = jpegImageReaderProxy.getSurface();
        Preconditions.checkNotNull((Object)surface);
        if (ImageProcessingUtil.nativeWriteJpegToSurface(jpegBytes, surface) != 0) {
            Logger.e(TAG, "Failed to enqueue JPEG image.");
            return null;
        }
        ImageProxy imageProxy = jpegImageReaderProxy.acquireLatestImage();
        if (imageProxy == null) {
            Logger.e(TAG, "Failed to get acquire JPEG image.");
        }
        return imageProxy;
    }

    public static void copyBitmapToByteBuffer(@NonNull Bitmap bitmap, @NonNull ByteBuffer byteBuffer, int bufferStride) {
        int bitmapStride = bitmap.getRowBytes();
        int width = bitmap.getWidth();
        int height = bitmap.getHeight();
        ImageProcessingUtil.nativeCopyBetweenByteBufferAndBitmap(bitmap, byteBuffer, bitmapStride, bufferStride, width, height, false);
    }

    public static void copyByteBufferToBitmap(@NonNull Bitmap bitmap, @NonNull ByteBuffer byteBuffer, int bufferStride) {
        int bitmapStride = bitmap.getRowBytes();
        int width = bitmap.getWidth();
        int height = bitmap.getHeight();
        ImageProcessingUtil.nativeCopyBetweenByteBufferAndBitmap(bitmap, byteBuffer, bufferStride, bitmapStride, width, height, true);
    }

    public static boolean writeJpegBytesToSurface(@NonNull Surface surface, @NonNull byte[] jpegBytes) {
        Preconditions.checkNotNull((Object)jpegBytes);
        Preconditions.checkNotNull((Object)surface);
        if (ImageProcessingUtil.nativeWriteJpegToSurface(jpegBytes, surface) != 0) {
            Logger.e(TAG, "Failed to enqueue JPEG image.");
            return false;
        }
        return true;
    }

    public static boolean convertYuvToJpegBytesIntoSurface(@NonNull Image image, @IntRange(from=1L, to=100L) int jpegQuality, int rotationDegrees, @NonNull Surface outputSurface) {
        return ImageProcessingUtil.convertYuvToJpegBytesIntoSurface(new AndroidImageProxy(image), jpegQuality, rotationDegrees, outputSurface);
    }

    public static boolean convertYuvToJpegBytesIntoSurface(@NonNull ImageProxy imageProxy, @IntRange(from=1L, to=100L) int jpegQuality, int rotationDegrees, @NonNull Surface outputSurface) {
        try {
            byte[] jpegBytes = ImageUtil.yuvImageToJpegByteArray(imageProxy, null, jpegQuality, rotationDegrees);
            return ImageProcessingUtil.writeJpegBytesToSurface(outputSurface, jpegBytes);
        }
        catch (ImageUtil.CodecFailedException e) {
            Logger.e(TAG, "Failed to encode YUV to JPEG", e);
            return false;
        }
    }

    @Nullable
    public static ImageProxy convertYUVToRGB(@NonNull ImageProxy imageProxy, @NonNull ImageReaderProxy rgbImageReaderProxy, @Nullable ByteBuffer rgbConvertedBuffer, @IntRange(from=0L, to=359L) int rotationDegrees, boolean onePixelShiftEnabled) {
        ImageProxy rgbImageProxy;
        if (!ImageProcessingUtil.isSupportedYUVFormat(imageProxy)) {
            Logger.e(TAG, "Unsupported format for YUV to RGB");
            return null;
        }
        long startTimeMillis = System.currentTimeMillis();
        if (!ImageProcessingUtil.isSupportedRotationDegrees(rotationDegrees)) {
            Logger.e(TAG, "Unsupported rotation degrees for rotate RGB");
            return null;
        }
        Result result = ImageProcessingUtil.convertYUVToRGBInternal(imageProxy, rgbImageReaderProxy.getSurface(), rgbConvertedBuffer, rotationDegrees, onePixelShiftEnabled);
        if (result == Result.ERROR_CONVERSION) {
            Logger.e(TAG, "YUV to RGB conversion failure");
            return null;
        }
        if (Log.isLoggable((String)"MH", (int)3)) {
            Logger.d(TAG, String.format(Locale.US, "Image processing performance profiling, duration: [%d], image count: %d", System.currentTimeMillis() - startTimeMillis, sImageCount));
            ++sImageCount;
        }
        if ((rgbImageProxy = rgbImageReaderProxy.acquireLatestImage()) == null) {
            Logger.e(TAG, "YUV to RGB acquireLatestImage failure");
            return null;
        }
        SingleCloseImageProxy wrappedRgbImageProxy = new SingleCloseImageProxy(rgbImageProxy);
        wrappedRgbImageProxy.addOnImageCloseListener(image -> {
            if (rgbImageProxy != null && imageProxy != null) {
                imageProxy.close();
            }
        });
        return wrappedRgbImageProxy;
    }

    @NonNull
    public static Bitmap convertYUVToBitmap(@NonNull ImageProxy imageProxy) {
        if (imageProxy.getFormat() != 35) {
            throw new IllegalArgumentException("Input image format must be YUV_420_888");
        }
        int imageWidth = imageProxy.getWidth();
        int imageHeight = imageProxy.getHeight();
        int srcStrideY = imageProxy.getPlanes()[0].getRowStride();
        int srcStrideU = imageProxy.getPlanes()[1].getRowStride();
        int srcStrideV = imageProxy.getPlanes()[2].getRowStride();
        int srcPixelStrideY = imageProxy.getPlanes()[0].getPixelStride();
        int srcPixelStrideUV = imageProxy.getPlanes()[1].getPixelStride();
        Bitmap bitmap = Bitmap.createBitmap((int)imageProxy.getWidth(), (int)imageProxy.getHeight(), (Bitmap.Config)Bitmap.Config.ARGB_8888);
        int bitmapStride = bitmap.getRowBytes();
        int result = ImageProcessingUtil.nativeConvertAndroid420ToBitmap(imageProxy.getPlanes()[0].getBuffer(), srcStrideY, imageProxy.getPlanes()[1].getBuffer(), srcStrideU, imageProxy.getPlanes()[2].getBuffer(), srcStrideV, srcPixelStrideY, srcPixelStrideUV, bitmap, bitmapStride, imageWidth, imageHeight);
        if (result != 0) {
            throw new UnsupportedOperationException("YUV to RGB conversion failed");
        }
        return bitmap;
    }

    public static boolean applyPixelShiftForYUV(@NonNull ImageProxy imageProxy) {
        if (!ImageProcessingUtil.isSupportedYUVFormat(imageProxy)) {
            Logger.e(TAG, "Unsupported format for YUV to RGB");
            return false;
        }
        Result result = ImageProcessingUtil.applyPixelShiftInternal(imageProxy);
        if (result == Result.ERROR_CONVERSION) {
            Logger.e(TAG, "One pixel shift for YUV failure");
            return false;
        }
        return true;
    }

    @Nullable
    public static ImageProxy rotateYUV(@NonNull ImageProxy imageProxy, @NonNull ImageReaderProxy rotatedImageReaderProxy, @NonNull ImageWriter rotatedImageWriter, @NonNull ByteBuffer yRotatedBuffer, @NonNull ByteBuffer uRotatedBuffer, @NonNull ByteBuffer vRotatedBuffer, @IntRange(from=0L, to=359L) int rotationDegrees) {
        if (!ImageProcessingUtil.isSupportedYUVFormat(imageProxy)) {
            Logger.e(TAG, "Unsupported format for rotate YUV");
            return null;
        }
        if (!ImageProcessingUtil.isSupportedRotationDegrees(rotationDegrees)) {
            Logger.e(TAG, "Unsupported rotation degrees for rotate YUV");
            return null;
        }
        Result result = Result.ERROR_CONVERSION;
        if (Build.VERSION.SDK_INT >= 23 && rotationDegrees > 0) {
            result = ImageProcessingUtil.rotateYUVInternal(imageProxy, rotatedImageWriter, yRotatedBuffer, uRotatedBuffer, vRotatedBuffer, rotationDegrees);
        }
        if (result == Result.ERROR_CONVERSION) {
            Logger.e(TAG, "rotate YUV failure");
            return null;
        }
        ImageProxy rotatedImageProxy = rotatedImageReaderProxy.acquireLatestImage();
        if (rotatedImageProxy == null) {
            Logger.e(TAG, "YUV rotation acquireLatestImage failure");
            return null;
        }
        SingleCloseImageProxy wrappedRotatedImageProxy = new SingleCloseImageProxy(rotatedImageProxy);
        wrappedRotatedImageProxy.addOnImageCloseListener(image -> {
            if (rotatedImageProxy != null && imageProxy != null) {
                imageProxy.close();
            }
        });
        return wrappedRotatedImageProxy;
    }

    private static boolean isSupportedYUVFormat(@NonNull ImageProxy imageProxy) {
        return imageProxy.getFormat() == 35 && imageProxy.getPlanes().length == 3;
    }

    private static boolean isSupportedRotationDegrees(@IntRange(from=0L, to=359L) int rotationDegrees) {
        return rotationDegrees == 0 || rotationDegrees == 90 || rotationDegrees == 180 || rotationDegrees == 270;
    }

    @NonNull
    private static Result convertYUVToRGBInternal(@NonNull ImageProxy imageProxy, @NonNull Surface surface, @Nullable ByteBuffer rgbConvertedBuffer, int rotation, boolean onePixelShiftEnabled) {
        int imageWidth = imageProxy.getWidth();
        int imageHeight = imageProxy.getHeight();
        int srcStrideY = imageProxy.getPlanes()[0].getRowStride();
        int srcStrideU = imageProxy.getPlanes()[1].getRowStride();
        int srcStrideV = imageProxy.getPlanes()[2].getRowStride();
        int srcPixelStrideY = imageProxy.getPlanes()[0].getPixelStride();
        int srcPixelStrideUV = imageProxy.getPlanes()[1].getPixelStride();
        int startOffsetY = onePixelShiftEnabled ? srcPixelStrideY : 0;
        int startOffsetU = onePixelShiftEnabled ? srcPixelStrideUV : 0;
        int startOffsetV = onePixelShiftEnabled ? srcPixelStrideUV : 0;
        int result = ImageProcessingUtil.nativeConvertAndroid420ToABGR(imageProxy.getPlanes()[0].getBuffer(), srcStrideY, imageProxy.getPlanes()[1].getBuffer(), srcStrideU, imageProxy.getPlanes()[2].getBuffer(), srcStrideV, srcPixelStrideY, srcPixelStrideUV, surface, rgbConvertedBuffer, imageWidth, imageHeight, startOffsetY, startOffsetU, startOffsetV, rotation);
        if (result != 0) {
            return Result.ERROR_CONVERSION;
        }
        return Result.SUCCESS;
    }

    @NonNull
    private static Result applyPixelShiftInternal(@NonNull ImageProxy imageProxy) {
        int imageWidth = imageProxy.getWidth();
        int imageHeight = imageProxy.getHeight();
        int srcStrideY = imageProxy.getPlanes()[0].getRowStride();
        int srcStrideU = imageProxy.getPlanes()[1].getRowStride();
        int srcStrideV = imageProxy.getPlanes()[2].getRowStride();
        int srcPixelStrideY = imageProxy.getPlanes()[0].getPixelStride();
        int srcPixelStrideUV = imageProxy.getPlanes()[1].getPixelStride();
        int startOffsetY = srcPixelStrideY;
        int startOffsetU = srcPixelStrideUV;
        int startOffsetV = srcPixelStrideUV;
        int result = ImageProcessingUtil.nativeShiftPixel(imageProxy.getPlanes()[0].getBuffer(), srcStrideY, imageProxy.getPlanes()[1].getBuffer(), srcStrideU, imageProxy.getPlanes()[2].getBuffer(), srcStrideV, srcPixelStrideY, srcPixelStrideUV, imageWidth, imageHeight, startOffsetY, startOffsetU, startOffsetV);
        if (result != 0) {
            return Result.ERROR_CONVERSION;
        }
        return Result.SUCCESS;
    }

    @RequiresApi(value=23)
    @Nullable
    private static Result rotateYUVInternal(@NonNull ImageProxy imageProxy, @NonNull ImageWriter rotatedImageWriter, @NonNull ByteBuffer yRotatedBuffer, @NonNull ByteBuffer uRotatedBuffer, @NonNull ByteBuffer vRotatedBuffer, int rotationDegrees) {
        int imageWidth = imageProxy.getWidth();
        int imageHeight = imageProxy.getHeight();
        int srcStrideY = imageProxy.getPlanes()[0].getRowStride();
        int srcStrideU = imageProxy.getPlanes()[1].getRowStride();
        int srcStrideV = imageProxy.getPlanes()[2].getRowStride();
        int srcPixelStrideUV = imageProxy.getPlanes()[1].getPixelStride();
        Image rotatedImage = ImageWriterCompat.dequeueInputImage(rotatedImageWriter);
        if (rotatedImage == null) {
            return Result.ERROR_CONVERSION;
        }
        int result = ImageProcessingUtil.nativeRotateYUV(imageProxy.getPlanes()[0].getBuffer(), srcStrideY, imageProxy.getPlanes()[1].getBuffer(), srcStrideU, imageProxy.getPlanes()[2].getBuffer(), srcStrideV, srcPixelStrideUV, rotatedImage.getPlanes()[0].getBuffer(), rotatedImage.getPlanes()[0].getRowStride(), rotatedImage.getPlanes()[0].getPixelStride(), rotatedImage.getPlanes()[1].getBuffer(), rotatedImage.getPlanes()[1].getRowStride(), rotatedImage.getPlanes()[1].getPixelStride(), rotatedImage.getPlanes()[2].getBuffer(), rotatedImage.getPlanes()[2].getRowStride(), rotatedImage.getPlanes()[2].getPixelStride(), yRotatedBuffer, uRotatedBuffer, vRotatedBuffer, imageWidth, imageHeight, rotationDegrees);
        if (result != 0) {
            return Result.ERROR_CONVERSION;
        }
        ImageWriterCompat.queueInputImage(rotatedImageWriter, rotatedImage);
        return Result.SUCCESS;
    }

    private static native int nativeCopyBetweenByteBufferAndBitmap(Bitmap var0, ByteBuffer var1, int var2, int var3, int var4, int var5, boolean var6);

    private static native int nativeWriteJpegToSurface(@NonNull byte[] var0, @NonNull Surface var1);

    private static native int nativeConvertAndroid420ToABGR(@NonNull ByteBuffer var0, int var1, @NonNull ByteBuffer var2, int var3, @NonNull ByteBuffer var4, int var5, int var6, int var7, @Nullable Surface var8, @Nullable ByteBuffer var9, int var10, int var11, int var12, int var13, int var14, int var15);

    private static native int nativeConvertAndroid420ToBitmap(@NonNull ByteBuffer var0, int var1, @NonNull ByteBuffer var2, int var3, @NonNull ByteBuffer var4, int var5, int var6, int var7, @NonNull Bitmap var8, int var9, int var10, int var11);

    private static native int nativeShiftPixel(@NonNull ByteBuffer var0, int var1, @NonNull ByteBuffer var2, int var3, @NonNull ByteBuffer var4, int var5, int var6, int var7, int var8, int var9, int var10, int var11, int var12);

    private static native int nativeRotateYUV(@NonNull ByteBuffer var0, int var1, @NonNull ByteBuffer var2, int var3, @NonNull ByteBuffer var4, int var5, int var6, @NonNull ByteBuffer var7, int var8, int var9, @NonNull ByteBuffer var10, int var11, int var12, @NonNull ByteBuffer var13, int var14, int var15, @NonNull ByteBuffer var16, @NonNull ByteBuffer var17, @NonNull ByteBuffer var18, int var19, int var20, int var21);

    static {
        System.loadLibrary(JNI_LIB_NAME);
    }

    static enum Result {
        UNKNOWN,
        SUCCESS,
        ERROR_CONVERSION;

    }
}

