/*
 * Decompiled with CFR 0.152.
 */
package androidx.camera.core.processing;

import android.graphics.Bitmap;
import android.graphics.SurfaceTexture;
import android.os.Handler;
import android.os.HandlerThread;
import android.util.Size;
import android.view.Surface;
import androidx.annotation.IntRange;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;
import androidx.arch.core.util.Function;
import androidx.camera.core.DynamicRange;
import androidx.camera.core.ImageProcessingUtil;
import androidx.camera.core.Logger;
import androidx.camera.core.SurfaceOutput;
import androidx.camera.core.SurfaceRequest;
import androidx.camera.core.impl.utils.MatrixExt;
import androidx.camera.core.impl.utils.TransformUtils;
import androidx.camera.core.impl.utils.executor.CameraXExecutors;
import androidx.camera.core.impl.utils.futures.Futures;
import androidx.camera.core.processing.AutoValue_DefaultSurfaceProcessor_PendingSnapshot;
import androidx.camera.core.processing.OpenGlRenderer;
import androidx.camera.core.processing.ShaderProvider;
import androidx.camera.core.processing.SurfaceProcessorInternal;
import androidx.camera.core.processing.util.GLUtils;
import androidx.concurrent.futures.CallbackToFutureAdapter;
import androidx.core.util.Consumer;
import androidx.core.util.Preconditions;
import com.google.auto.value.AutoValue;
import com.google.common.util.concurrent.ListenableFuture;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.atomic.AtomicBoolean;
import kotlin.Triple;

public class DefaultSurfaceProcessor
implements SurfaceProcessorInternal,
SurfaceTexture.OnFrameAvailableListener {
    private static final String TAG = "DefaultSurfaceProcessor";
    private final OpenGlRenderer mGlRenderer;
    @VisibleForTesting
    final HandlerThread mGlThread;
    private final Executor mGlExecutor;
    @VisibleForTesting
    final Handler mGlHandler;
    private final AtomicBoolean mIsReleaseRequested = new AtomicBoolean(false);
    private final float[] mTextureMatrix = new float[16];
    private final float[] mSurfaceOutputMatrix = new float[16];
    final Map<SurfaceOutput, Surface> mOutputSurfaces = new LinkedHashMap<SurfaceOutput, Surface>();
    private int mInputSurfaceCount = 0;
    private boolean mIsReleased = false;
    private final List<PendingSnapshot> mPendingSnapshots = new ArrayList<PendingSnapshot>();

    DefaultSurfaceProcessor(@NonNull DynamicRange dynamicRange) {
        this(dynamicRange, Collections.emptyMap());
    }

    DefaultSurfaceProcessor(@NonNull DynamicRange dynamicRange, @NonNull Map<GLUtils.InputFormat, ShaderProvider> shaderProviderOverrides) {
        this.mGlThread = new HandlerThread("CameraX-GL Thread");
        this.mGlThread.start();
        this.mGlHandler = new Handler(this.mGlThread.getLooper());
        this.mGlExecutor = CameraXExecutors.newHandlerExecutor(this.mGlHandler);
        this.mGlRenderer = new OpenGlRenderer();
        try {
            this.initGlRenderer(dynamicRange, shaderProviderOverrides);
        }
        catch (RuntimeException e) {
            this.release();
            throw e;
        }
    }

    @Override
    public void onInputSurface(@NonNull SurfaceRequest surfaceRequest) {
        if (this.mIsReleaseRequested.get()) {
            surfaceRequest.willNotProvideSurface();
            return;
        }
        this.executeSafely(() -> {
            ++this.mInputSurfaceCount;
            SurfaceTexture surfaceTexture = new SurfaceTexture(this.mGlRenderer.getTextureName());
            surfaceTexture.setDefaultBufferSize(surfaceRequest.getResolution().getWidth(), surfaceRequest.getResolution().getHeight());
            Surface surface = new Surface(surfaceTexture);
            surfaceRequest.setTransformationInfoListener(this.mGlExecutor, transformationInfo -> {
                GLUtils.InputFormat inputFormat = GLUtils.InputFormat.DEFAULT;
                if (surfaceRequest.getDynamicRange().is10BitHdr() && transformationInfo.hasCameraTransform()) {
                    inputFormat = GLUtils.InputFormat.YUV;
                }
                this.mGlRenderer.setInputFormat(inputFormat);
            });
            surfaceRequest.provideSurface(surface, this.mGlExecutor, (Consumer<SurfaceRequest.Result>)((Consumer)result -> {
                surfaceRequest.clearTransformationInfoListener();
                surfaceTexture.setOnFrameAvailableListener(null);
                surfaceTexture.release();
                surface.release();
                --this.mInputSurfaceCount;
                this.checkReadyToRelease();
            }));
            surfaceTexture.setOnFrameAvailableListener((SurfaceTexture.OnFrameAvailableListener)this, this.mGlHandler);
        }, surfaceRequest::willNotProvideSurface);
    }

    @Override
    public void onOutputSurface(@NonNull SurfaceOutput surfaceOutput) {
        if (this.mIsReleaseRequested.get()) {
            surfaceOutput.close();
            return;
        }
        this.executeSafely(() -> {
            Surface surface = surfaceOutput.getSurface(this.mGlExecutor, (Consumer<SurfaceOutput.Event>)((Consumer)event -> {
                surfaceOutput.close();
                Surface removedSurface = this.mOutputSurfaces.remove(surfaceOutput);
                if (removedSurface != null) {
                    this.mGlRenderer.unregisterOutputSurface(removedSurface);
                }
            }));
            this.mGlRenderer.registerOutputSurface(surface);
            this.mOutputSurfaces.put(surfaceOutput, surface);
        }, surfaceOutput::close);
    }

    @Override
    public void release() {
        if (this.mIsReleaseRequested.getAndSet(true)) {
            return;
        }
        this.executeSafely(() -> {
            this.mIsReleased = true;
            this.checkReadyToRelease();
        });
    }

    @Override
    @NonNull
    public ListenableFuture<Void> snapshot(@IntRange(from=0L, to=100L) int jpegQuality, @IntRange(from=0L, to=359L) int rotationDegrees) {
        return Futures.nonCancellationPropagating(CallbackToFutureAdapter.getFuture(completer -> {
            AutoValue_DefaultSurfaceProcessor_PendingSnapshot pendingSnapshot = PendingSnapshot.of(jpegQuality, rotationDegrees, (CallbackToFutureAdapter.Completer<Void>)completer);
            this.executeSafely(() -> this.mPendingSnapshots.add(pendingSnapshot), () -> completer.setException((Throwable)new Exception("Failed to snapshot: OpenGLRenderer not ready.")));
            return "DefaultSurfaceProcessor#snapshot";
        }));
    }

    public void onFrameAvailable(@NonNull SurfaceTexture surfaceTexture) {
        if (this.mIsReleaseRequested.get()) {
            return;
        }
        surfaceTexture.updateTexImage();
        surfaceTexture.getTransformMatrix(this.mTextureMatrix);
        Triple jpegOutput = null;
        for (Map.Entry<SurfaceOutput, Surface> entry : this.mOutputSurfaces.entrySet()) {
            Surface surface = entry.getValue();
            SurfaceOutput surfaceOutput = entry.getKey();
            surfaceOutput.updateTransformMatrix(this.mSurfaceOutputMatrix, this.mTextureMatrix);
            if (surfaceOutput.getFormat() == 34) {
                try {
                    this.mGlRenderer.render(surfaceTexture.getTimestamp(), this.mSurfaceOutputMatrix, surface);
                }
                catch (RuntimeException e) {
                    Logger.e(TAG, "Failed to render with OpenGL.", e);
                }
                continue;
            }
            Preconditions.checkState((surfaceOutput.getFormat() == 256 ? 1 : 0) != 0, (String)("Unsupported format: " + surfaceOutput.getFormat()));
            Preconditions.checkState((jpegOutput == null ? 1 : 0) != 0, (String)"Only one JPEG output is supported.");
            jpegOutput = new Triple((Object)surface, (Object)surfaceOutput.getSize(), (Object)((float[])this.mSurfaceOutputMatrix.clone()));
        }
        try {
            this.takeSnapshotAndDrawJpeg(jpegOutput);
        }
        catch (RuntimeException e) {
            this.failAllPendingSnapshots(e);
        }
    }

    @WorkerThread
    private void takeSnapshotAndDrawJpeg(@Nullable Triple<Surface, Size, float[]> jpegOutput) {
        if (this.mPendingSnapshots.isEmpty()) {
            return;
        }
        if (jpegOutput == null) {
            this.failAllPendingSnapshots(new Exception("Failed to snapshot: no JPEG Surface."));
            return;
        }
        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream();){
            byte[] jpegBytes = null;
            int jpegQuality = -1;
            int rotationDegrees = -1;
            Bitmap bitmap = null;
            Iterator<PendingSnapshot> iterator = this.mPendingSnapshots.iterator();
            while (iterator.hasNext()) {
                PendingSnapshot pendingSnapshot = iterator.next();
                if (rotationDegrees != pendingSnapshot.getRotationDegrees() || bitmap == null) {
                    rotationDegrees = pendingSnapshot.getRotationDegrees();
                    if (bitmap != null) {
                        bitmap.recycle();
                    }
                    bitmap = this.getBitmap((Size)jpegOutput.getSecond(), (float[])jpegOutput.getThird(), rotationDegrees);
                    jpegQuality = -1;
                }
                if (jpegQuality != pendingSnapshot.getJpegQuality()) {
                    outputStream.reset();
                    jpegQuality = pendingSnapshot.getJpegQuality();
                    bitmap.compress(Bitmap.CompressFormat.JPEG, jpegQuality, (OutputStream)outputStream);
                    jpegBytes = outputStream.toByteArray();
                }
                ImageProcessingUtil.writeJpegBytesToSurface((Surface)jpegOutput.getFirst(), Objects.requireNonNull(jpegBytes));
                pendingSnapshot.getCompleter().set(null);
                iterator.remove();
            }
        }
        catch (IOException e) {
            this.failAllPendingSnapshots(e);
        }
    }

    private void failAllPendingSnapshots(@NonNull Throwable throwable) {
        for (PendingSnapshot pendingSnapshot : this.mPendingSnapshots) {
            pendingSnapshot.getCompleter().setException(throwable);
        }
        this.mPendingSnapshots.clear();
    }

    @NonNull
    private Bitmap getBitmap(@NonNull Size size, @NonNull float[] textureTransform, int rotationDegrees) {
        float[] snapshotTransform = (float[])textureTransform.clone();
        MatrixExt.preRotate(snapshotTransform, rotationDegrees, 0.5f, 0.5f);
        MatrixExt.preVerticalFlip(snapshotTransform, 0.5f);
        size = TransformUtils.rotateSize(size, rotationDegrees);
        return this.mGlRenderer.snapshot(size, snapshotTransform);
    }

    @WorkerThread
    private void checkReadyToRelease() {
        if (this.mIsReleased && this.mInputSurfaceCount == 0) {
            for (SurfaceOutput surfaceOutput : this.mOutputSurfaces.keySet()) {
                surfaceOutput.close();
            }
            for (PendingSnapshot pendingSnapshot : this.mPendingSnapshots) {
                pendingSnapshot.getCompleter().setException((Throwable)new Exception("Failed to snapshot: DefaultSurfaceProcessor is released."));
            }
            this.mOutputSurfaces.clear();
            this.mGlRenderer.release();
            this.mGlThread.quit();
        }
    }

    private void initGlRenderer(@NonNull DynamicRange dynamicRange, @NonNull Map<GLUtils.InputFormat, ShaderProvider> shaderProviderOverrides) {
        ListenableFuture initFuture = CallbackToFutureAdapter.getFuture(completer -> {
            this.executeSafely(() -> {
                try {
                    this.mGlRenderer.init(dynamicRange, shaderProviderOverrides);
                    completer.set(null);
                }
                catch (RuntimeException e) {
                    completer.setException((Throwable)e);
                }
            });
            return "Init GlRenderer";
        });
        try {
            initFuture.get();
        }
        catch (InterruptedException | ExecutionException e) {
            Throwable cause;
            Throwable throwable = cause = e instanceof ExecutionException ? e.getCause() : e;
            if (cause instanceof RuntimeException) {
                throw (RuntimeException)cause;
            }
            throw new IllegalStateException("Failed to create DefaultSurfaceProcessor", cause);
        }
    }

    private void executeSafely(@NonNull Runnable runnable) {
        this.executeSafely(runnable, () -> {});
    }

    private void executeSafely(@NonNull Runnable runnable, @NonNull Runnable onFailure) {
        try {
            this.mGlExecutor.execute(() -> {
                if (this.mIsReleased) {
                    onFailure.run();
                } else {
                    runnable.run();
                }
            });
        }
        catch (RejectedExecutionException e) {
            Logger.w(TAG, "Unable to executor runnable", e);
            onFailure.run();
        }
    }

    @AutoValue
    static abstract class PendingSnapshot {
        PendingSnapshot() {
        }

        @IntRange(from=0L, to=100L)
        abstract int getJpegQuality();

        @IntRange(from=0L, to=359L)
        abstract int getRotationDegrees();

        @NonNull
        abstract CallbackToFutureAdapter.Completer<Void> getCompleter();

        @NonNull
        static AutoValue_DefaultSurfaceProcessor_PendingSnapshot of(@IntRange(from=0L, to=100L) int jpegQuality, @IntRange(from=0L, to=359L) int rotationDegrees, @NonNull CallbackToFutureAdapter.Completer<Void> completer) {
            return new AutoValue_DefaultSurfaceProcessor_PendingSnapshot(jpegQuality, rotationDegrees, completer);
        }
    }

    public static class Factory {
        private static Function<DynamicRange, SurfaceProcessorInternal> sSupplier = DefaultSurfaceProcessor::new;

        private Factory() {
        }

        @NonNull
        public static SurfaceProcessorInternal newInstance(@NonNull DynamicRange dynamicRange) {
            return (SurfaceProcessorInternal)sSupplier.apply((Object)dynamicRange);
        }

        @VisibleForTesting
        public static void setSupplier(@NonNull Function<DynamicRange, SurfaceProcessorInternal> supplier) {
            sSupplier = supplier;
        }
    }
}

