/*
 * Decompiled with CFR 0.152.
 */
package androidx.camera.core.processing.concurrent;

import android.graphics.SurfaceTexture;
import android.os.Handler;
import android.os.HandlerThread;
import android.view.Surface;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;
import androidx.camera.core.CompositionSettings;
import androidx.camera.core.DynamicRange;
import androidx.camera.core.Logger;
import androidx.camera.core.ProcessingException;
import androidx.camera.core.SurfaceOutput;
import androidx.camera.core.SurfaceRequest;
import androidx.camera.core.impl.utils.executor.CameraXExecutors;
import androidx.camera.core.processing.ShaderProvider;
import androidx.camera.core.processing.SurfaceProcessorInternal;
import androidx.camera.core.processing.concurrent.DualOpenGlRenderer;
import androidx.camera.core.processing.util.GLUtils;
import androidx.concurrent.futures.CallbackToFutureAdapter;
import androidx.core.util.Consumer;
import com.google.common.util.concurrent.ListenableFuture;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.atomic.AtomicBoolean;
import kotlin.jvm.functions.Function3;

public class DualSurfaceProcessor
implements SurfaceProcessorInternal,
SurfaceTexture.OnFrameAvailableListener {
    private static final String TAG = "DualSurfaceProcessor";
    private final DualOpenGlRenderer mGlRenderer;
    @VisibleForTesting
    final HandlerThread mGlThread;
    private final Executor mGlExecutor;
    @VisibleForTesting
    final Handler mGlHandler;
    private int mInputSurfaceCount = 0;
    private boolean mIsReleased = false;
    private final AtomicBoolean mIsReleaseRequested = new AtomicBoolean(false);
    final Map<SurfaceOutput, Surface> mOutputSurfaces = new LinkedHashMap<SurfaceOutput, Surface>();
    @Nullable
    private SurfaceTexture mPrimarySurfaceTexture;
    @Nullable
    private SurfaceTexture mSecondarySurfaceTexture;

    DualSurfaceProcessor(@NonNull DynamicRange dynamicRange, @NonNull CompositionSettings primaryCompositionSettings, @NonNull CompositionSettings secondaryCompositionSettings) {
        this(dynamicRange, Collections.emptyMap(), primaryCompositionSettings, secondaryCompositionSettings);
    }

    DualSurfaceProcessor(@NonNull DynamicRange dynamicRange, @NonNull Map<GLUtils.InputFormat, ShaderProvider> shaderProviderOverrides, @NonNull CompositionSettings primaryCompositionSettings, @NonNull CompositionSettings secondaryCompositionSettings) {
        this.mGlThread = new HandlerThread("CameraX-GL Thread");
        this.mGlThread.start();
        this.mGlHandler = new Handler(this.mGlThread.getLooper());
        this.mGlExecutor = CameraXExecutors.newHandlerExecutor(this.mGlHandler);
        this.mGlRenderer = new DualOpenGlRenderer(primaryCompositionSettings, secondaryCompositionSettings);
        try {
            this.initGlRenderer(dynamicRange, shaderProviderOverrides);
        }
        catch (RuntimeException e) {
            this.release();
            throw e;
        }
    }

    @Override
    public void onInputSurface(@NonNull SurfaceRequest surfaceRequest) throws ProcessingException {
        if (this.mIsReleaseRequested.get()) {
            surfaceRequest.willNotProvideSurface();
            return;
        }
        this.executeSafely(() -> {
            ++this.mInputSurfaceCount;
            SurfaceTexture surfaceTexture = new SurfaceTexture(this.mGlRenderer.getTextureName(surfaceRequest.isPrimary()));
            surfaceTexture.setDefaultBufferSize(surfaceRequest.getResolution().getWidth(), surfaceRequest.getResolution().getHeight());
            Surface surface = new Surface(surfaceTexture);
            surfaceRequest.provideSurface(surface, this.mGlExecutor, (Consumer<SurfaceRequest.Result>)((Consumer)result -> {
                surfaceTexture.setOnFrameAvailableListener(null);
                surfaceTexture.release();
                surface.release();
                --this.mInputSurfaceCount;
                this.checkReadyToRelease();
            }));
            if (surfaceRequest.isPrimary()) {
                this.mPrimarySurfaceTexture = surfaceTexture;
            } else {
                this.mSecondarySurfaceTexture = surfaceTexture;
                surfaceTexture.setOnFrameAvailableListener((SurfaceTexture.OnFrameAvailableListener)this, this.mGlHandler);
            }
        }, surfaceRequest::willNotProvideSurface);
    }

    @Override
    public void onOutputSurface(@NonNull SurfaceOutput surfaceOutput) throws ProcessingException {
        if (this.mIsReleaseRequested.get()) {
            surfaceOutput.close();
            return;
        }
        this.executeSafely(() -> {
            Surface surface = surfaceOutput.getSurface(this.mGlExecutor, (Consumer<SurfaceOutput.Event>)((Consumer)event -> {
                surfaceOutput.close();
                Surface removedSurface = this.mOutputSurfaces.remove(surfaceOutput);
                if (removedSurface != null) {
                    this.mGlRenderer.unregisterOutputSurface(removedSurface);
                }
            }));
            this.mGlRenderer.registerOutputSurface(surface);
            this.mOutputSurfaces.put(surfaceOutput, surface);
        }, surfaceOutput::close);
    }

    @Override
    public void release() {
        if (this.mIsReleaseRequested.getAndSet(true)) {
            return;
        }
        this.executeSafely(() -> {
            this.mIsReleased = true;
            this.checkReadyToRelease();
        });
    }

    public void onFrameAvailable(SurfaceTexture surfaceTexture) {
        if (this.mIsReleaseRequested.get()) {
            return;
        }
        if (this.mPrimarySurfaceTexture == null || this.mSecondarySurfaceTexture == null) {
            return;
        }
        this.mPrimarySurfaceTexture.updateTexImage();
        this.mSecondarySurfaceTexture.updateTexImage();
        for (Map.Entry<SurfaceOutput, Surface> entry : this.mOutputSurfaces.entrySet()) {
            Surface surface = entry.getValue();
            SurfaceOutput surfaceOutput = entry.getKey();
            if (surfaceOutput.getFormat() != 34) continue;
            try {
                this.mGlRenderer.render(surfaceTexture.getTimestamp(), surface, surfaceOutput, this.mPrimarySurfaceTexture, this.mSecondarySurfaceTexture);
            }
            catch (RuntimeException e) {
                Logger.e(TAG, "Failed to render with OpenGL.", e);
            }
        }
    }

    private void initGlRenderer(@NonNull DynamicRange dynamicRange, @NonNull Map<GLUtils.InputFormat, ShaderProvider> shaderProviderOverrides) {
        ListenableFuture initFuture = CallbackToFutureAdapter.getFuture(completer -> {
            this.executeSafely(() -> {
                try {
                    this.mGlRenderer.init(dynamicRange, shaderProviderOverrides);
                    completer.set(null);
                }
                catch (RuntimeException e) {
                    completer.setException((Throwable)e);
                }
            });
            return "Init GlRenderer";
        });
        try {
            initFuture.get();
        }
        catch (InterruptedException | ExecutionException e) {
            Throwable cause;
            Throwable throwable = cause = e instanceof ExecutionException ? e.getCause() : e;
            if (cause instanceof RuntimeException) {
                throw (RuntimeException)cause;
            }
            throw new IllegalStateException("Failed to create DefaultSurfaceProcessor", cause);
        }
    }

    private void executeSafely(@NonNull Runnable runnable) {
        this.executeSafely(runnable, () -> {});
    }

    private void executeSafely(@NonNull Runnable runnable, @NonNull Runnable onFailure) {
        try {
            this.mGlExecutor.execute(() -> {
                if (this.mIsReleased) {
                    onFailure.run();
                } else {
                    runnable.run();
                }
            });
        }
        catch (RejectedExecutionException e) {
            Logger.w(TAG, "Unable to executor runnable", e);
            onFailure.run();
        }
    }

    @WorkerThread
    private void checkReadyToRelease() {
        if (this.mIsReleased && this.mInputSurfaceCount == 0) {
            for (SurfaceOutput surfaceOutput : this.mOutputSurfaces.keySet()) {
                surfaceOutput.close();
            }
            this.mOutputSurfaces.clear();
            this.mGlRenderer.release();
            this.mGlThread.quit();
        }
    }

    public static class Factory {
        private static Function3<DynamicRange, CompositionSettings, CompositionSettings, SurfaceProcessorInternal> sSupplier = DualSurfaceProcessor::new;

        private Factory() {
        }

        @NonNull
        public static SurfaceProcessorInternal newInstance(@NonNull DynamicRange dynamicRange, @NonNull CompositionSettings primaryCompositionSettings, @NonNull CompositionSettings secondaryCompositionSettings) {
            return (SurfaceProcessorInternal)sSupplier.invoke((Object)dynamicRange, (Object)primaryCompositionSettings, (Object)secondaryCompositionSettings);
        }

        @VisibleForTesting
        public static void setSupplier(@NonNull Function3<DynamicRange, CompositionSettings, CompositionSettings, SurfaceProcessorInternal> supplier) {
            sSupplier = supplier;
        }
    }
}

