/*
 * Decompiled with CFR 0.152.
 */
package androidx.camera.core.processing.util;

import android.opengl.EGL14;
import android.opengl.EGLConfig;
import android.opengl.EGLDisplay;
import android.opengl.EGLSurface;
import android.opengl.GLES20;
import android.opengl.Matrix;
import android.util.Log;
import android.util.Size;
import android.view.Surface;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RestrictTo;
import androidx.camera.core.DynamicRange;
import androidx.camera.core.Logger;
import androidx.camera.core.processing.ShaderProvider;
import androidx.camera.core.processing.util.OutputSurface;
import androidx.core.util.Preconditions;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

@RestrictTo(value={RestrictTo.Scope.LIBRARY_GROUP})
public final class GLUtils {
    public static final String VERSION_UNKNOWN = "0.0";
    public static final String TAG = "GLUtils";
    public static final int EGL_GL_COLORSPACE_KHR = 12445;
    public static final int EGL_GL_COLORSPACE_BT2020_HLG_EXT = 13632;
    public static final String VAR_TEXTURE_COORD = "vTextureCoord";
    public static final String VAR_TEXTURE = "sTexture";
    public static final int PIXEL_STRIDE = 4;
    public static final int[] EMPTY_ATTRIBS = new int[]{12344};
    public static final int[] HLG_SURFACE_ATTRIBS = new int[]{12445, 13632, 12344};
    public static final String DEFAULT_VERTEX_SHADER = String.format(Locale.US, "uniform mat4 uTexMatrix;\nuniform mat4 uTransMatrix;\nattribute vec4 aPosition;\nattribute vec4 aTextureCoord;\nvarying vec2 %s;\nvoid main() {\n    gl_Position = uTransMatrix * aPosition;\n    %s = (uTexMatrix * aTextureCoord).xy;\n}\n", "vTextureCoord", "vTextureCoord");
    public static final String HDR_VERTEX_SHADER = String.format(Locale.US, "#version 300 es\nin vec4 aPosition;\nin vec4 aTextureCoord;\nuniform mat4 uTexMatrix;\nuniform mat4 uTransMatrix;\nout vec2 %s;\nvoid main() {\n  gl_Position = uTransMatrix * aPosition;\n  %s = (uTexMatrix * aTextureCoord).xy;\n}\n", "vTextureCoord", "vTextureCoord");
    public static final String BLANK_VERTEX_SHADER = "uniform mat4 uTransMatrix;\nattribute vec4 aPosition;\nvoid main() {\n    gl_Position = uTransMatrix * aPosition;\n}\n";
    public static final String BLANK_FRAGMENT_SHADER = "precision mediump float;\nuniform float uAlphaScale;\nvoid main() {\n    gl_FragColor = vec4(0.0, 0.0, 0.0, uAlphaScale);\n}\n";
    private static final ShaderProvider SHADER_PROVIDER_DEFAULT = new ShaderProvider(){

        @Override
        @NonNull
        public String createFragmentShader(@NonNull String samplerVarName, @NonNull String fragCoordsVarName) {
            return String.format(Locale.US, "#extension GL_OES_EGL_image_external : require\nprecision mediump float;\nvarying vec2 %s;\nuniform samplerExternalOES %s;\nuniform float uAlphaScale;\nvoid main() {\n    vec4 src = texture2D(%s, %s);\n    gl_FragColor = vec4(src.rgb, src.a * uAlphaScale);\n}\n", fragCoordsVarName, samplerVarName, samplerVarName, fragCoordsVarName);
        }
    };
    private static final ShaderProvider SHADER_PROVIDER_HDR_DEFAULT = new ShaderProvider(){

        @Override
        @NonNull
        public String createFragmentShader(@NonNull String samplerVarName, @NonNull String fragCoordsVarName) {
            return String.format(Locale.US, "#version 300 es\n#extension GL_OES_EGL_image_external_essl3 : require\nprecision mediump float;\nuniform samplerExternalOES %s;\nuniform float uAlphaScale;\nin vec2 %s;\nout vec4 outColor;\n\nvoid main() {\n  vec4 src = texture(%s, %s);\n  outColor = vec4(src.rgb, src.a * uAlphaScale);\n}", samplerVarName, fragCoordsVarName, samplerVarName, fragCoordsVarName);
        }
    };
    private static final ShaderProvider SHADER_PROVIDER_HDR_YUV = new ShaderProvider(){

        @Override
        @NonNull
        public String createFragmentShader(@NonNull String samplerVarName, @NonNull String fragCoordsVarName) {
            return String.format(Locale.US, "#version 300 es\n#extension GL_EXT_YUV_target : require\nprecision mediump float;\nuniform __samplerExternal2DY2YEXT %s;\nuniform float uAlphaScale;\nin vec2 %s;\nout vec4 outColor;\n\nvec3 yuvToRgb(vec3 yuv) {\n  const vec3 yuvOffset = vec3(0.0625, 0.5, 0.5);\n  const mat3 yuvToRgbColorMat = mat3(\n    1.1689f, 1.1689f, 1.1689f,\n    0.0000f, -0.1881f, 2.1502f,\n    1.6853f, -0.6530f, 0.0000f\n  );\n  return clamp(yuvToRgbColorMat * (yuv - yuvOffset), 0.0, 1.0);\n}\n\nvoid main() {\n  vec3 srcYuv = texture(%s, %s).xyz;\n  vec3 srcRgb = yuvToRgb(srcYuv);\n  outColor = vec4(srcRgb, uAlphaScale);\n}", samplerVarName, fragCoordsVarName, samplerVarName, fragCoordsVarName);
        }
    };
    public static final float[] VERTEX_COORDS = new float[]{-1.0f, -1.0f, 1.0f, -1.0f, -1.0f, 1.0f, 1.0f, 1.0f};
    public static final FloatBuffer VERTEX_BUF = GLUtils.createFloatBuffer(VERTEX_COORDS);
    public static final float[] TEX_COORDS = new float[]{0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 1.0f, 1.0f, 1.0f};
    public static final FloatBuffer TEX_BUF = GLUtils.createFloatBuffer(TEX_COORDS);
    public static final int SIZEOF_FLOAT = 4;
    public static final OutputSurface NO_OUTPUT_SURFACE = OutputSurface.of(EGL14.EGL_NO_SURFACE, 0, 0);

    private GLUtils() {
    }

    @NonNull
    public static EGLSurface createWindowSurface(@NonNull EGLDisplay eglDisplay, @NonNull EGLConfig eglConfig, @NonNull Surface surface, @NonNull int[] surfaceAttrib) {
        EGLSurface eglSurface = EGL14.eglCreateWindowSurface((EGLDisplay)eglDisplay, (EGLConfig)eglConfig, (Object)surface, (int[])surfaceAttrib, (int)0);
        GLUtils.checkEglErrorOrThrow("eglCreateWindowSurface");
        if (eglSurface == null) {
            throw new IllegalStateException("surface was null");
        }
        return eglSurface;
    }

    public static int loadShader(int shaderType, @NonNull String source) {
        int shader = GLES20.glCreateShader((int)shaderType);
        GLUtils.checkGlErrorOrThrow("glCreateShader type=" + shaderType);
        GLES20.glShaderSource((int)shader, (String)source);
        GLES20.glCompileShader((int)shader);
        int[] compiled = new int[1];
        GLES20.glGetShaderiv((int)shader, (int)35713, (int[])compiled, (int)0);
        if (compiled[0] == 0) {
            Logger.w(TAG, "Could not compile shader: " + source);
            String shaderLog = GLES20.glGetShaderInfoLog((int)shader);
            GLES20.glDeleteShader((int)shader);
            throw new IllegalStateException("Could not compile shader type " + shaderType + ":" + shaderLog);
        }
        return shader;
    }

    public static int querySurface(@NonNull EGLDisplay eglDisplay, @NonNull EGLSurface eglSurface, int what) {
        int[] value = new int[1];
        EGL14.eglQuerySurface((EGLDisplay)eglDisplay, (EGLSurface)eglSurface, (int)what, (int[])value, (int)0);
        return value[0];
    }

    @NonNull
    public static Size getSurfaceSize(@NonNull EGLDisplay eglDisplay, @NonNull EGLSurface eglSurface) {
        int width = GLUtils.querySurface(eglDisplay, eglSurface, 12375);
        int height = GLUtils.querySurface(eglDisplay, eglSurface, 12374);
        return new Size(width, height);
    }

    @NonNull
    public static FloatBuffer createFloatBuffer(@NonNull float[] coords) {
        ByteBuffer bb = ByteBuffer.allocateDirect(coords.length * 4);
        bb.order(ByteOrder.nativeOrder());
        FloatBuffer fb = bb.asFloatBuffer();
        fb.put(coords);
        fb.position(0);
        return fb;
    }

    @NonNull
    public static EGLSurface createPBufferSurface(@NonNull EGLDisplay eglDisplay, @NonNull EGLConfig eglConfig, int width, int height) {
        int[] surfaceAttrib = new int[]{12375, width, 12374, height, 12344};
        EGLSurface eglSurface = EGL14.eglCreatePbufferSurface((EGLDisplay)eglDisplay, (EGLConfig)eglConfig, (int[])surfaceAttrib, (int)0);
        GLUtils.checkEglErrorOrThrow("eglCreatePbufferSurface");
        if (eglSurface == null) {
            throw new IllegalStateException("surface was null");
        }
        return eglSurface;
    }

    @NonNull
    public static Map<InputFormat, Program2D> createPrograms(@NonNull DynamicRange dynamicRange, @NonNull Map<InputFormat, ShaderProvider> shaderProviderOverrides) {
        HashMap<InputFormat, Program2D> programs = new HashMap<InputFormat, Program2D>();
        for (InputFormat inputFormat : InputFormat.values()) {
            Program2D program;
            ShaderProvider shaderProviderOverride = shaderProviderOverrides.get((Object)inputFormat);
            if (shaderProviderOverride != null) {
                program = new SamplerShaderProgram(dynamicRange, shaderProviderOverride);
            } else if (inputFormat == InputFormat.YUV || inputFormat == InputFormat.DEFAULT) {
                program = new SamplerShaderProgram(dynamicRange, inputFormat);
            } else {
                ShaderProvider defaultShaderProviderOverride;
                Preconditions.checkState((inputFormat == InputFormat.UNKNOWN ? 1 : 0) != 0, (String)("Unhandled input format: " + (Object)((Object)inputFormat)));
                program = dynamicRange.is10BitHdr() ? new BlankShaderProgram() : ((defaultShaderProviderOverride = shaderProviderOverrides.get((Object)InputFormat.DEFAULT)) != null ? new SamplerShaderProgram(dynamicRange, defaultShaderProviderOverride) : new SamplerShaderProgram(dynamicRange, InputFormat.DEFAULT));
            }
            Log.d((String)TAG, (String)("Shader program for input format " + (Object)((Object)inputFormat) + " created: " + program));
            programs.put(inputFormat, program);
        }
        return programs;
    }

    public static int createTexture() {
        int[] textures = new int[1];
        GLES20.glGenTextures((int)1, (int[])textures, (int)0);
        GLUtils.checkGlErrorOrThrow("glGenTextures");
        int texId = textures[0];
        GLES20.glBindTexture((int)36197, (int)texId);
        GLUtils.checkGlErrorOrThrow("glBindTexture " + texId);
        GLES20.glTexParameteri((int)36197, (int)10241, (int)9728);
        GLES20.glTexParameteri((int)36197, (int)10240, (int)9729);
        GLES20.glTexParameteri((int)36197, (int)10242, (int)33071);
        GLES20.glTexParameteri((int)36197, (int)10243, (int)33071);
        GLUtils.checkGlErrorOrThrow("glTexParameter");
        return texId;
    }

    @NonNull
    public static float[] create4x4IdentityMatrix() {
        float[] matrix = new float[16];
        Matrix.setIdentityM((float[])matrix, (int)0);
        return matrix;
    }

    public static void checkLocationOrThrow(int location, @NonNull String label) {
        if (location < 0) {
            throw new IllegalStateException("Unable to locate '" + label + "' in program");
        }
    }

    public static void checkEglErrorOrThrow(@NonNull String op) {
        int error = EGL14.eglGetError();
        if (error != 12288) {
            throw new IllegalStateException(op + ": EGL error: 0x" + Integer.toHexString(error));
        }
    }

    public static void checkGlErrorOrThrow(@NonNull String op) {
        int error = GLES20.glGetError();
        if (error != 0) {
            throw new IllegalStateException(op + ": GL error 0x" + Integer.toHexString(error));
        }
    }

    public static void checkEglErrorOrLog(@NonNull String op) {
        try {
            GLUtils.checkEglErrorOrThrow(op);
        }
        catch (IllegalStateException e) {
            Logger.e(TAG, e.toString(), e);
        }
    }

    public static void checkInitializedOrThrow(@NonNull AtomicBoolean initialized, boolean shouldInitialized) {
        boolean result = shouldInitialized == initialized.get();
        String message = shouldInitialized ? "OpenGlRenderer is not initialized" : "OpenGlRenderer is already initialized";
        Preconditions.checkState((boolean)result, (String)message);
    }

    public static void checkGlThreadOrThrow(@Nullable Thread thread) {
        Preconditions.checkState((thread == Thread.currentThread() ? 1 : 0) != 0, (String)"Method call must be called on the GL thread.");
    }

    @NonNull
    public static String getGlVersionNumber() {
        String glVersion = GLES20.glGetString((int)7938);
        Pattern pattern = Pattern.compile("OpenGL ES ([0-9]+)\\.([0-9]+).*");
        Matcher matcher = pattern.matcher(glVersion);
        if (matcher.find()) {
            String major = (String)Preconditions.checkNotNull((Object)matcher.group(1));
            String minor = (String)Preconditions.checkNotNull((Object)matcher.group(2));
            return major + "." + minor;
        }
        return VERSION_UNKNOWN;
    }

    @NonNull
    public static int[] chooseSurfaceAttrib(@NonNull String eglExtensions, @NonNull DynamicRange dynamicRange) {
        int[] attribs = EMPTY_ATTRIBS;
        if (dynamicRange.getEncoding() == 3) {
            if (eglExtensions.contains("EGL_EXT_gl_colorspace_bt2020_hlg")) {
                attribs = HLG_SURFACE_ATTRIBS;
            } else {
                Logger.w(TAG, "Dynamic range uses HLG encoding, but device does not support EGL_EXT_gl_colorspace_bt2020_hlg.Fallback to default colorspace.");
            }
        }
        return attribs;
    }

    public static int generateFbo() {
        int[] fbos = new int[1];
        GLES20.glGenFramebuffers((int)1, (int[])fbos, (int)0);
        GLUtils.checkGlErrorOrThrow("glGenFramebuffers");
        return fbos[0];
    }

    public static int generateTexture() {
        int[] textures = new int[1];
        GLES20.glGenTextures((int)1, (int[])textures, (int)0);
        GLUtils.checkGlErrorOrThrow("glGenTextures");
        return textures[0];
    }

    public static void deleteTexture(int texture) {
        int[] textures = new int[]{texture};
        GLES20.glDeleteTextures((int)1, (int[])textures, (int)0);
        GLUtils.checkGlErrorOrThrow("glDeleteTextures");
    }

    public static void deleteFbo(int fbo) {
        int[] fbos = new int[]{fbo};
        GLES20.glDeleteFramebuffers((int)1, (int[])fbos, (int)0);
        GLUtils.checkGlErrorOrThrow("glDeleteFramebuffers");
    }

    private static String getFragmentShaderSource(@NonNull ShaderProvider shaderProvider) {
        try {
            String source = shaderProvider.createFragmentShader(VAR_TEXTURE, VAR_TEXTURE_COORD);
            if (source == null || !source.contains(VAR_TEXTURE_COORD) || !source.contains(VAR_TEXTURE)) {
                throw new IllegalArgumentException("Invalid fragment shader");
            }
            return source;
        }
        catch (Throwable t) {
            if (t instanceof IllegalArgumentException) {
                throw t;
            }
            throw new IllegalArgumentException("Unable retrieve fragment shader source", t);
        }
    }

    public static enum InputFormat {
        UNKNOWN,
        DEFAULT,
        YUV;

    }

    public static class SamplerShaderProgram
    extends Program2D {
        private int mSamplerLoc = -1;
        private int mTexMatrixLoc = -1;
        private int mTexCoordLoc = -1;

        public SamplerShaderProgram(@NonNull DynamicRange dynamicRange, @NonNull InputFormat inputFormat) {
            this(dynamicRange, SamplerShaderProgram.resolveDefaultShaderProvider(dynamicRange, inputFormat));
        }

        public SamplerShaderProgram(@NonNull DynamicRange dynamicRange, @NonNull ShaderProvider shaderProvider) {
            super(dynamicRange.is10BitHdr() ? HDR_VERTEX_SHADER : DEFAULT_VERTEX_SHADER, GLUtils.getFragmentShaderSource(shaderProvider));
            this.loadLocations();
        }

        @Override
        public void use() {
            super.use();
            GLES20.glUniform1i((int)this.mSamplerLoc, (int)0);
            GLES20.glEnableVertexAttribArray((int)this.mTexCoordLoc);
            GLUtils.checkGlErrorOrThrow("glEnableVertexAttribArray");
            int coordsPerTex = 2;
            int texStride = 0;
            GLES20.glVertexAttribPointer((int)this.mTexCoordLoc, (int)coordsPerTex, (int)5126, (boolean)false, (int)texStride, (Buffer)TEX_BUF);
            GLUtils.checkGlErrorOrThrow("glVertexAttribPointer");
        }

        public void updateTextureMatrix(@NonNull float[] textureMat) {
            GLES20.glUniformMatrix4fv((int)this.mTexMatrixLoc, (int)1, (boolean)false, (float[])textureMat, (int)0);
            GLUtils.checkGlErrorOrThrow("glUniformMatrix4fv");
        }

        private void loadLocations() {
            ((Program2D)this).loadLocations();
            this.mSamplerLoc = GLES20.glGetUniformLocation((int)this.mProgramHandle, (String)GLUtils.VAR_TEXTURE);
            GLUtils.checkLocationOrThrow(this.mSamplerLoc, GLUtils.VAR_TEXTURE);
            this.mTexCoordLoc = GLES20.glGetAttribLocation((int)this.mProgramHandle, (String)"aTextureCoord");
            GLUtils.checkLocationOrThrow(this.mTexCoordLoc, "aTextureCoord");
            this.mTexMatrixLoc = GLES20.glGetUniformLocation((int)this.mProgramHandle, (String)"uTexMatrix");
            GLUtils.checkLocationOrThrow(this.mTexMatrixLoc, "uTexMatrix");
        }

        private static ShaderProvider resolveDefaultShaderProvider(@NonNull DynamicRange dynamicRange, @Nullable InputFormat inputFormat) {
            if (dynamicRange.is10BitHdr()) {
                Preconditions.checkArgument((inputFormat != InputFormat.UNKNOWN ? 1 : 0) != 0, (Object)("No default sampler shader available for" + (Object)((Object)inputFormat)));
                if (inputFormat == InputFormat.YUV) {
                    return SHADER_PROVIDER_HDR_YUV;
                }
                return SHADER_PROVIDER_HDR_DEFAULT;
            }
            return SHADER_PROVIDER_DEFAULT;
        }
    }

    public static class BlankShaderProgram
    extends Program2D {
        public BlankShaderProgram() {
            super(GLUtils.BLANK_VERTEX_SHADER, GLUtils.BLANK_FRAGMENT_SHADER);
        }
    }

    public static abstract class Program2D {
        protected int mProgramHandle;
        protected int mTransMatrixLoc = -1;
        protected int mAlphaScaleLoc = -1;
        protected int mPositionLoc = -1;

        protected Program2D(@NonNull String vertexShaderSource, @NonNull String fragmentShaderSource) {
            int vertexShader = -1;
            int fragmentShader = -1;
            int program = -1;
            try {
                vertexShader = GLUtils.loadShader(35633, vertexShaderSource);
                fragmentShader = GLUtils.loadShader(35632, fragmentShaderSource);
                program = GLES20.glCreateProgram();
                GLUtils.checkGlErrorOrThrow("glCreateProgram");
                GLES20.glAttachShader((int)program, (int)vertexShader);
                GLUtils.checkGlErrorOrThrow("glAttachShader");
                GLES20.glAttachShader((int)program, (int)fragmentShader);
                GLUtils.checkGlErrorOrThrow("glAttachShader");
                GLES20.glLinkProgram((int)program);
                int[] linkStatus = new int[1];
                GLES20.glGetProgramiv((int)program, (int)35714, (int[])linkStatus, (int)0);
                if (linkStatus[0] != 1) {
                    throw new IllegalStateException("Could not link program: " + GLES20.glGetProgramInfoLog((int)program));
                }
                this.mProgramHandle = program;
            }
            catch (IllegalArgumentException | IllegalStateException e) {
                if (vertexShader != -1) {
                    GLES20.glDeleteShader((int)vertexShader);
                }
                if (fragmentShader != -1) {
                    GLES20.glDeleteShader((int)fragmentShader);
                }
                if (program != -1) {
                    GLES20.glDeleteProgram((int)program);
                }
                throw e;
            }
            this.loadLocations();
        }

        public void use() {
            GLES20.glUseProgram((int)this.mProgramHandle);
            GLUtils.checkGlErrorOrThrow("glUseProgram");
            GLES20.glEnableVertexAttribArray((int)this.mPositionLoc);
            GLUtils.checkGlErrorOrThrow("glEnableVertexAttribArray");
            int coordsPerVertex = 2;
            int vertexStride = 0;
            GLES20.glVertexAttribPointer((int)this.mPositionLoc, (int)coordsPerVertex, (int)5126, (boolean)false, (int)vertexStride, (Buffer)VERTEX_BUF);
            GLUtils.checkGlErrorOrThrow("glVertexAttribPointer");
            this.updateTransformMatrix(GLUtils.create4x4IdentityMatrix());
            this.updateAlpha(1.0f);
        }

        public void updateTransformMatrix(@NonNull float[] transformMat) {
            GLES20.glUniformMatrix4fv((int)this.mTransMatrixLoc, (int)1, (boolean)false, (float[])transformMat, (int)0);
            GLUtils.checkGlErrorOrThrow("glUniformMatrix4fv");
        }

        public void updateAlpha(float alpha) {
            GLES20.glUniform1f((int)this.mAlphaScaleLoc, (float)alpha);
            GLUtils.checkGlErrorOrThrow("glUniform1f");
        }

        public void delete() {
            GLES20.glDeleteProgram((int)this.mProgramHandle);
        }

        private void loadLocations() {
            this.mPositionLoc = GLES20.glGetAttribLocation((int)this.mProgramHandle, (String)"aPosition");
            GLUtils.checkLocationOrThrow(this.mPositionLoc, "aPosition");
            this.mTransMatrixLoc = GLES20.glGetUniformLocation((int)this.mProgramHandle, (String)"uTransMatrix");
            GLUtils.checkLocationOrThrow(this.mTransMatrixLoc, "uTransMatrix");
            this.mAlphaScaleLoc = GLES20.glGetUniformLocation((int)this.mProgramHandle, (String)"uAlphaScale");
            GLUtils.checkLocationOrThrow(this.mAlphaScaleLoc, "uAlphaScale");
        }
    }
}

