/*
 * Decompiled with CFR 0.152.
 */
package androidx.camera.video.internal.audio;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.camera.core.Logger;
import androidx.camera.video.internal.audio.AudioSettings;
import androidx.camera.video.internal.audio.AudioStream;
import androidx.camera.video.internal.audio.AudioUtils;
import androidx.core.util.Preconditions;
import java.nio.ByteBuffer;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

@RequiresApi(value=21)
public class SilentAudioStream
implements AudioStream {
    private static final String TAG = "SilentAudioStream";
    private final AtomicBoolean mIsStarted = new AtomicBoolean(false);
    private final AtomicBoolean mIsReleased = new AtomicBoolean(false);
    private final int mBytesPerFrame;
    private final int mSampleRate;
    @Nullable
    private byte[] mZeroBytes;
    private long mCurrentReadTimeNs;
    @Nullable
    private AudioStream.AudioStreamCallback mAudioStreamCallback;
    @Nullable
    private Executor mCallbackExecutor;

    public SilentAudioStream(@NonNull AudioSettings audioSettings) {
        this.mBytesPerFrame = audioSettings.getBytesPerFrame();
        this.mSampleRate = audioSettings.getSampleRate();
    }

    @Override
    public void setCallback(@Nullable AudioStream.AudioStreamCallback callback, @Nullable Executor executor) {
        Preconditions.checkState((!this.mIsStarted.get() ? 1 : 0) != 0, (String)"AudioStream can not be started when setCallback.");
        this.checkNotReleasedOrThrow();
        Preconditions.checkArgument((callback == null || executor != null ? 1 : 0) != 0, (Object)"executor can't be null with non-null callback.");
        this.mAudioStreamCallback = callback;
        this.mCallbackExecutor = executor;
    }

    @Override
    public void start() {
        this.checkNotReleasedOrThrow();
        if (this.mIsStarted.getAndSet(true)) {
            return;
        }
        this.mCurrentReadTimeNs = SilentAudioStream.currentSystemTimeNs();
        this.notifySilenced();
    }

    @Override
    public void stop() {
        this.checkNotReleasedOrThrow();
        this.mIsStarted.set(false);
    }

    @Override
    public void release() {
        this.mIsReleased.getAndSet(true);
    }

    @Override
    @NonNull
    public AudioStream.PacketInfo read(@NonNull ByteBuffer byteBuffer) {
        this.checkNotReleasedOrThrow();
        this.checkStartedOrThrow();
        long requiredFrameCount = AudioUtils.sizeToFrameCount(byteBuffer.remaining(), this.mBytesPerFrame);
        int requiredSize = (int)AudioUtils.frameCountToSize(requiredFrameCount, this.mBytesPerFrame);
        if (requiredSize <= 0) {
            return AudioStream.PacketInfo.of(0, this.mCurrentReadTimeNs);
        }
        long requiredDurationNs = AudioUtils.frameCountToDurationNs(requiredFrameCount, this.mSampleRate);
        long nextReadTimeNs = this.mCurrentReadTimeNs + requiredDurationNs;
        SilentAudioStream.blockUntilSystemTimeReached(nextReadTimeNs);
        this.writeSilenceToBuffer(byteBuffer, requiredSize);
        AudioStream.PacketInfo packetInfo = AudioStream.PacketInfo.of(requiredSize, this.mCurrentReadTimeNs);
        this.mCurrentReadTimeNs = nextReadTimeNs;
        return packetInfo;
    }

    private void writeSilenceToBuffer(@NonNull ByteBuffer byteBuffer, int sizeInBytes) {
        Preconditions.checkState((sizeInBytes <= byteBuffer.remaining() ? 1 : 0) != 0);
        if (this.mZeroBytes == null || this.mZeroBytes.length < sizeInBytes) {
            this.mZeroBytes = new byte[sizeInBytes];
        }
        int originalPosition = byteBuffer.position();
        byteBuffer.put(this.mZeroBytes, 0, sizeInBytes).limit(originalPosition + sizeInBytes).position(originalPosition);
    }

    private void notifySilenced() {
        AudioStream.AudioStreamCallback callback = this.mAudioStreamCallback;
        Executor executor = this.mCallbackExecutor;
        if (callback != null && executor != null) {
            executor.execute(() -> callback.onSilenceStateChanged(true));
        }
    }

    private void checkNotReleasedOrThrow() {
        Preconditions.checkState((!this.mIsReleased.get() ? 1 : 0) != 0, (String)"AudioStream has been released.");
    }

    private void checkStartedOrThrow() {
        Preconditions.checkState((boolean)this.mIsStarted.get(), (String)"AudioStream has not been started.");
    }

    private static long currentSystemTimeNs() {
        return System.nanoTime();
    }

    private static void blockUntilSystemTimeReached(long nextReadTimeNs) {
        long requiredBlockTimeNs = nextReadTimeNs - SilentAudioStream.currentSystemTimeNs();
        if (requiredBlockTimeNs > 0L) {
            try {
                Thread.sleep(TimeUnit.NANOSECONDS.toMillis(requiredBlockTimeNs));
            }
            catch (InterruptedException e) {
                Logger.w((String)TAG, (String)"Ignore interruption", (Throwable)e);
            }
        }
    }
}

