/*
 * Copyright 2019 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.compose

import androidx.compose.SlotTable.Companion.EMPTY
import kotlin.collections.isNotEmpty

internal typealias Change<N> = (
    applier: Applier<N>,
    slots: SlotWriter,
    lifecycleManager: LifeCycleManager
) -> Unit

private class GroupInfo(
    /** The current location of the slot relative to the start location of the pending slot changes
     */
    var slotIndex: Int,

    /** The current location of the first node relative the start location of the pending node
     * changes
     */
    var nodeIndex: Int,

    /** The current number of nodes the group contains after changes have been applied */
    var nodeCount: Int
)

internal interface LifeCycleManager {
    fun entering(holder: CompositionLifecycleObserverHolder): CompositionLifecycleObserverHolder
    fun leaving(holder: CompositionLifecycleObserverHolder)
}

/**
 * Pending starts when the key is different than expected indicating that the structure of the tree
 * changed. It is used to determine how to update the nodes and the slot table when changes to the
 * structure of the tree is detected.
 */
private class Pending(
    val parentKeyInfo: KeyInfo,
    val keyInfos: MutableList<KeyInfo>,
    val startIndex: Int
) {
    var groupIndex: Int = 0

    init {
        require(startIndex >= 0) { "Invalid start index" }
    }

    var nodeCount = parentKeyInfo.nodes

    private val usedKeys = mutableListOf<KeyInfo>()
    private val groupInfos = run {
        var runningNodeIndex = 0
        val result = hashMapOf<Any?, GroupInfo>()
        for (index in 0 until keyInfos.size) {
            val key = keyInfos[index]
            result[key] = GroupInfo(index, runningNodeIndex, key.nodes)
            runningNodeIndex += key.nodes
        }
        result
    }

    /**
     * A multi-map of keys from the previous composition. The keys can be retrieved in the order
     * they were generated by the previous composition.
     */
    val keyMap by lazy {
        multiMap<Any, KeyInfo>().also {
            for (index in 0 until keyInfos.size) {
                val keyInfo = keyInfos[index]
                @Suppress("ReplacePutWithAssignment")
                it.put(keyInfo.key, keyInfo)
            }
        }
    }

    /**
     * Get the next key information for the given key.
     */
    fun getNext(key: Any): KeyInfo? = keyMap.pop(key)

    /**
     * Record that this key info was generated.
     */
    fun recordUsed(keyInfo: KeyInfo) = usedKeys.add(keyInfo)

    val used: List<KeyInfo> get() = usedKeys

    // TODO(chuckj): This is a correct but expensive implementation (worst cases of O(N^2)). Rework
    // to O(N)
    fun registerMoveSlot(from: Int, to: Int) {
        if (from > to) {
            groupInfos.values.forEach { group ->
                val position = group.slotIndex
                if (position == from) group.slotIndex = to
                else if (position in to until from) group.slotIndex = position + 1
            }
        } else if (to > from) {
            groupInfos.values.forEach { group ->
                val position = group.slotIndex
                if (position == from) group.slotIndex = to
                else if (position in (from + 1) until to) group.slotIndex = position - 1
            }
        }
    }

    fun registerMoveNode(from: Int, to: Int, count: Int) {
        if (from > to) {
            groupInfos.values.forEach { group ->
                val position = group.nodeIndex
                if (position in from until from + count) group.nodeIndex = to + (position - from)
                else if (position in to until from) group.nodeIndex = position + count
            }
        } else if (to > from) {
            groupInfos.values.forEach { group ->
                val position = group.nodeIndex
                if (position in from until from + count) group.nodeIndex = to + (position - from)
                else if (position in (from + 1) until to) group.nodeIndex = position - count
            }
        }
    }

    fun registerInsert(keyInfo: KeyInfo, insertIndex: Int) {
        groupInfos[keyInfo] = GroupInfo(-1, insertIndex, 0)
    }

    fun updateNodeCount(keyInfo: KeyInfo?, newCount: Int) {
        groupInfos[keyInfo]?.let {
            val index = it.nodeIndex
            val difference = newCount - it.nodeCount
            it.nodeCount = newCount
            if (difference != 0) {
                nodeCount += difference
                groupInfos.values.forEach { group ->
                    if (group.nodeIndex >= index && group != it)
                        group.nodeIndex += difference
                }
            }
        }
    }

    fun slotPositionOf(keyInfo: KeyInfo) = groupInfos[keyInfo]?.slotIndex ?: -1
    fun nodePositionOf(keyInfo: KeyInfo) = groupInfos[keyInfo]?.nodeIndex ?: -1
    fun updatedNodeCountOf(keyInfo: KeyInfo) = groupInfos[keyInfo]?.nodeCount ?: keyInfo.nodes
}

private val RootKey = OpaqueKey("root")

private class Invalidation(
    val scope: RecomposeScope,
    var location: Int
)

interface ScopeUpdateScope {
    fun updateScope(block: (Composer<*>) -> Unit)
}

internal enum class InvalidationResult {
    /**
     * The invalidation was ignored because the associated recompose scope is no longer part of the
     * composition or has yet to be entered in the composition. This could occur for invalidations
     * called on scopes that are no longer part of composition or if the scope was invalidated
     * before the applyChanges() was called that will enter the scope into the composition.
     */
    IGNORED,

    /**
     * The composition is not currently composing and the invalidation was recorded for a future
     * composition. A recomposition requested to be scheduled.
     */
    SCHEDULED,

    /**
     * The composition that owns the recompose scope is actively composing but the scope has
     * already been composed or is in the process of composing. The invalidation is treated as
     * SCHEDULED above.
     */
    DEFERRED,

    /**
     * The composition that owns the recompose scope is actively composing and the invalidated
     * scope has not been composed yet but will be recomposed before the composition completes. A
     * new recomposition was not scheduled for this invalidation.
     */
    IMMINENT
}

internal class RecomposeScope(var composer: Composer<*>) : ScopeUpdateScope {
    var anchor: Anchor? = null
    val valid: Boolean get() = anchor?.valid ?: false
    var used = false

    private var block: ((Composer<*>) -> Unit)? = null

    @Suppress("UNUSED_PARAMETER")
    fun <N> compose(composer: Composer<N>) {
        block?.invoke(composer) ?: error("Invalid restart scope")
    }

    fun invalidate(): InvalidationResult = composer.invalidate(this)

    // Called caller of endRestartGroup()
    override fun updateScope(block: (Composer<*>) -> Unit) { this.block = block }
}

class ProvidedValue<T> internal constructor(val ambient: Ambient<T>, val value: T)

internal typealias AmbientMap = BuildableMap<Ambient<Any?>, State<Any?>>

@Suppress("UNCHECKED_CAST")
internal fun <T> AmbientMap.contains(key: Ambient<T>) = this.containsKey(key as Ambient<Any?>)

@Suppress("UNCHECKED_CAST")
internal fun <T> AmbientMap.getValueOf(key: Ambient<T>) = this[key as Ambient<Any?>]?.value as T

@Composable
private fun ambientMapOf(values: Array<out ProvidedValue<*>>): AmbientMap {
    val result: AmbientMap = buildableMapOf()
    return result.mutate {
        for (provided in values) {
            @Suppress("UNCHECKED_CAST")
            it[provided.ambient as Ambient<Any?>] = provided.ambient.provided(provided.value)
        }
    }
}

interface ComposerValidator {
    fun changed(value: Int): Boolean
    fun <T> changed(value: T): Boolean
}

// TODO(lmr): this could be named MutableTreeComposer
/**
 * Implementation of a composer for mutable tree.
 */
open class Composer<N>(
    /**
     * Backing storage for the composition
     */
    val slotTable: SlotTable,

    /**
     * An adapter that applies changes to the tree using the Applier abstraction.
     */
    private val applier: Applier<N>,

    /**
     * Manager for scheduling recompositions.
     */
    private val recomposer: Recomposer
) : ComposerValidator {
    private val changes = mutableListOf<Change<N>>()
    private val lifecycleObservers = mutableMapOf<
            CompositionLifecycleObserverHolder,
            CompositionLifecycleObserverHolder
            >()
    private val pendingStack = Stack<Pending?>()
    private var pending: Pending? = null
    private var nodeIndex: Int = 0
    private var nodeIndexStack = IntStack()
    private var groupNodeCount: Int = 0
    private var groupNodeCountStack = IntStack()
    private var collectKeySources = false
    private val keyHashesStack = IntStack()

    private var childrenAllowed = true
    private var invalidations: MutableList<Invalidation> = mutableListOf()
    private val entersStack = IntStack()
    private val providersStack = Stack<Pair<Int, AmbientMap>>().apply {
        push(-1 to buildableMapOf())
    }
    private var providersInvalidStack = IntStack()

    private val invalidateStack = Stack<RecomposeScope>()

    internal var parentReference: CompositionReference? = null
    internal var isComposing = false

    private val changesAppliedObservers = mutableListOf<() -> Unit>()

    private fun dispatchChangesAppliedObservers() {
        trace("Compose:dispatchChangesAppliedObservers") {
            val listeners = changesAppliedObservers.toTypedArray()
            changesAppliedObservers.clear()
            listeners.forEach { it() }
        }
    }

    internal fun addChangesAppliedObserver(l: () -> Unit) {
        changesAppliedObservers.add(l)
    }

    internal fun removeChangesAppliedObserver(l: () -> Unit) {
        changesAppliedObservers.remove(l)
    }

    // Temporary to allow staged changes. This will move into a sub-object that represents an active
    // composition created by startRoot() and recomposeComponentRange()
    private lateinit var slots: SlotReader

    protected fun composeRoot(block: () -> Unit) {
        startRoot()
        block()
        endRoot()
    }

    inline fun call(
        key: Any,
        invalid: ComposerValidator.() -> Boolean,
        block: () -> Unit
    ) {
        startGroup(key)
        if (this.invalid() || !skipping) {
            startGroup(invocation)
            block()
            endGroup()
        } else {
            skipCurrentGroup()
        }
        endGroup()
    }

    /**
     * Start the composition. This should be called, and only be called, as the first group in
     * the composition.
     */
    fun startRoot() {
        slots = slotTable.openReader()
        startGroup(RootKey)
        parentReference?.let { parentRef ->
            val parentScope = parentRef.getAmbientScope()
            providersInvalidStack.push(changed(parentScope).asInt())
            providersStack.push(0 to parentScope)
        }
    }

    /**
     * End the composition. This should be called, and only be called, to end the first group in
     * the composition.
     */
    fun endRoot() {
        endGroup()
        if (parentReference != null) {
            providersStack.pop()
        }
        finalizeCompose()
        slots.close()
    }

    /**
     * Discard a pending composition because an error was encountered during composition
     */
    fun abortRoot() {
        cleanUpCompose()
        pendingStack.clear()
        keyHashesStack.clear()
        nodeIndexStack.clear()
        groupNodeCountStack.clear()
        entersStack.clear()
        providersInvalidStack.clear()
        invalidateStack.clear()
        slots.close()
        currentCompoundKeyHash = 0
    }

    /**
     * True if the composition is currently scheduling nodes to be inserted into the tree. During
     * first composition this is always true. During recomposition this is true when new nodes
     * are being scheduled to be added to the tree.
     */
    val inserting: Boolean get() = slots.inEmpty

    /**
     * True if the composition should be checking if the composable functions can be skipped.
     */
    val skipping: Boolean get() = !inserting && !providersInvalidStack.peekOr(0).asBool()

    /**
     * Returns the hash of the compound key calculated as a combination of the keys of all the
     * currently started groups via [startGroup].
     */
    var currentCompoundKeyHash: Int = 0
        private set

    /**
     * Start collecting key source information. This enables enables the tool API to be able to
     * determine the source location of where groups and nodes are created.
     */
    fun collectKeySourceInformation() {
        collectKeySources = true
    }

    /**
     * Apply the changes to the tree that were collected during the last composition.
     */
    fun applyChanges() {
        trace("Compose:applyChanges") {
            invalidateStack.clear()
            val enters = mutableSetOf<CompositionLifecycleObserverHolder>()
            val leaves = mutableSetOf<CompositionLifecycleObserverHolder>()
            val manager = object : LifeCycleManager {
                override fun entering(
                    holder: CompositionLifecycleObserverHolder
                ): CompositionLifecycleObserverHolder {
                    return lifecycleObservers.getOrPut(holder) {
                        enters.add(holder)
                        holder
                    }.apply { count++ }
                }

                override fun leaving(holder: CompositionLifecycleObserverHolder) {
                    if (--holder.count == 0) {
                        leaves.add(holder)
                    }
                }
            }

            val invalidationAnchors = invalidations.map { slotTable.anchor(it.location) to it }

            // Apply all changes
            slotTable.write { slots ->
                changes.forEach { change -> change(applier, slots, manager) }
                changes.clear()
            }

            @Suppress("ReplaceManualRangeWithIndicesCalls") // Avoids allocation of an iterator
            for (index in 0 until invalidationAnchors.size) {
                val (anchor, invalidation) = invalidationAnchors[index]
                invalidation.location = slotTable.anchorLocation(anchor)
            }

            trace("Compose:lifecycles") {
                // Send lifecycle leaves
                if (leaves.isNotEmpty()) {
                    for (holder in leaves.reversed()) {
                        // The count of the holder might be greater than 0 here as it might leave one
                        // part of the composition and reappear in another. Only send a leave if the
                        // count is still 0 after all changes have been applied.
                        if (holder.count == 0) {
                            holder.instance.onLeave()
                            lifecycleObservers.remove(holder)
                        }
                    }
                }

                // Send lifecycle enters
                if (enters.isNotEmpty()) {
                    for (holder in enters) {
                        holder.instance.onEnter()
                    }
                }
            }

            dispatchChangesAppliedObservers()
        }
    }

    /**
     * Start a group with the given key. During recomposition if the currently expected group does
     * not match the given key a group the groups emitted in the same parent group are inspected
     * to determine if one of them has this key and that group the first such group is moved
     * (along with any nodes emitted by the group) to the current position and composition
     * continues. If no group with this key is found, then the composition shifts into insert
     * mode and new nodes are added at the current position.
     *
     *  @param key The key for the group
     */
    fun startGroup(key: Any) = start(key, START_GROUP)

    /**
     * End the current group.
     */
    fun endGroup() = end(END_GROUP)

    fun startExpr(key: Any) = startGroup(key)

    fun endExpr() = endGroup()

    private fun skipGroup() {
        recordSkip(START_GROUP)
        groupNodeCount += slots.skipGroup()
    }

    /**
     * Start emitting a node. It is required that one of [emitNode] or [createNode] is called
     * after [startNode]. Similar to [startGroup], if, during recomposition, the current node
     * does not have the provided key a node with that key is scanned for and moved into the
     * current position if found, if no such node is found the composition switches into insert
     * mode and a the node is scheduled to be inserted at the current location.
     *
     * @param key the key for the node.
     */
    fun startNode(key: Any) {
        start(key, START_NODE)
        childrenAllowed = false
    }

    // Deprecated
    fun <T : N> emitNode(factory: () -> T) {
        if (inserting) {
            val insertIndex = nodeIndexStack.peek()
            // The pending is the pending information for where the node is being inserted.
            // pending will be null here when the parent was inserted too.
            pending?.let { it.nodeCount++ }
            groupNodeCount++
            recordOperation { applier, slots, _ ->
                val node = factory()
                slots.update(node)
                applier.insert(insertIndex, node)
                applier.down(node)
            }
        } else {
            recordDown()
            slots.next() // Skip node slot
        }
        childrenAllowed = true
    }

    /**
     * Schedule a node to be created and inserted at the current location. This is only valid to call when the
     * composer is inserting.
     */
    @Suppress("UNUSED")
    fun <T : N> createNode(factory: () -> T) {
        val insertIndex = nodeIndexStack.peek()
        // see emitNode
        pending?.let { it.nodeCount++ }
        groupNodeCount++
        recordOperation { applier, slots, _ ->
            val node = factory()
            slots.update(node)
            applier.insert(insertIndex, node)
            applier.down(node)
        }
        childrenAllowed = true
    }

    /**
     * Schedule the given node to be inserted. This is only valid to call when the composer is
     * inserting.
     */
    fun emitNode(node: N) {
        require(inserting) { "emitNode() called when not inserting" }
        val insertIndex = nodeIndexStack.peek()
        // see emitNode
        pending?.let { it.nodeCount++ }
        groupNodeCount++
        recordOperation { applier, slots, _ ->
            slots.update(node)
            applier.insert(insertIndex, node)
            applier.down(node)
        }
        childrenAllowed = true
    }

    /**
     * Return the instance of the node that was inserted at the given location. This is only
     * valid to call when the composition is not inserting. This must be called at the same
     * location as [emitNode] or [createNode] as called even if the value is unused.
     */
    fun useNode(): N {
        require(!inserting) { "useNode() called while inserting" }
        recordDown()
        val result = slots.next()
        childrenAllowed = true
        @Suppress("UNCHECKED_CAST")
        return result as N
    }

    /**
     * Called to end the node group.
     */
    fun endNode() {
        end(END_NODE)
    }

    /**
     * Schedule a change to be applied to a node's property. This change will be applied to the
     * node that is the current node in the tree which was either created by [createNode],
     * emitted by [emitNode] or returned by [useNode].
     */
    fun <V, T> apply(value: V, block: T.(V) -> Unit) {
        recordOperation { applier, _, _ ->
            @Suppress("UNCHECKED_CAST")
            (applier.current as T).block(value)
        }
    }

    /**
     * Create a composed key that can be used in calls to [startGroup] or [startNode]. This will
     * use the key stored at the current location in the slot table to avoid allocating a new key.
     */
    fun joinKey(left: Any?, right: Any?): Any =
        getKey(slots.groupKey, left, right) ?: JoinedKey(left, right)

    /**
     * Return the next value in the slot table and advance the current location.
     */
    fun nextSlot(): Any? = slots.next()

    /**
     * Schedule a slot advance to happen during [applyChanges].
     */
    fun skipValue() = recordSlotNext()

    /**
     * Determine if the current slot table value is equal to the given value, if true, the value
     * is scheduled to be skipped during [applyChanges] and [changes] return false; otherwise
     * [applyChanges] will update the slot table to [value]. In either case the composer's slot
     * table is advanced.
     *
     * @param value the value to be compared.
     */
    override fun <T> changed(value: T): Boolean {
        return if (nextSlot() != value || inserting) {
            updateValue(value)
            true
        } else {
            skipValue()
            false
        }
    }

    // TODO: Add more overloads for common primitive types like String and Float etc to avoid boxing
    override fun changed(value: Int): Boolean {
        return if (nextSlot() != value || inserting) {
            updateValue(value)
            true
        } else {
            skipValue()
            false
        }
    }

    /**
     * Schedule the current value in the slot table to be updated to [value].
     *
     * @param value the value to schedule to be written to the slot table.
     */
    fun updateValue(value: Any?) {
        recordOperation { _, slots, lifecycleManager ->
            val previous = if (value is CompositionLifecycleObserver) {
                val slotValue = lifecycleManager.entering(
                    CompositionLifecycleObserverHolder(
                        value
                    )
                )
                slots.update(slotValue)
            } else slots.update(value)
            if (previous is CompositionLifecycleObserverHolder) {
                lifecycleManager.leaving(previous)
            }
        }
    }

    /**
     * Return the provider scope provided at [location]
     */
    private fun providedScopeAt(location: Int): AmbientMap {
        require(slots.groupKey(location) == provider)
        val valuesGroupSize = slots.groupSize(location + 1)
        val mapLocation = location + valuesGroupSize + 3 // one for each group slot
        @Suppress("UNCHECKED_CAST")
        return slots.get(mapLocation) as AmbientMap
    }

    /**
     * Return the current ambient scope which was provided by a parent group.
     */
    private fun currentAmbientScope(): AmbientMap = providersStack.peek().second

    /**
     * Return the ambient scope for the location provided. If this is while the composer is
     * composing then this is a query from a sub-composition that is being recomposed by this
     * compose which might be inserting the sub-composition. In that case the current scope
     * is the correct scope.
     */
    private fun ambientScopeAt(location: Int): AmbientMap {
        if (isComposing) {
            // The sub-composer is being composed as part of a nested composition then use the
            // current ambient scope as the one in the slot table might be out of date.
            return currentAmbientScope()
        }

        // Find the nearest provider group to location and return the scope recorded there.
        val groupPath = slotTable.groupPathTo(location)
        return slotTable.read { reader ->
            groupPath.lastOrNull { reader.groupKey(it) == provider }?.let {
                providedScopeAt(it)
            } ?: buildableMapOf()
        }
    }

    /**
     * Record [scope] as the current scope provided by the current group.
     */
    private fun pushProviderScope(scope: AmbientMap) {
        providersStack.push(slots.startStack.peekOr(0) to scope)
    }

    /**
     * Update (or create) the slots to record the providers. The providers maps are first the
     * scope followed by the map used to augment the parent scope. Both are needed to detect
     * inserts, updates and deletes to the providers.
     */
    private fun updateProviderMapGroup(
        parentScope: AmbientMap,
        currentProviders: AmbientMap
    ): AmbientMap {
        val providerScope = parentScope.mutate { it.putAll(currentProviders) }
        startGroup(providerMaps)
        changed(providerScope)
        changed(currentProviders)
        endGroup()
        return providerScope
    }

    internal fun startProviders(values: Array<out ProvidedValue<*>>) {
        val parentScope = currentAmbientScope()
        startGroup(provider)
        // The group is needed here because ambientMapOf() might change the number or kind of
        // slots consumed depending on the content of values to remember, for example, the value
        // holders used last time.
        startGroup(providerValues)
        val currentProviders = invokeComposableForResult(this) { ambientMapOf(values) }
        endGroup()
        val providersStackSize = providersStack.size
        val invalid = if (inserting) {
            val providers = updateProviderMapGroup(parentScope, currentProviders)
            pushProviderScope(providers)
            false
        } else {
            val current = slots.current

            @Suppress("UNCHECKED_CAST")
            val oldScope = slots.get(current + 1) as AmbientMap

            @Suppress("UNCHECKED_CAST")
            val oldValues = slots.get(current + 2) as AmbientMap

            // skipping is true iff parentScope has not changed.
            if (!skipping || oldValues != currentProviders) {
                val providers = updateProviderMapGroup(parentScope, currentProviders)
                pushProviderScope(providers)

                // Compare against the old scope as currentProviders might have modified the scope
                // back to the previous value. This could happen, for example, if currentProviders
                // and parentScope have a key in common and the oldScope had the same value as
                // currentProviders for that key. If the scope has not changed, because these
                // providers obscure a change in the parent as described above, re-enable skipping
                // for the child region.
                providers != oldScope
            } else {
                // Nothing has changed
                skipGroup()
                pushProviderScope(oldScope)
                false
            }
        }

        // If the provider scope has changed then we need prevent skipping until endProviders()
        // is called.
        providersInvalidStack.push(invalid.asInt())

        require(providersStackSize + 1 == providersStack.size) {
            "The provider stack was not updated correctly"
        }

        startGroup(invocation)
    }

    internal fun endProviders() {
        endGroup()
        val group = slots.startStack.peekOr(0)
        endGroup()
        val (topGroup, _) = providersStack.pop()
        require(group == topGroup)
        providersInvalidStack.pop()
    }

    @PublishedApi
    internal fun <T> consume(key: Ambient<T>): T = resolveAmbient(key, currentAmbientScope())

    /**
     * Create or use a memoized `CompositionReference` instance at this position in the slot table.
     */
    fun buildReference(): CompositionReference {
        startGroup(reference)

        // NOTE(lmr): VERY important to call nextValue() here instead of nextSlot()
        var ref = nextValue() as? CompositionReference
        if (ref != null && !inserting) {
            skipValue()
        } else {
            val scope = invalidateStack.peek()
            scope.used = true
            ref = CompositionReferenceImpl(scope)
            updateValue(ref)
        }
        endGroup()

        return ref
    }

    private fun <T> resolveAmbient(key: Ambient<T>, scope: AmbientMap): T {
        if (scope.contains(key)) return scope.getValueOf(key)

        val ref = parentReference

        if (ref != null) {
            return ref.getAmbient(key)
        }

        return key.defaultValueHolder.value
    }

    internal fun <T> parentAmbient(key: Ambient<T>): T = resolveAmbient(key, currentAmbientScope())

    private fun <T> parentAmbient(key: Ambient<T>, location: Int): T =
        resolveAmbient(key, ambientScopeAt(location))

    /**
     * The number of changes that have been scheduled to be applied during [applyChanges].
     *
     * Slot table movement (skipping groups and nodes) will be coalesced so this number is
     * possibly less than the total changes detected.
     */
    val changeCount get() = changes.size

    internal val currentRecomposeScope: RecomposeScope?
        get() =
            invalidateStack.let { if (it.isNotEmpty()) it.peek() else null }

    private fun start(key: Any, action: SlotAction) {
        require(childrenAllowed) { "A call to createNode(), emitNode() or useNode() expected" }

        if (action == START_GROUP || action == START_GROUP_SAME_KEY) {
            updateCompoundKeyWhenWeEnterGroup(key)
        }

        // Check for the insert fast path. If we are already inserting (creating nodes) then
        // there is no need to track insert, deletes and moves with a pending changes object.
        if (inserting) {
            slots.beginEmpty()
            if (collectKeySources)
                recordSourceKeyInfo(key)
            recordOperation { _, slots, _ ->
                slots.start(key, action)
            }
            pending?.let { pending ->
                val insertKeyInfo = KeyInfo(key, -1, 0, -1)
                pending.registerInsert(insertKeyInfo, nodeIndex - pending.startIndex)
                pending.recordUsed(insertKeyInfo)
            }
            enterGroup(action, null)
            return
        }

        if (pending == null) {
            val slotKey = slots.groupKey
            if (slotKey == key) {
                // The group is the same as what was generated last time.
                slots.start(key, action)
                recordStart(key, action)
            } else {
                val nodes = slots.parentNodes - slots.nodeIndex
                pending = Pending(
                    KeyInfo(0, -1, nodes, -1),
                    slots.extractKeys(),
                    nodeIndex
                )
            }
        }

        val pending = pending
        var newPending: Pending? = null
        if (pending != null) {
            // Check to see if the key was generated last time from the keys collected above.
            val keyInfo = pending.getNext(key)
            if (keyInfo != null) {
                // This group was generated last time, use it.
                pending.recordUsed(keyInfo)

                // Move the slot table to the location where the information about this group is
                // stored. The slot information will move once the changes are applied so moving the
                // current of the slot table is sufficient.
                slots.current = keyInfo.location
                slots.start(key, action)

                // Determine what index this group is in. This is used for inserting nodes into the
                // group.
                nodeIndex = pending.nodePositionOf(keyInfo) + pending.startIndex

                // Determine how to move the slot group to the correct position.
                val relativePosition = pending.slotPositionOf(keyInfo)
                val currentRelativePosition = relativePosition - pending.groupIndex
                pending.registerMoveSlot(relativePosition, pending.groupIndex)
                if (currentRelativePosition > 0) {
                    // The slot group must be moved, record the move to be performed during apply.
                    recordOperation { _, slots, _ ->
                        slots.moveGroup(currentRelativePosition)
                        slots.start(key, action)
                    }
                } else {
                    // The slot group is already in the correct location. This can happen, for
                    // example, during an insert. If only one group is inserted, for example, the
                    // slot groups of the sibling after the insert will be in the right locations
                    // and need not be moved.
                    recordStart(key, action)
                }
            } else {
                // The group is new, go into insert mode. All child groups will be inserted until
                // this group is complete.
                if (!slots.inEmpty) {
                    recordOperation { _, slots, _ -> slots.beginInsert() }
                }
                slots.beginEmpty()
                if (collectKeySources)
                    recordSourceKeyInfo(key)
                recordOperation { _, slots, _ ->
                    slots.start(key, action)
                }
                val insertKeyInfo = KeyInfo(key, -1, 0, -1)
                pending.registerInsert(insertKeyInfo, nodeIndex - pending.startIndex)
                pending.recordUsed(insertKeyInfo)
                newPending = Pending(
                    insertKeyInfo, mutableListOf(),
                    if (action == START_NODE) 0 else nodeIndex
                )
            }
        }

        enterGroup(action, newPending)
    }

    private fun enterGroup(action: SlotAction, newPending: Pending?) {
        // When entering a group all the information about the parent should be saved, to be
        // restored when end() is called, and all the tracking counters set to initial state for the
        // group.
        pendingStack.push(pending)
        this.pending = newPending
        this.nodeIndexStack.push(nodeIndex)
        if (action == START_NODE) nodeIndex = 0
        this.groupNodeCountStack.push(groupNodeCount)
        groupNodeCount = 0
    }

    private fun end(action: SlotAction) {
        // All the changes to the group (or node) have been recorded. All new nodes have been
        // inserted but it has yet to determine which need to be removed or moved. Note that the
        // changes are relative to the first change in the list of nodes that are changing.
        if (action == END_GROUP) {
            updateCompoundKeyWhenWeExitGroup()
        }
        var expectedNodeCount = groupNodeCount
        val pending = pending
        if (pending != null && pending.keyInfos.size > 0) {
            // previous contains the list of keys as they were generated in the previous composition
            val previous = pending.keyInfos

            // current contains the list of keys in the order they need to be in the new composition
            val current = pending.used

            // usedKeys contains the keys that were used in the new composition, therefore if a key
            // doesn't exist in this set, it needs to be removed.
            val usedKeys = current.toSet()

            val movedKeys = mutableSetOf<KeyInfo>()
            var currentIndex = 0
            val currentEnd = current.size
            var previousIndex = 0
            val previousEnd = previous.size

            // Traverse the list of changes to determine startNode movement
            var nodeOffset = 0
            while (previousIndex < previousEnd) {
                val previousInfo = previous[previousIndex]
                if (!usedKeys.contains(previousInfo)) {
                    // If the key info was not used the group was deleted, remove the nodes in the
                    // group
                    val deleteOffset = pending.nodePositionOf(previousInfo)
                    recordRemoveNode(
                        deleteOffset +
                                pending.startIndex, previousInfo.nodes
                    )
                    pending.updateNodeCount(previousInfo, 0)
                    recordOperation { _, slots, lifecycleManager ->
                        removeCurrentGroup(slots, lifecycleManager)
                    }

                    // Remove any invalidations pending for the group being removed. These are no
                    // longer part of the composition. The group being composed is one after the
                    // start of the group.
                    invalidations.removeRange(
                        previousInfo.location,
                        previousInfo.location + slots.groupSize(previousInfo.location)
                    )
                    previousIndex++
                    continue
                }

                if (previousInfo in movedKeys) {
                    // If the group was already moved to the correct location, skip it.
                    previousIndex++
                    continue
                }

                if (currentIndex < currentEnd) {
                    // At this point current should match previous unless the group is new or was
                    // moved.
                    val currentInfo = current[currentIndex]
                    if (currentInfo !== previousInfo) {
                        val nodePosition = pending.nodePositionOf(currentInfo)
                        if (nodePosition != nodeOffset) {
                            val updatedCount = pending.updatedNodeCountOf(currentInfo)
                            recordMoveNode(
                                nodePosition + pending.startIndex,
                                nodeOffset + pending.startIndex, updatedCount
                            )
                            pending.registerMoveNode(nodePosition, nodeOffset, updatedCount)
                            movedKeys.add(currentInfo)
                        } // else the nodes are already in the correct position
                    } else {
                        // The correct nodes are in the right location
                        previousIndex++
                    }
                    currentIndex++
                    nodeOffset += pending.updatedNodeCountOf(currentInfo)
                }
            }

            // If there are any current nodes left they where inserted into the right location
            // during when the group began so the rest are ignored.

            realizeMovement()

            // We have now processed the entire list so move the slot table to the end of the list
            // by moving to the last key and skipping it.
            if (previous.size > 0) {
                slots.reportUncertainNodeCount()
                slots.current = previous[previous.size - 1].location
                slots.skipGroup()
            }
        }

        // Detect removing nodes at the end. No pending is created in this case we just have more
        // nodes in the previous composition than we expect (i.e. we are not yet at an end)
        val removeIndex = nodeIndex
        while (!slots.isGroupEnd) {
            val startSlot = slots.current
            val nodesToRemove = slots.skipGroup()
            recordRemoveNode(removeIndex, nodesToRemove)
            recordOperation { _, slots, lifeCycleManager ->
                removeCurrentGroup(slots, lifeCycleManager)
            }
            invalidations.removeRange(startSlot, slots.current)
            slots.reportUncertainNodeCount()
        }

        if (action == END_GROUP) slots.endGroup() else {
            expectedNodeCount = 1
            slots.endNode()
        }
        realizeMovement()
        if (action == END_NODE) recordUp()
        recordEnd(action)

        val inserting = slots.inEmpty
        if (inserting) {
            slots.endEmpty()
            if (!slots.inEmpty) recordOperation { _, slots, _ -> slots.endInsert() }
        }

        // Restore the parent's state updating them if they have changed based on changes in the
        // children. For example, if a group generates nodes then the number of generated nodes will
        // increment the node index and the group's node count. If the parent is tracking structural
        // changes in pending then restore that too.
        val previousPending = pendingStack.pop()
        if (previousPending != null) {
            // Update the parent count of nodes
            previousPending.updateNodeCount(pending?.parentKeyInfo, expectedNodeCount)
            if (!inserting) {
                previousPending.groupIndex++
            }
        }
        this.pending = previousPending
        this.nodeIndex = nodeIndexStack.pop() + expectedNodeCount
        this.groupNodeCount = this.groupNodeCountStack.pop() + expectedNodeCount
    }

    /**
     * Skip to a sibling group that contains location given. This also ensures the nodeIndex is
     * correctly updated to reflect any groups skipped.
     */
    private fun skipToGroupContaining(location: Int) {
        while (slots.current < location) {
            if (slots.isGroupEnd) return
            if (slots.isGroup) {
                if (location < slots.groupSize + slots.current) return
                recordSkip(if (slots.isNode) START_NODE else END_NODE)
                nodeIndex += slots.skipGroup()
            } else {
                recordSlotNext()
                slots.next()
            }
        }
    }

    /**
     * Enter a group that contains the location. This updates the composer state as if the group was
     * generated with no changes.
     */
    private fun recordEnters(location: Int) {
        trace("Compose:recordEnters") {
            while (true) {
                skipToGroupContaining(location)
                require(
                    slots.isGroup && location >= slots.current &&
                            location < slots.current + slots.groupSize
                ) {
                    "Could not find group at $location"
                }
                if (slots.current == location) {
                    return
                } else {
                    enterGroup(
                        if (slots.isNode) START_NODE
                        else START_GROUP, null
                    )
                    if (slots.isNode) {
                        recordStart(EMPTY, START_NODE)
                        recordDown()
                        entersStack.push(END_NODE)
                        slots.startNode(EMPTY)
                        slots.next() // skip navigation slot
                        nodeIndex = 0
                    } else {
                        updateCompoundKeyWhenWeEnterGroup(slots.groupKey)
                        // If the current group is an ambient provider, add the map to the ambient
                        // scope stack
                        if (slots.groupKey == provider) {
                            val current = slots.current
                            providersStack.push(current to providedScopeAt(current))
                        }
                        recordStart(EMPTY, START_GROUP)
                        entersStack.push(END_GROUP)
                        slots.startGroup(EMPTY)
                    }
                }
            }
        }
    }

    /**
     * Exit any groups that were entered until a sibling of maxLocation is reached.
     */
    private fun recordExits(maxLocation: Int, minStack: Int) {
        trace("Compose:recordExits") {
            var currentProviderScope = providersStack.peek().first
            while (entersStack.size > minStack) {
                skipToGroupContaining(maxLocation)
                if (slots.isGroupEnd) {
                    // If the current group is a provider scope, pop it off the stack as this
                    // exits the scope.
                    if (slots.startStack.peekOr(0) == currentProviderScope) {
                        providersStack.pop()
                        currentProviderScope = providersStack.peek().first
                    }
                    end(entersStack.pop())
                } else return
            }
        }
    }

    private fun recomposeComponentRange(start: Int, end: Int) {
        val wasComposing = isComposing
        isComposing = true
        var recomposed = false

        var firstInRange = invalidations.firstInRange(start, end)
        val enterStackSize = entersStack.size
        while (firstInRange != null) {
            val location = firstInRange.location

            invalidations.removeLocation(location)

            recordExits(location, enterStackSize)
            recordEnters(location)

            composeScope(firstInRange.scope)

            recomposed = true

            // Using slots.current here ensures composition always walks forward even if a component
            // before the current composition is invalidated when performing this composition. Any
            // such components will be considered invalid for the next composition. Skipping them
            // prevents potential infinite recomposes at the cost of potentially missing a compose
            // as well as simplifies the apply as it always modifies the slot table in a forward
            // direction.
            firstInRange = invalidations.firstInRange(slots.current, end)
        }

        if (recomposed) {
            recordExits(end, enterStackSize)
        } else {
            // No recompositions were requested in the range, skip it.
            skipGroup()
        }
        isComposing = wasComposing
    }

    internal fun invalidate(scope: RecomposeScope): InvalidationResult {
        val location = scope.anchor?.location(slotTable)
            ?: return InvalidationResult.IGNORED // The scope never entered the composition
        if (location < 0)
            return InvalidationResult.IGNORED // The scope was removed from the composition

        invalidations.insertIfMissing(location, scope)
        if (isComposing && location > slots.current) {
            // if we are invalidating a scope that is going to be traversed during this
            // composition.
            return InvalidationResult.IMMINENT
        }
        if (parentReference != null) {
            parentReference?.invalidate()
        } else {
            recomposer.scheduleRecompose(this)
        }
        return if (isComposing) InvalidationResult.DEFERRED else InvalidationResult.SCHEDULED
    }

    /**
     * Skip a group. This is only valid to call if the composition is not inserting.
     */
    fun skipCurrentGroup() {
        if (invalidations.isEmpty()) {
            skipGroup()
        } else {
            recomposeComponentRange(slots.current, slots.current + slots.groupSize)
        }
    }

    private fun composeScope(scope: RecomposeScope) {
        scope.compose(this)
    }

    private fun scheduleAnchor(scope: RecomposeScope) {
        recordOperation { _, slots, _ ->
            scope.anchor = slots.anchor(slots.parentIndex)
        }
    }

    /**
     * Start a restart group. A restart group creates a recompose scope and sets it as the current
     * recompose scope of the composition. If the recompose scope is invalidated then this group
     * will be recomposed. A recompose scope can be invalidated by calling the lambda returned by
     * [androidx.compose.invalidate].
     */
    fun startRestartGroup(key: Any) {
        val location = slots.current
        startGroup(key)
        if (inserting) {
            val scope = RecomposeScope(this)
            invalidateStack.push(scope)
            updateValue(scope)
        } else {
            invalidations.removeLocation(location)
            val scope = slots.next() as RecomposeScope
            invalidateStack.push(scope)
            skipValue()
        }
    }

    /**
     * End a restart group. If the recompose scope was marked used during composition then a
     * [ScopeUpdateScope] is returned that allows attaching a lambda that will produce the same
     * composition as was produced by this group (including calling [startRestartGroup] and
     * [endRestartGroup]).
     */
    fun endRestartGroup(): ScopeUpdateScope? {
        // This allows for the invalidate stack to be out of sync since this might be called during exception stack
        // unwinding that might have not called the doneJoin/endRestartGroup in the wrong order.
        val scope = if (invalidateStack.isNotEmpty()) invalidateStack.pop()
            else null
        val result = if (scope != null && scope.used) {
            if (scope.anchor == null) {
                scheduleAnchor(scope)
            }
            scope
        } else {
            null
        }
        end(END_GROUP)
        return result
    }

    /**
     * Synchronously recompose all invalidated groups. This collects the changes which must be
     * applied by [applyChanges] to have an effect.
     */
    fun recompose(): Boolean {
        if (invalidations.isNotEmpty()) {
            trace("Compose:recompose") {
                nodeIndex = 0
                var complete = false
                try {
                    startRoot()
                    skipCurrentGroup()
                    endRoot()
                    complete = true
                } finally {
                    if (!complete) abortRoot()
                }
                finalizeCompose()
            }
            return true
        }
        return false
    }

    internal fun hasInvalidations() = invalidations.isNotEmpty()

    private fun record(change: Change<N>) {
        changes.add(change)
    }

    private fun recordOperation(change: Change<N>) {
        realizeSlots()
        record(change)
    }

    private var slotsActionStartStack = IntStack()
    private var slotActions = SlotActions()

    // Slot movement
    private fun realizeSlots() {
        val actionsSize = slotActions.actionSize
        if (actionsSize > 0) {
            if (actionsSize == 1) {
                when (val action = slotActions.first()) {
                    START_GROUP -> {
                        val key = slotActions.getKey(0)
                        record { _, slots, _ -> slots.startGroup(key) }
                    }
                    START_GROUP_SAME_KEY -> record { _, slots, _ -> slots.startGroup(EMPTY) }
                    END_GROUP -> record { _, slots, _ -> slots.endGroup() }
                    SKIP_GROUP -> record { _, slots, _ -> slots.skipGroup() }
                    START_NODE -> {
                        val key = slotActions.getKey(0)
                        record { _, slots, _ -> slots.startNode(key) }
                    }
                    START_NODE_SAME_KEY -> record { _, slots, _ -> slots.startNode(EMPTY) }
                    END_NODE -> record { _, slots, _ -> slots.endNode() }
                    SKIP_NODE -> record { _, slots, _ -> slots.skipNode() }
                    DOWN -> record { applier, slots, _ ->
                        @Suppress("UNCHECKED_CAST")
                        applier.down(slots.skip() as N)
                    }
                    UP -> record { applier, _, _ -> applier.up() }
                    else -> record { _, slots, _ -> slots.current += action - SKIP_SLOTS }
                }
                slotActions.clear()
                slotsActionStartStack.clear()
            } else {
                val actions = slotActions.clone()
                slotActions.clear()
                slotsActionStartStack.clear()
                record { applier, slots, _ ->
                    var currentKey = 0
                    for (index in 0 until actions.actionSize) {
                        val action = actions.actions[index]
                        when (action) {
                            START_GROUP -> slots.startGroup(actions.getKey(currentKey++))
                            START_GROUP_SAME_KEY -> slots.startGroup(EMPTY)
                            END_GROUP -> slots.endGroup()
                            SKIP_GROUP -> slots.skipGroup()
                            START_NODE -> slots.startNode(actions.getKey(currentKey++))
                            START_NODE_SAME_KEY -> slots.startNode(EMPTY)
                            END_NODE -> slots.endNode()
                            SKIP_NODE -> slots.skipNode()
                            DOWN -> {
                                @Suppress("UNCHECKED_CAST")
                                applier.down(slots.skip() as N)
                            }
                            UP -> applier.up()
                            else -> slots.current += action - SKIP_SLOTS
                        }
                    }
                }
            }
        }
    }

    private fun finalRealizeSlots() {
        when (slotActions.actionSize) {
            0 -> Unit
            1 -> if (slotActions.first() == SKIP_GROUP) slotActions.clear() else realizeSlots()
            2 -> if (slotActions.first() >= SKIP_NODE &&
                slotActions.last() == SKIP_GROUP
            ) slotActions.clear()
            else realizeSlots()
            else -> realizeSlots()
        }
    }

    private fun finalizeCompose() {
        finalRealizeSlots()
        require(pendingStack.isEmpty()) { "Start/end imbalance" }
        require(providersStack.size == 1) {
            "Provider stack imbalance, stack size ${providersStack.size}"
        }
        cleanUpCompose()
    }

    private fun cleanUpCompose() {
        pending = null
        nodeIndex = 0
        groupNodeCount = 0
        providersStack.clear()
        providersStack.push(-1 to buildableMapOf())
    }

    private fun recordSlotNext(count: Int = 1) {
        require(count >= 1) { "Invalid call to recordSlotNext()" }
        val actionsSize = slotActions.actionSize
        if (actionsSize > 0) {
            // If the last action was also a skip just add this one to the last one
            val last = slotActions.last()
            if (last >= SKIP_SLOTS) {
                slotActions.remove(1)
                slotActions.add(last + count)
                return
            }
        }
        slotActions.add(SKIP_SLOTS + count)
    }

    private fun recordStart(key: Any, action: SlotAction) {
        slotsActionStartStack.push(slotActions.actionSize)
        slotActions.addStart(key, action)
    }

    private fun recordEnd(action: SlotAction) {
        if (slotsActionStartStack.isEmpty()) {
            slotActions.add(action)
        } else {
            // skip the entire group
            val startLocation = slotsActionStartStack.pop()
            slotActions.remove(slotActions.actionSize - startLocation)
            recordSkip(action)
        }
    }

    private fun recordSkip(action: SlotAction) {
        when (action) {
            START_GROUP, END_GROUP -> slotActions.add(SKIP_GROUP)
            START_NODE, END_NODE -> slotActions.add(SKIP_NODE)
            else -> error("Invalid skip action")
        }
    }

    private fun recordDown() {
        slotActions.add(DOWN)
    }

    private fun recordUp() {
        slotActions.add(UP)
    }

    private var previousRemove = -1
    private var previousMoveFrom = -1
    private var previousMoveTo = -1
    private var previousCount = 0

    private fun recordRemoveNode(nodeIndex: Int, count: Int) {
        if (count > 0) {
            require(nodeIndex >= 0) { "Invalid remove index $nodeIndex" }
            if (previousRemove == nodeIndex) previousCount += count
            else {
                realizeMovement()
                previousRemove = nodeIndex
                previousCount = count
            }
        }
    }

    private fun recordMoveNode(from: Int, to: Int, count: Int) {
        if (count > 0) {
            if (previousCount > 0 && previousMoveFrom == from - previousCount &&
                previousMoveTo == to - previousCount
            ) {
                previousCount += count
            } else {
                realizeMovement()
                previousMoveFrom = from
                previousMoveTo = to
                previousCount = count
            }
        }
    }

    private fun realizeMovement() {
        val count = previousCount
        previousCount = 0
        if (count > 0) {
            if (previousRemove >= 0) {
                val removeIndex = previousRemove
                previousRemove = -1
                recordOperation { applier, _, _ -> applier.remove(removeIndex, count) }
            } else {
                val from = previousMoveFrom
                previousMoveFrom = -1
                val to = previousMoveTo
                previousMoveTo = -1
                recordOperation { applier, _, _ -> applier.move(from, to, count) }
            }
        }
    }

    private inner class CompositionReferenceImpl(val scope: RecomposeScope) : CompositionReference,
        CompositionLifecycleObserver {

        val composers = mutableSetOf<Composer<*>>()

        override fun onEnter() {
            // do nothing
        }

        override fun onLeave() {
            composers.clear()
        }

        override fun <N> registerComposer(composer: Composer<N>) {
            composers.add(composer)
        }

        override fun invalidate() {
            // continue invalidating up the spine of AmbientReferences
            parentReference?.invalidate()

            invalidate(scope)
        }

        override fun <T> getAmbient(key: Ambient<T>): T {
            val anchor = scope.anchor
            return if (anchor != null && anchor.valid) {
                parentAmbient(key, anchor.location(slotTable))
            } else {
                // The composition is composing and the ambient has not landed in the slot table
                // yet. This is a synchronous read from a sub-composition so the current ambient
                parentAmbient(key)
            }
        }

        override fun getAmbientScope(): AmbientMap {
            return ambientScopeAt(scope.anchor?.location(slotTable) ?: 0)
        }
    }

    @UseExperimental(ExperimentalStdlibApi::class)
    private fun updateCompoundKeyWhenWeEnterGroup(groupKey: Any) {
        val keyHash = groupKey.hashCode()
        keyHashesStack.push(keyHash)
        currentCompoundKeyHash = currentCompoundKeyHash.rotateLeft(3) xor keyHash
    }

    @UseExperimental(ExperimentalStdlibApi::class)
    private fun updateCompoundKeyWhenWeExitGroup() {
        val keyHash = keyHashesStack.pop()
        currentCompoundKeyHash = (currentCompoundKeyHash xor keyHash).rotateRight(3)
    }
}

@Suppress("UNCHECKED_CAST")
/*inline */ class ComposerUpdater<N, T : N>(val composer: Composer<N>, val node: T) {
    inline fun set(
        value: Int,
        /*crossinline*/
        block: T.(value: Int) -> Unit
    ) = with(composer) {
        if (inserting || nextSlot() != value) {
            updateValue(value)
            node.block(value)
//            val appliedBlock: T.(value: Int) -> Unit = { block(it) }
//            composer.apply(value, appliedBlock)
        } else skipValue()
    }

    inline fun <reified V> set(
        value: V,
        /*crossinline*/
        block: T.(value: V) -> Unit
    ) = with(composer) {
        if (inserting || nextSlot() != value) {
            updateValue(value)
            node.block(value)
//            val appliedBlock: T.(value: V) -> Unit = { block(it) }
//            composer.apply(value, appliedBlock)
        } else skipValue()
    }

    inline fun update(
        value: Int,
        /*crossinline*/
        block: T.(value: Int) -> Unit
    ) = with(composer) {
        if (inserting || nextSlot() != value) {
            updateValue(value)
            node.block(value)
//            val appliedBlock: T.(value: Int) -> Unit = { block(it) }
//            if (!inserting) composer.apply(value, appliedBlock)
        } else skipValue()
    }

    inline fun <reified V> update(
        value: V,
        /*crossinline*/
        block: T.(value: V) -> Unit
    ) = with(composer) {
        if (inserting || nextSlot() != value) {
            updateValue(value)
            node.block(value)
//            val appliedBlock: T.(value: V) -> Unit = { block(it) }
//            if (!inserting) composer.apply(value, appliedBlock)
        } else skipValue()
    }
}

/**
 * Get the next value of the slot table. This will unwrap lifecycle observer holders to return
 * lifecycle observer and should be used instead of [Composer.nextSlot].
 */
fun <N> Composer<N>.nextValue(): Any? = nextSlot().let {
    if (it is CompositionLifecycleObserverHolder) it.instance else it
}

/**
 * Cache a value in the composition. During initial composition [block] is called to produce the
 * value that is then * stored in the slot table. During recomposition, if [valid] is true the
 * value is obtained from the slot table and [block] is not invoked. If [valid] is false a new
 * value is produced by calling [block] and the slot table is updated to contain the new value.
 */
inline fun <N, T> Composer<N>.cache(valid: Boolean = true, block: () -> T): T {
    var result = nextValue()
    if (result === EMPTY || !valid) {
        val value = block()
        updateValue(value)
        result = value
    } else skipValue()

    @Suppress("UNCHECKED_CAST")
    return result as T
}

private fun removeCurrentGroup(slots: SlotWriter, lifecycleManager: LifeCycleManager) {
    // Notify the lifecycle manager of any observers leaving the slot table
    // The notification order should ensure that listeners are notified of leaving
    // in opposite order that they are notified of entering.

    // To ensure this order, we call `enters` as a pre-order traversal
    // of the group tree, and then call `leaves` in the inverse order.

    var groupEnd = Int.MAX_VALUE
    var index = 0
    val groupEndStack = IntStack()

    for (slot in slots.groupSlots()) {
        when (slot) {
            is CompositionLifecycleObserverHolder -> {
                lifecycleManager.leaving(slot)
            }
            is GroupStart -> {
                groupEndStack.push(groupEnd)
                groupEnd = index + slot.slots
            }
        }

        index++

        while (index >= groupEnd) {
            groupEnd = groupEndStack.pop()
        }
    }

    if (groupEndStack.isNotEmpty()) error("Invalid slot structure")

    // Remove the item from the slot table and notify the FrameManager if any
    // anchors are orphaned by removing the slots.
    if (slots.removeGroup()) {
        FrameManager.scheduleCleanup()
    }
}

internal typealias SlotAction = Int

internal const val START_GROUP: SlotAction = 1
internal const val START_GROUP_SAME_KEY: SlotAction = START_GROUP + 1
internal const val END_GROUP: SlotAction = START_GROUP_SAME_KEY + 1
internal const val SKIP_GROUP: SlotAction = END_GROUP + 1
internal const val START_NODE: SlotAction = SKIP_GROUP + 1
internal const val START_NODE_SAME_KEY: SlotAction = START_NODE + 1
internal const val END_NODE: SlotAction = START_NODE_SAME_KEY + 1
internal const val SKIP_NODE: SlotAction = END_NODE + 1
internal const val DOWN: SlotAction = SKIP_NODE + 1
internal const val UP: SlotAction = DOWN + 1
internal const val SKIP_SLOTS: SlotAction = UP + 1

const val DEFAULT_SLOT_ACTIONS_SIZE = 16
const val DEFAULT_SLOT_KEYS_SIZE = 8
private class SlotActions(
    var actions: IntArray = IntArray(DEFAULT_SLOT_ACTIONS_SIZE),
    var keys: Array<Any?> = Array(DEFAULT_SLOT_KEYS_SIZE) { null }
) {
    var actionSize: Int = 0
    var keySize: Int = 0

    private fun addRaw(action: SlotAction) {
        if (actionSize >= actions.size) {
            actions = actions.copyOf(kotlin.math.max(actionSize, actions.size * 2))
        }
        actions[actionSize++] = action
    }

    fun add(action: SlotAction) {
        require(action != START_GROUP && action != START_NODE)
        addRaw(action)
    }

    fun addStart(key: Any, action: SlotAction) {
        require(action == START_GROUP || action == START_NODE)
        if (key == EMPTY) {
            addRaw(if (action == START_GROUP) START_GROUP_SAME_KEY else START_NODE_SAME_KEY)
        } else {
            addRaw(action)
            if (keySize >= keys.size) {
                keys = keys.copyOf(kotlin.math.max(keySize, keys.size * 2))
            }
            keys[keySize++] = key
        }
    }

    fun getKey(index: Int): Any {
        require(index in 0 until keySize)
        return keys[index]!!
    }

    fun remove(count: Int) {
        require(count <= actionSize) { "Removing too many actions" }
        val oldSize = actionSize
        actionSize -= count

        // Remove any keys introduced by start groups
        for (index in actionSize until oldSize)
            if (actions[index].let { it == START_GROUP || it == START_NODE })
                keySize--
    }

    fun clear() {
        actionSize = 0
        keySize = 0
    }

    fun clone(): SlotActions = SlotActions(
        actions.copyOf(this.actionSize), keys.copyOf(this.keySize)
    ).also {
        it.actionSize = actionSize
        it.keySize = keySize
    }

    override fun toString(): String = actions.take(actionSize).joinToString {
        when (it) {
            START_GROUP -> "START_GROUP"
            START_GROUP_SAME_KEY -> "START_GROUP_SAME_KEY"
            END_GROUP -> "END_GROUP"
            SKIP_GROUP -> "SKIP_GROUP"
            START_NODE -> "START_NODE"
            START_NODE_SAME_KEY -> "START_NODE_SAME_KEY"
            END_NODE -> "END_NODE"
            SKIP_NODE -> "SKIP_NODE"
            DOWN -> "DOWN"
            UP -> "UP"
            else -> "SKIP_SLOTS(${it - SKIP_SLOTS})"
        }
    }

    fun first(): SlotAction = actions[0]
    fun last(): SlotAction = actions[actionSize - 1]
}

// Mutable list
private fun <K, V> multiMap() = HashMap<K, LinkedHashSet<V>>()

private fun <K, V> HashMap<K, LinkedHashSet<V>>.put(key: K, value: V) = getOrPut(key) {
    LinkedHashSet()
}.add(value)

private fun <K, V> HashMap<K, LinkedHashSet<V>>.remove(key: K, value: V) =
    get(key)?.let {
        it.remove(value)
        if (it.isEmpty()) remove(key)
    }

private fun <K, V> HashMap<K, LinkedHashSet<V>>.pop(key: K) = get(key)?.firstOrNull()?.also {
    remove(key, it)
}

private fun SlotReader.start(key: Any, action: SlotAction) =
    if (action == START_NODE) startNode(key)
    else startGroup(key)

private fun SlotWriter.start(key: Any, action: SlotAction) =
    if (action == START_NODE) startNode(key)
    else startGroup(key)

private fun getKey(value: Any?, left: Any?, right: Any?): Any? = (value as? JoinedKey)?.let {
    if (it.left == left && it.right == right) value
    else getKey(it.left, left, right) ?: getKey(
        it.right,
        left,
        right
    )
}

// Invalidation helpers
private fun MutableList<Invalidation>.findLocation(location: Int): Int =
    binarySearch { it.location.compareTo(location) }

private fun MutableList<Invalidation>.insertIfMissing(location: Int, scope: RecomposeScope) {
    val index = findLocation(location)
    if (index < 0) {
        add(-(index + 1), Invalidation(scope, location))
    }
}

private fun MutableList<Invalidation>.firstInRange(start: Int, end: Int): Invalidation? {
    val index = findLocation(start).let { if (it < 0) -(it + 1) else it }
    if (index < size) {
        val firstInvalidation = get(index)
        if (firstInvalidation.location <= end) return firstInvalidation
    }
    return null
}

private fun MutableList<Invalidation>.removeLocation(location: Int) {
    val index = findLocation(location)
    if (index >= 0) removeAt(index)
}

private fun MutableList<Invalidation>.removeRange(start: Int, end: Int) {
    val index = findLocation(start).let { if (it < 0) -(it + 1) else it }
    while (index < size) {
        val validation = get(index)
        if (validation.location <= end) removeAt(index)
        else break
    }
}

private fun Boolean.asInt() = if (this) 1 else 0
private fun Int.asBool() = this != 0

object NullCompilationScope {
    val composer = Unit
}

inline fun <T> escapeCompose(block: NullCompilationScope.() -> T) = NullCompilationScope.block()

@Composable
val currentComposer: Composer<*> get() {
    throw NotImplementedError("Implemented as an intrinsic")
}

// TODO: get rid of the need for this when we merge FrameManager and Recomposer together!
internal var currentComposerInternal: Composer<*>? = null

internal fun invokeComposable(composer: Composer<*>, composable: @Composable() () -> Unit) {
    @Suppress("UNCHECKED_CAST")
    val realFn = composable as Function1<Composer<*>, Unit>
    realFn(composer)
}

internal fun <T> invokeComposableForResult(
    composer: Composer<*>,
    composable: @Composable() () -> T
): T {
    @Suppress("UNCHECKED_CAST")
    val realFn = composable as Function1<Composer<*>, T>
    return realFn(composer)
}

@PublishedApi
internal val invocation = Any()

@PublishedApi
internal val provider = Any()

@PublishedApi
internal val providerValues = Any()

@PublishedApi
internal val providerMaps = Any()

@PublishedApi
internal val consumer = Any()

@PublishedApi
internal val reference = Any()