/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.databinding.tool.writer

import android.databinding.tool.LibTypes
import android.databinding.tool.ext.L
import android.databinding.tool.ext.N
import android.databinding.tool.ext.T
import android.databinding.tool.ext.classSpec
import android.databinding.tool.ext.constructorSpec
import android.databinding.tool.ext.fieldSpec
import android.databinding.tool.ext.javaFile
import android.databinding.tool.ext.methodSpec
import android.databinding.tool.ext.parameterSpec
import android.databinding.tool.ext.toClassName
import android.databinding.tool.ext.toTypeName
import android.databinding.tool.store.GenClassInfoLog
import com.squareup.javapoet.ClassName
import com.squareup.javapoet.CodeBlock
import com.squareup.javapoet.FieldSpec
import com.squareup.javapoet.MethodSpec
import com.squareup.javapoet.TypeName
import javax.lang.model.element.Modifier

class BaseLayoutBinderWriter(val model: BaseLayoutModel, val libTypes: LibTypes) {
    companion object {
        private val DEPRECATED = ClassName.get(java.lang.Deprecated::class.java)

        private val JAVADOC_BINDING_COMPONENT = """
            This method receives DataBindingComponent instance as type Object instead of
            type DataBindingComponent to avoid causing too many compilation errors if
            compilation fails for another reason.
            https://issuetracker.google.com/issues/116541301
            """.trimIndent()
    }

    private val binderTypeName = ClassName.get(model.bindingClassPackage, model.bindingClassName)
    private val viewDataBinding = libTypes.viewDataBinding.toClassName()
    private val nonNull = libTypes.nonNull.toClassName()
    private val nullable = libTypes.nullable.toClassName()
    private val dataBindingComponent = TypeName.OBJECT
    private val dataBindingUtil = libTypes.dataBindingUtil.toClassName()
    private val bindable = libTypes.bindable.toClassName()

    fun write() = javaFile(binderTypeName.packageName(), createType()) {
        addFileComment("Generated by data binding compiler. Do not edit!")
    }

    private fun createType() = classSpec(binderTypeName) {
        superclass(viewDataBinding)
        addModifiers(Modifier.ABSTRACT, Modifier.PUBLIC)
        addFields(createBindingTargetFields())
        addFields(createVariableFields())
        addMethod(createConstructor())
        addMethods(createGettersAndSetters())
        addMethods(createStaticInflaters())
    }

    private fun createStaticInflaters(): List<MethodSpec> {
        val inflaterParam = parameterSpec(ANDROID_LAYOUT_INFLATER, "inflater") {
            addAnnotation(nonNull)
        }
        val viewGroupParam = parameterSpec(ANDROID_VIEW_GROUP, "root") {
            addAnnotation(nullable)
        }
        val viewParam = parameterSpec(ANDROID_VIEW, "view") {
            addAnnotation(nonNull)
        }
        val componentParam = parameterSpec(dataBindingComponent, "component") {
            addAnnotation(nullable)
        }
        val rLayoutFile = CodeBlock.of("$T.$N", ClassName.get(model.modulePackage, "R", "layout"),
            model.baseFileName)
        val attachToRootParam = parameterSpec(TypeName.BOOLEAN, "attachToRoot")
        return listOf(
                methodSpec("inflate") {
                    addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                    addParameter(inflaterParam)
                    addParameter(viewGroupParam)
                    addParameter(attachToRootParam)
                    returns(binderTypeName)
                    addAnnotation(nonNull)
                    addStatement("return inflate($N, $N, $N, $T.getDefaultComponent())",
                            inflaterParam, viewGroupParam, attachToRootParam,
                            dataBindingUtil)
                },
                methodSpec("inflate") {
                    addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                    addParameter(inflaterParam)
                    addParameter(viewGroupParam)
                    addParameter(attachToRootParam)
                    addParameter(componentParam)
                    returns(binderTypeName)
                    addAnnotation(nonNull)
                    addAnnotation(DEPRECATED)
                    addJavadoc(JAVADOC_BINDING_COMPONENT)
                    addJavadoc("\n@Deprecated Use DataBindingUtil.inflate(inflater, R.layout.$L, $N, $N, $N)\n",
                            model.baseFileName, viewGroupParam, attachToRootParam, componentParam)
                    addStatement("return $T.<$T>inflateInternal($N, $L, $N, $N, $N)",
                            viewDataBinding, binderTypeName,
                            inflaterParam, rLayoutFile, viewGroupParam, attachToRootParam,
                            componentParam)
                },

                methodSpec("inflate") {
                    addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                    addParameter(inflaterParam)
                    returns(binderTypeName)
                    addAnnotation(nonNull)
                    addStatement("return inflate($N, $T.getDefaultComponent())",
                            inflaterParam, dataBindingUtil)
                },

                methodSpec("inflate") {
                    addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                    addParameter(inflaterParam)
                    addParameter(componentParam)
                    returns(binderTypeName)
                    addAnnotation(nonNull)
                    addAnnotation(DEPRECATED)
                    addJavadoc(JAVADOC_BINDING_COMPONENT)
                    addJavadoc("\n@Deprecated Use DataBindingUtil.inflate(inflater, R.layout.$L, null, false, $N)\n",
                            model.baseFileName, componentParam)
                    addStatement("return $T.<$T>inflateInternal($N, $L, null, false, $N)",
                            viewDataBinding, binderTypeName, inflaterParam,
                            rLayoutFile, componentParam)
                },

                methodSpec("bind") {
                    addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                    addParameter(viewParam)
                    returns(binderTypeName)
                    addStatement("return bind($N, $T.getDefaultComponent())",
                            viewParam, dataBindingUtil)
                },
                methodSpec("bind") {
                    addModifiers(Modifier.PUBLIC, Modifier.STATIC)
                    addParameter(viewParam)
                    addParameter(componentParam)
                    returns(binderTypeName)
                    addAnnotation(DEPRECATED)
                    addJavadoc(JAVADOC_BINDING_COMPONENT)
                    addJavadoc("\n@Deprecated Use DataBindingUtil.bind($N, $N)\n",
                            viewParam, componentParam)
                    addStatement("return ($T)bind($N, $N, $L)",
                            binderTypeName, componentParam, viewParam, rLayoutFile)
                }
        )
    }

    private fun createGettersAndSetters(): List<MethodSpec> {
        return model.variables.flatMap { variable ->
            val typeName = variable.type.toTypeName(libTypes, model.importsByAlias)
            listOf(
                    methodSpec(model.setterName(variable)) {
                        addModifiers(Modifier.PUBLIC)
                        val param = parameterSpec(typeName, variable.name) {
                            if (!typeName.isPrimitive) {
                                addAnnotation(nullable)
                            }
                        }
                        returns(TypeName.VOID)
                        addParameter(param)
                        addModifiers(Modifier.ABSTRACT)
                        addModifiers(Modifier.PUBLIC)
                    },
                    methodSpec(model.getterName(variable)) {
                        addModifiers(Modifier.PUBLIC)
                        returns(typeName)
                        if (!typeName.isPrimitive) {
                            addAnnotation(nullable)
                        }
                        addStatement("return $L", model.fieldName(variable))
                    })
        }
    }

    private fun createConstructor() = constructorSpec {
        addModifiers(Modifier.PROTECTED)
        val componentParam = parameterSpec(dataBindingComponent, "_bindingComponent")
        val viewParam = parameterSpec(ANDROID_VIEW, "_root")
        val localFieldCountParam = parameterSpec(TypeName.INT, "_localFieldCount")
        addParameter(componentParam)
        addParameter(viewParam)
        addParameter(localFieldCountParam)
        // TODO de-dup construtor param names
        model.sortedTargets.filter { it.id != null }
                .forEach {
                    val fieldType = it.fieldType.toTypeName(libTypes, model.importsByAlias)
                    addParameter(parameterSpec(fieldType, model.fieldName(it)))
                }
        addStatement("super($N, $N, $N)", componentParam, viewParam, localFieldCountParam)
        model.sortedTargets.filter { it.id != null }.forEach {
            // todo might change if we start de-duping constructor params
            val fieldName = model.fieldName(it)
            addStatement("this.$1L = $1L", fieldName)
        }
    }

    fun generateClassInfo(): GenClassInfoLog.GenClass {
        return GenClassInfoLog.GenClass(
                qName = binderTypeName.toString(),
                modulePackage = model.modulePackage,
                variables = model.variables.associate {
                    Pair(it.name, it.type.toTypeName(libTypes, model.importsByAlias).toString())
                },
                implementations = model.generateImplInfo())
    }

    private fun createVariableFields(): List<FieldSpec> {
        return model.variables.map {
            fieldSpec(model.fieldName(it), it.type.toTypeName(libTypes, model.importsByAlias)) {
                addAnnotation(bindable) // mark them bindable to trigger BR gen
                addModifiers(Modifier.PROTECTED)
            }
        }
    }

    private fun createBindingTargetFields(): List<FieldSpec> {
        return model.sortedTargets
                .filter { it.id != null }
                .map {
                    val fieldType = it.fieldType.toTypeName(libTypes, model.importsByAlias)
                    fieldSpec(model.fieldName(it), fieldType) {
                        addModifiers(Modifier.FINAL)
                        addModifiers(if (it.id != null) Modifier.PUBLIC else Modifier.PRIVATE)
                        val (present, absent) = model.layoutConfigurationMembership(it)

                        if (absent.isNotEmpty()) {
                            addJavadoc(renderConfigurationJavadoc(present, absent))
                            addAnnotation(nullable)
                        } else {
                            addAnnotation(nonNull)
                        }
                    }
                }
    }
}
