/*
 * Decompiled with CFR 0.152.
 */
package androidx.input.motionprediction.kalman.matrix;

import androidx.annotation.NonNull;
import androidx.annotation.RestrictTo;
import java.util.Arrays;
import java.util.Locale;

@RestrictTo(value={RestrictTo.Scope.LIBRARY})
public class Matrix {
    private final int mRows;
    private final int mCols;
    private final double[] mMem;

    public Matrix(int rows, int cols) {
        this.mRows = rows;
        this.mCols = cols;
        this.mMem = new double[rows * cols];
    }

    public Matrix(int stride, @NonNull double[] values) {
        this.mRows = (values.length + stride - 1) / stride;
        this.mCols = stride;
        this.mMem = values;
        if (this.mMem.length != this.mRows * this.mCols) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "Invalid number of elements in 'values' Expected:%d Actual:%d", this.mMem.length, this.mRows & this.mCols));
        }
    }

    public Matrix(@NonNull Matrix src) {
        this.mRows = src.mRows;
        this.mCols = src.mCols;
        this.mMem = new double[this.mRows * this.mCols];
        System.arraycopy(src.mMem, 0, this.mMem, 0, this.mMem.length);
    }

    public int getNumRows() {
        return this.mRows;
    }

    public int getNumCols() {
        return this.mCols;
    }

    @NonNull
    public static Matrix identity(int width) {
        Matrix ret = new Matrix(width, width);
        Matrix.setIdentity(ret);
        return ret;
    }

    public static void setIdentity(@NonNull Matrix matrix) {
        Arrays.fill(matrix.mMem, 0.0);
        int width = matrix.mRows < matrix.mCols ? matrix.mRows : matrix.mCols;
        for (int i = 0; i < width; ++i) {
            matrix.put(i, i, 1.0);
        }
    }

    public double get(int i, int j) {
        if (0 > i || i >= this.mRows || 0 > j || j >= this.mCols) {
            throw new IndexOutOfBoundsException(String.format(Locale.ROOT, "Invalid matrix index value. i:%d j:%d not available in %s", i, j, this.shortString()));
        }
        return this.mMem[i * this.mCols + j];
    }

    public void put(int i, int j, double v) {
        if (0 > i || i >= this.mRows || 0 > j || j >= this.mCols) {
            throw new IndexOutOfBoundsException(String.format(Locale.ROOT, "Invalid matrix index value. i:%d j:%d not available in %s", i, j, this.shortString()));
        }
        this.mMem[i * this.mCols + j] = v;
    }

    public void fill(double value) {
        Arrays.fill(this.mMem, value);
    }

    public void scale(double alpha) {
        int size = this.mRows * this.mCols;
        int i = 0;
        while (i < size) {
            int n = i++;
            this.mMem[n] = this.mMem[n] * alpha;
        }
    }

    @NonNull
    public Matrix plus(@NonNull Matrix that) {
        if (this.mRows != that.mRows || this.mCols != that.mCols) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "The matrix dimensions are not the same. this:%s that:%s", this.shortString(), that.shortString()));
        }
        for (int i = 0; i < this.mMem.length; ++i) {
            this.mMem[i] = this.mMem[i] + that.mMem[i];
        }
        return this;
    }

    @NonNull
    public Matrix minus(@NonNull Matrix that) {
        if (this.mRows != that.mRows || this.mCols != that.mCols) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "The matrix dimensions are not the same. this:%s that:%s", this.shortString(), that.shortString()));
        }
        for (int i = 0; i < this.mMem.length; ++i) {
            this.mMem[i] = this.mMem[i] - that.mMem[i];
        }
        return this;
    }

    @NonNull
    public Matrix dot(@NonNull Matrix that, @NonNull Matrix result) {
        if (this.mRows != result.mRows || this.mCols != that.mRows || that.mCols != result.mCols) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "The matrices dimensions are not conformant for a dot matrix operation. this:%s that:%s result:%s", this.shortString(), that.shortString(), result.shortString()));
        }
        for (int i = 0; i < this.mRows; ++i) {
            for (int j = 0; j < that.mCols; ++j) {
                double s = 0.0;
                for (int k = 0; k < this.mCols; ++k) {
                    s += this.get(i, k) * that.get(k, j);
                }
                result.put(i, j, s);
            }
        }
        return result;
    }

    @NonNull
    public Matrix inverse(@NonNull Matrix scratch) {
        int j;
        int i;
        if (this.mRows != this.mCols) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "The matrix is not square. this:%s", this.shortString()));
        }
        if (scratch.mRows != this.mRows || scratch.mCols != 2 * this.mCols) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "The scratch matrix size is not correct. this:%s", scratch.shortString()));
        }
        for (i = 0; i < this.mRows; ++i) {
            for (j = 0; j < this.mCols; ++j) {
                scratch.put(i, j, this.get(i, j));
                scratch.put(i, this.mCols + j, i == j ? 1.0 : 0.0);
            }
        }
        for (i = 0; i < this.mRows; ++i) {
            double d;
            int ibest = i;
            double vbest = Math.abs(scratch.get(ibest, ibest));
            for (int ii = i + 1; ii < this.mRows; ++ii) {
                double v = Math.abs(scratch.get(ii, i));
                if (!(v > vbest)) continue;
                ibest = ii;
                vbest = v;
            }
            if (ibest != i) {
                for (int j2 = 0; j2 < scratch.mCols; ++j2) {
                    double t = scratch.get(i, j2);
                    scratch.put(i, j2, scratch.get(ibest, j2));
                    scratch.put(ibest, j2, t);
                }
            }
            if ((d = scratch.get(i, i)) == 0.0) {
                throw new ArithmeticException("Singular matrix");
            }
            for (int j3 = 0; j3 < scratch.mCols; ++j3) {
                scratch.put(i, j3, scratch.get(i, j3) / d);
            }
            for (int ii = i + 1; ii < this.mRows; ++ii) {
                d = scratch.get(ii, i);
                for (int j4 = 0; j4 < scratch.mCols; ++j4) {
                    scratch.put(ii, j4, scratch.get(ii, j4) - d * scratch.get(i, j4));
                }
            }
        }
        for (i = this.mRows - 1; i >= 0; --i) {
            for (int ii = 0; ii < i; ++ii) {
                double d = scratch.get(ii, i);
                for (int j5 = 0; j5 < scratch.mCols; ++j5) {
                    scratch.put(ii, j5, scratch.get(ii, j5) - d * scratch.get(i, j5));
                }
            }
        }
        for (i = 0; i < this.mRows; ++i) {
            for (j = 0; j < this.mCols; ++j) {
                this.put(i, j, scratch.get(i, this.mCols + j));
            }
        }
        return this;
    }

    @NonNull
    public Matrix dotTranspose(@NonNull Matrix that, @NonNull Matrix result) {
        if (this.mRows != result.mRows || this.mCols != that.mCols || that.mRows != result.mCols) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "The matrices dimensions are not conformant for a transpose operation. this:%s that:%s result:%s", this.shortString(), that.shortString(), result.shortString()));
        }
        for (int i = 0; i < this.mRows; ++i) {
            for (int j = 0; j < that.mRows; ++j) {
                double s = 0.0;
                for (int k = 0; k < this.mCols; ++k) {
                    s += this.get(i, k) * that.get(j, k);
                }
                result.put(i, j, s);
            }
        }
        return result;
    }

    public boolean equals(Object that) {
        if (this == that) {
            return true;
        }
        if (!(that instanceof Matrix)) {
            return false;
        }
        Matrix other = (Matrix)that;
        if (this.mRows != other.mRows) {
            return false;
        }
        if (this.mCols != other.mCols) {
            return false;
        }
        for (int i = 0; i < this.mMem.length; ++i) {
            if (this.mMem[i] == other.mMem[i]) continue;
            return false;
        }
        return true;
    }

    public int hashCode() {
        int h = this.mRows * 101 + this.mCols;
        for (double m : this.mMem) {
            h = h * 37 + Double.hashCode(m);
        }
        return h;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder(this.mRows * this.mCols * 8);
        sb.append(this.mRows).append("x").append(this.mCols).append(" [");
        for (int i = 0; i < this.mMem.length; ++i) {
            if (i > 0) {
                sb.append(i % this.mCols == 0 ? "; " : ", ");
            }
            sb.append(this.mMem[i]);
        }
        sb.append("]");
        return sb.toString();
    }

    private String shortString() {
        return "(" + this.mRows + "x" + this.mCols + ")";
    }
}

