/*
 * Decompiled with CFR 0.152.
 */
package androidx.media3.effect;

import android.graphics.SurfaceTexture;
import android.view.Surface;
import androidx.annotation.Nullable;
import androidx.media3.common.FrameInfo;
import androidx.media3.common.GlObjectsProvider;
import androidx.media3.common.GlTextureInfo;
import androidx.media3.common.VideoFrameProcessingException;
import androidx.media3.common.util.Assertions;
import androidx.media3.common.util.GlUtil;
import androidx.media3.common.util.Log;
import androidx.media3.common.util.Util;
import androidx.media3.effect.DebugTraceUtil;
import androidx.media3.effect.ExternalShaderProgram;
import androidx.media3.effect.TextureManager;
import androidx.media3.effect.VideoFrameProcessingTaskExecutor;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

final class ExternalTextureManager
implements TextureManager {
    private static final String TAG = "ExtTexMgr";
    private static final String TIMER_THREAD_NAME = "ExtTexMgr:Timer";
    private static final long SURFACE_TEXTURE_TIMEOUT_MS = Util.DEVICE.contains("emulator") ? 10000L : 500L;
    private final GlObjectsProvider glObjectsProvider;
    private final VideoFrameProcessingTaskExecutor videoFrameProcessingTaskExecutor;
    private final ExternalShaderProgram externalShaderProgram;
    private final int externalTexId;
    private final Surface surface;
    private final SurfaceTexture surfaceTexture;
    private final float[] textureTransformMatrix;
    private final Queue<FrameInfo> pendingFrames;
    private final ScheduledExecutorService forceEndOfStreamExecutorService;
    private final AtomicInteger externalShaderProgramInputCapacity;
    private int numberOfFramesToDropOnBecomingAvailable;
    private int availableFrameCount;
    private boolean currentInputStreamEnded;
    @Nullable
    private volatile FrameInfo currentFrame;
    @Nullable
    private volatile VideoFrameProcessingTaskExecutor.Task onFlushCompleteTask;
    @Nullable
    private Future<?> forceSignalEndOfStreamFuture;
    private boolean shouldRejectIncomingFrames;

    public ExternalTextureManager(GlObjectsProvider glObjectsProvider, ExternalShaderProgram externalShaderProgram, VideoFrameProcessingTaskExecutor videoFrameProcessingTaskExecutor) throws VideoFrameProcessingException {
        this.glObjectsProvider = glObjectsProvider;
        this.externalShaderProgram = externalShaderProgram;
        this.videoFrameProcessingTaskExecutor = videoFrameProcessingTaskExecutor;
        try {
            this.externalTexId = GlUtil.createExternalTexture();
        }
        catch (GlUtil.GlException e) {
            throw new VideoFrameProcessingException((Throwable)e);
        }
        this.surfaceTexture = new SurfaceTexture(this.externalTexId);
        this.textureTransformMatrix = new float[16];
        this.pendingFrames = new ConcurrentLinkedQueue<FrameInfo>();
        this.forceEndOfStreamExecutorService = Util.newSingleThreadScheduledExecutor((String)TIMER_THREAD_NAME);
        this.externalShaderProgramInputCapacity = new AtomicInteger();
        this.surfaceTexture.setOnFrameAvailableListener(unused -> videoFrameProcessingTaskExecutor.submit(() -> {
            DebugTraceUtil.logEvent("VFP-SurfaceTextureInput", -9223372036854775807L);
            if (this.numberOfFramesToDropOnBecomingAvailable > 0) {
                --this.numberOfFramesToDropOnBecomingAvailable;
                this.surfaceTexture.updateTexImage();
                this.maybeExecuteAfterFlushTask();
            } else if (this.shouldRejectIncomingFrames) {
                this.surfaceTexture.updateTexImage();
                Log.w((String)TAG, (String)("Dropping frame received on SurfaceTexture after forcing EOS: " + this.surfaceTexture.getTimestamp() / 1000L));
            } else {
                if (this.currentInputStreamEnded) {
                    this.restartForceSignalEndOfStreamTimer();
                }
                ++this.availableFrameCount;
                this.maybeQueueFrameToExternalShaderProgram();
            }
        }));
        this.surface = new Surface(this.surfaceTexture);
    }

    @Override
    public void setDefaultBufferSize(int width, int height) {
        this.surfaceTexture.setDefaultBufferSize(width, height);
    }

    @Override
    public Surface getInputSurface() {
        return this.surface;
    }

    @Override
    public void onReadyToAcceptInputFrame() {
        this.videoFrameProcessingTaskExecutor.submit(() -> {
            this.externalShaderProgramInputCapacity.incrementAndGet();
            this.maybeQueueFrameToExternalShaderProgram();
        });
    }

    @Override
    public void onInputFrameProcessed(GlTextureInfo inputTexture) {
        this.videoFrameProcessingTaskExecutor.submit(() -> {
            this.currentFrame = null;
            if (this.currentInputStreamEnded && this.pendingFrames.isEmpty()) {
                this.currentInputStreamEnded = false;
                this.externalShaderProgram.signalEndOfCurrentInputStream();
                DebugTraceUtil.logEvent("ExternalTextureManager-SignalEOS", Long.MIN_VALUE);
                this.cancelForceSignalEndOfStreamTimer();
            } else {
                this.maybeQueueFrameToExternalShaderProgram();
            }
        });
    }

    @Override
    public void setOnFlushCompleteListener(@Nullable VideoFrameProcessingTaskExecutor.Task task) {
        this.onFlushCompleteTask = task;
    }

    @Override
    public void onFlush() {
        this.videoFrameProcessingTaskExecutor.submit(this::flush);
    }

    @Override
    public void registerInputFrame(FrameInfo frame) {
        this.pendingFrames.add(frame);
        this.videoFrameProcessingTaskExecutor.submit(() -> {
            this.shouldRejectIncomingFrames = false;
        });
    }

    @Override
    public int getPendingFrameCount() {
        return this.pendingFrames.size();
    }

    @Override
    public void signalEndOfCurrentInputStream() {
        this.videoFrameProcessingTaskExecutor.submit(() -> {
            if (this.pendingFrames.isEmpty() && this.currentFrame == null) {
                this.externalShaderProgram.signalEndOfCurrentInputStream();
                DebugTraceUtil.logEvent("ExternalTextureManager-SignalEOS", Long.MIN_VALUE);
                this.cancelForceSignalEndOfStreamTimer();
            } else {
                this.currentInputStreamEnded = true;
                this.restartForceSignalEndOfStreamTimer();
            }
        });
    }

    @Override
    public void release() {
        this.surfaceTexture.release();
        this.surface.release();
        this.forceEndOfStreamExecutorService.shutdownNow();
    }

    private void maybeExecuteAfterFlushTask() {
        if (this.onFlushCompleteTask == null || this.numberOfFramesToDropOnBecomingAvailable > 0) {
            return;
        }
        this.videoFrameProcessingTaskExecutor.submitWithHighPriority(this.onFlushCompleteTask);
    }

    private void restartForceSignalEndOfStreamTimer() {
        this.cancelForceSignalEndOfStreamTimer();
        this.forceSignalEndOfStreamFuture = this.forceEndOfStreamExecutorService.schedule(() -> this.videoFrameProcessingTaskExecutor.submit(this::forceSignalEndOfStream), SURFACE_TEXTURE_TIMEOUT_MS, TimeUnit.MILLISECONDS);
    }

    private void cancelForceSignalEndOfStreamTimer() {
        if (this.forceSignalEndOfStreamFuture != null) {
            this.forceSignalEndOfStreamFuture.cancel(false);
        }
        this.forceSignalEndOfStreamFuture = null;
    }

    private void forceSignalEndOfStream() {
        Log.w((String)TAG, (String)Util.formatInvariant((String)"Forcing EOS after missing %d frames for %d ms, with available frame count: %d", (Object[])new Object[]{this.pendingFrames.size(), SURFACE_TEXTURE_TIMEOUT_MS, this.availableFrameCount}));
        this.currentInputStreamEnded = false;
        this.currentFrame = null;
        this.pendingFrames.clear();
        this.shouldRejectIncomingFrames = true;
        this.removeAllSurfaceTextureFrames();
        this.signalEndOfCurrentInputStream();
    }

    private void flush() {
        this.numberOfFramesToDropOnBecomingAvailable = this.pendingFrames.size() - this.availableFrameCount;
        this.removeAllSurfaceTextureFrames();
        this.externalShaderProgramInputCapacity.set(0);
        this.currentFrame = null;
        this.pendingFrames.clear();
        this.maybeExecuteAfterFlushTask();
    }

    private void removeAllSurfaceTextureFrames() {
        while (this.availableFrameCount > 0) {
            --this.availableFrameCount;
            this.surfaceTexture.updateTexImage();
        }
    }

    private void maybeQueueFrameToExternalShaderProgram() {
        if (this.externalShaderProgramInputCapacity.get() == 0 || this.availableFrameCount == 0 || this.currentFrame != null) {
            return;
        }
        this.surfaceTexture.updateTexImage();
        --this.availableFrameCount;
        this.currentFrame = this.pendingFrames.peek();
        FrameInfo currentFrame = (FrameInfo)Assertions.checkStateNotNull((Object)this.currentFrame);
        this.externalShaderProgramInputCapacity.decrementAndGet();
        this.surfaceTexture.getTransformMatrix(this.textureTransformMatrix);
        this.externalShaderProgram.setTextureTransformMatrix(this.textureTransformMatrix);
        long frameTimeNs = this.surfaceTexture.getTimestamp();
        long offsetToAddUs = currentFrame.offsetToAddUs;
        long presentationTimeUs = frameTimeNs / 1000L + offsetToAddUs;
        this.externalShaderProgram.queueInputFrame(this.glObjectsProvider, new GlTextureInfo(this.externalTexId, -1, -1, currentFrame.width, currentFrame.height), presentationTimeUs);
        Assertions.checkStateNotNull((Object)this.pendingFrames.remove());
        DebugTraceUtil.logEvent("VFP-QueueFrame", presentationTimeUs);
    }
}

