/*
 * Decompiled with CFR 0.152.
 */
package androidx.paging;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.List;

final class PagedStorage<T>
extends AbstractList<T> {
    private static final List PLACEHOLDER_LIST = new ArrayList();
    private int mLeadingNullCount;
    private final ArrayList<List<T>> mPages;
    private int mTrailingNullCount;
    private int mPositionOffset;
    private int mLoadedCount;
    private int mStorageCount;
    private int mPageSize;
    private int mNumberPrepended;
    private int mNumberAppended;

    PagedStorage() {
        this.mLeadingNullCount = 0;
        this.mPages = new ArrayList();
        this.mTrailingNullCount = 0;
        this.mPositionOffset = 0;
        this.mLoadedCount = 0;
        this.mStorageCount = 0;
        this.mPageSize = 1;
        this.mNumberPrepended = 0;
        this.mNumberAppended = 0;
    }

    PagedStorage(int leadingNulls, List<T> page, int trailingNulls) {
        this();
        super.init(leadingNulls, page, trailingNulls, 0);
    }

    private PagedStorage(PagedStorage<T> other) {
        this.mLeadingNullCount = other.mLeadingNullCount;
        this.mPages = new ArrayList<List<T>>(other.mPages);
        this.mTrailingNullCount = other.mTrailingNullCount;
        this.mPositionOffset = other.mPositionOffset;
        this.mLoadedCount = other.mLoadedCount;
        this.mStorageCount = other.mStorageCount;
        this.mPageSize = other.mPageSize;
        this.mNumberPrepended = other.mNumberPrepended;
        this.mNumberAppended = other.mNumberAppended;
    }

    PagedStorage<T> snapshot() {
        return new PagedStorage<T>(this);
    }

    private void init(int leadingNulls, List<T> page, int trailingNulls, int positionOffset) {
        this.mLeadingNullCount = leadingNulls;
        this.mPages.clear();
        this.mPages.add(page);
        this.mTrailingNullCount = trailingNulls;
        this.mPositionOffset = positionOffset;
        this.mStorageCount = this.mLoadedCount = page.size();
        this.mPageSize = page.size();
        this.mNumberPrepended = 0;
        this.mNumberAppended = 0;
    }

    void init(int leadingNulls, @NonNull List<T> page, int trailingNulls, int positionOffset, @NonNull Callback callback) {
        this.init(leadingNulls, page, trailingNulls, positionOffset);
        callback.onInitialized(this.size());
    }

    @Override
    public T get(int i) {
        int pageInternalIndex;
        int localPageIndex;
        if (i < 0 || i >= this.size()) {
            throw new IndexOutOfBoundsException("Index: " + i + ", Size: " + this.size());
        }
        int localIndex = i - this.mLeadingNullCount;
        if (localIndex < 0 || localIndex >= this.mStorageCount) {
            return null;
        }
        if (this.isTiled()) {
            localPageIndex = localIndex / this.mPageSize;
            pageInternalIndex = localIndex % this.mPageSize;
        } else {
            int pageSize;
            pageInternalIndex = localIndex;
            int localPageCount = this.mPages.size();
            for (localPageIndex = 0; localPageIndex < localPageCount && (pageSize = this.mPages.get(localPageIndex).size()) <= pageInternalIndex; pageInternalIndex -= pageSize, ++localPageIndex) {
            }
        }
        List<T> page = this.mPages.get(localPageIndex);
        if (page == null || page.size() == 0) {
            return null;
        }
        return page.get(pageInternalIndex);
    }

    boolean isTiled() {
        return this.mPageSize > 0;
    }

    int getLeadingNullCount() {
        return this.mLeadingNullCount;
    }

    int getTrailingNullCount() {
        return this.mTrailingNullCount;
    }

    int getStorageCount() {
        return this.mStorageCount;
    }

    int getNumberAppended() {
        return this.mNumberAppended;
    }

    int getNumberPrepended() {
        return this.mNumberPrepended;
    }

    int getPageCount() {
        return this.mPages.size();
    }

    int getLoadedCount() {
        return this.mLoadedCount;
    }

    int getPositionOffset() {
        return this.mPositionOffset;
    }

    int getMiddleOfLoadedRange() {
        return this.mLeadingNullCount + this.mPositionOffset + this.mStorageCount / 2;
    }

    @Override
    public int size() {
        return this.mLeadingNullCount + this.mStorageCount + this.mTrailingNullCount;
    }

    int computeLeadingNulls() {
        List<T> page;
        int total = this.mLeadingNullCount;
        int pageCount = this.mPages.size();
        for (int i = 0; i < pageCount && ((page = this.mPages.get(i)) == null || page == PLACEHOLDER_LIST); ++i) {
            total += this.mPageSize;
        }
        return total;
    }

    int computeTrailingNulls() {
        List<T> page;
        int total = this.mTrailingNullCount;
        for (int i = this.mPages.size() - 1; i >= 0 && ((page = this.mPages.get(i)) == null || page == PLACEHOLDER_LIST); --i) {
            total += this.mPageSize;
        }
        return total;
    }

    private boolean needsTrim(int maxSize, int requiredRemaining, int localPageIndex) {
        List<T> page = this.mPages.get(localPageIndex);
        return page == null || this.mLoadedCount > maxSize && this.mPages.size() > 2 && page != PLACEHOLDER_LIST && this.mLoadedCount - page.size() >= requiredRemaining;
    }

    boolean needsTrimFromFront(int maxSize, int requiredRemaining) {
        return this.needsTrim(maxSize, requiredRemaining, 0);
    }

    boolean needsTrimFromEnd(int maxSize, int requiredRemaining) {
        return this.needsTrim(maxSize, requiredRemaining, this.mPages.size() - 1);
    }

    boolean shouldPreTrimNewPage(int maxSize, int requiredRemaining, int countToBeAdded) {
        return this.mLoadedCount + countToBeAdded > maxSize && this.mPages.size() > 1 && this.mLoadedCount >= requiredRemaining;
    }

    boolean trimFromFront(boolean insertNulls, int maxSize, int requiredRemaining, @NonNull Callback callback) {
        int totalRemoved = 0;
        while (this.needsTrimFromFront(maxSize, requiredRemaining)) {
            List<T> page = this.mPages.remove(0);
            int removed = page == null ? this.mPageSize : page.size();
            totalRemoved += removed;
            this.mStorageCount -= removed;
            this.mLoadedCount -= page == null ? 0 : page.size();
        }
        if (totalRemoved > 0) {
            if (insertNulls) {
                int previousLeadingNulls = this.mLeadingNullCount;
                this.mLeadingNullCount += totalRemoved;
                callback.onPagesSwappedToPlaceholder(previousLeadingNulls, totalRemoved);
            } else {
                this.mPositionOffset += totalRemoved;
                callback.onPagesRemoved(this.mLeadingNullCount, totalRemoved);
            }
        }
        return totalRemoved > 0;
    }

    boolean trimFromEnd(boolean insertNulls, int maxSize, int requiredRemaining, @NonNull Callback callback) {
        int totalRemoved = 0;
        while (this.needsTrimFromEnd(maxSize, requiredRemaining)) {
            List<T> page = this.mPages.remove(this.mPages.size() - 1);
            int removed = page == null ? this.mPageSize : page.size();
            totalRemoved += removed;
            this.mStorageCount -= removed;
            this.mLoadedCount -= page == null ? 0 : page.size();
        }
        if (totalRemoved > 0) {
            int newEndPosition = this.mLeadingNullCount + this.mStorageCount;
            if (insertNulls) {
                this.mTrailingNullCount += totalRemoved;
                callback.onPagesSwappedToPlaceholder(newEndPosition, totalRemoved);
            } else {
                callback.onPagesRemoved(newEndPosition, totalRemoved);
            }
        }
        return totalRemoved > 0;
    }

    T getFirstLoadedItem() {
        return this.mPages.get(0).get(0);
    }

    T getLastLoadedItem() {
        List<T> page = this.mPages.get(this.mPages.size() - 1);
        return page.get(page.size() - 1);
    }

    void prependPage(@NonNull List<T> page, @NonNull Callback callback) {
        int count = page.size();
        if (count == 0) {
            callback.onEmptyPrepend();
            return;
        }
        if (this.mPageSize > 0 && count != this.mPageSize) {
            this.mPageSize = this.mPages.size() == 1 && count > this.mPageSize ? count : -1;
        }
        this.mPages.add(0, page);
        this.mLoadedCount += count;
        this.mStorageCount += count;
        int changedCount = Math.min(this.mLeadingNullCount, count);
        int addedCount = count - changedCount;
        if (changedCount != 0) {
            this.mLeadingNullCount -= changedCount;
        }
        this.mPositionOffset -= addedCount;
        this.mNumberPrepended += count;
        callback.onPagePrepended(this.mLeadingNullCount, changedCount, addedCount);
    }

    void appendPage(@NonNull List<T> page, @NonNull Callback callback) {
        int count = page.size();
        if (count == 0) {
            callback.onEmptyAppend();
            return;
        }
        if (this.mPageSize > 0 && (this.mPages.get(this.mPages.size() - 1).size() != this.mPageSize || count > this.mPageSize)) {
            this.mPageSize = -1;
        }
        this.mPages.add(page);
        this.mLoadedCount += count;
        this.mStorageCount += count;
        int changedCount = Math.min(this.mTrailingNullCount, count);
        int addedCount = count - changedCount;
        if (changedCount != 0) {
            this.mTrailingNullCount -= changedCount;
        }
        this.mNumberAppended += count;
        callback.onPageAppended(this.mLeadingNullCount + this.mStorageCount - count, changedCount, addedCount);
    }

    boolean pageWouldBeBoundary(int positionOfPage, boolean trimFromFront) {
        if (this.mPageSize < 1 || this.mPages.size() < 2) {
            throw new IllegalStateException("Trimming attempt before sufficient load");
        }
        if (positionOfPage < this.mLeadingNullCount) {
            return trimFromFront;
        }
        if (positionOfPage >= this.mLeadingNullCount + this.mStorageCount) {
            return !trimFromFront;
        }
        int localPageIndex = (positionOfPage - this.mLeadingNullCount) / this.mPageSize;
        if (trimFromFront) {
            for (int i = 0; i < localPageIndex; ++i) {
                if (this.mPages.get(i) == null) continue;
                return false;
            }
        } else {
            for (int i = this.mPages.size() - 1; i > localPageIndex; --i) {
                if (this.mPages.get(i) == null) continue;
                return false;
            }
        }
        return true;
    }

    void initAndSplit(int leadingNulls, @NonNull List<T> multiPageList, int trailingNulls, int positionOffset, int pageSize, @NonNull Callback callback) {
        int pageCount = (multiPageList.size() + (pageSize - 1)) / pageSize;
        for (int i = 0; i < pageCount; ++i) {
            int beginInclusive = i * pageSize;
            int endExclusive = Math.min(multiPageList.size(), (i + 1) * pageSize);
            List<T> sublist = multiPageList.subList(beginInclusive, endExclusive);
            if (i == 0) {
                int initialTrailingNulls = trailingNulls + multiPageList.size() - sublist.size();
                this.init(leadingNulls, sublist, initialTrailingNulls, positionOffset);
                continue;
            }
            int insertPosition = leadingNulls + beginInclusive;
            this.insertPage(insertPosition, sublist, null);
        }
        callback.onInitialized(this.size());
    }

    void tryInsertPageAndTrim(int position, @NonNull List<T> page, int lastLoad, int maxSize, int requiredRemaining, @NonNull Callback callback) {
        boolean pageInserted;
        boolean trim = maxSize != Integer.MAX_VALUE;
        boolean trimFromFront = lastLoad > this.getMiddleOfLoadedRange();
        boolean bl = pageInserted = !trim || !this.shouldPreTrimNewPage(maxSize, requiredRemaining, page.size()) || !this.pageWouldBeBoundary(position, trimFromFront);
        if (pageInserted) {
            this.insertPage(position, page, callback);
        } else {
            int localPageIndex = (position - this.mLeadingNullCount) / this.mPageSize;
            this.mPages.set(localPageIndex, null);
            this.mStorageCount -= page.size();
            if (trimFromFront) {
                this.mPages.remove(0);
                this.mLeadingNullCount += page.size();
            } else {
                this.mPages.remove(this.mPages.size() - 1);
                this.mTrailingNullCount += page.size();
            }
        }
        if (trim) {
            if (trimFromFront) {
                this.trimFromFront(true, maxSize, requiredRemaining, callback);
            } else {
                this.trimFromEnd(true, maxSize, requiredRemaining, callback);
            }
        }
    }

    public void insertPage(int position, @NonNull List<T> page, @Nullable Callback callback) {
        int newPageSize = page.size();
        if (newPageSize != this.mPageSize) {
            boolean onlyEndPagePresent;
            int size = this.size();
            boolean addingLastPage = position == size - size % this.mPageSize && newPageSize < this.mPageSize;
            boolean bl = onlyEndPagePresent = this.mTrailingNullCount == 0 && this.mPages.size() == 1 && newPageSize > this.mPageSize;
            if (!onlyEndPagePresent && !addingLastPage) {
                throw new IllegalArgumentException("page introduces incorrect tiling");
            }
            if (onlyEndPagePresent) {
                this.mPageSize = newPageSize;
            }
        }
        int pageIndex = position / this.mPageSize;
        this.allocatePageRange(pageIndex, pageIndex);
        int localPageIndex = pageIndex - this.mLeadingNullCount / this.mPageSize;
        List<T> oldPage = this.mPages.get(localPageIndex);
        if (oldPage != null && oldPage != PLACEHOLDER_LIST) {
            throw new IllegalArgumentException("Invalid position " + position + ": data already loaded");
        }
        this.mPages.set(localPageIndex, page);
        this.mLoadedCount += newPageSize;
        if (callback != null) {
            callback.onPageInserted(position, newPageSize);
        }
    }

    void allocatePageRange(int minimumPage, int maximumPage) {
        int newStorageAllocated;
        int leadingNullPages = this.mLeadingNullCount / this.mPageSize;
        if (minimumPage < leadingNullPages) {
            for (int i = 0; i < leadingNullPages - minimumPage; ++i) {
                this.mPages.add(0, null);
            }
            newStorageAllocated = (leadingNullPages - minimumPage) * this.mPageSize;
            this.mStorageCount += newStorageAllocated;
            this.mLeadingNullCount -= newStorageAllocated;
            leadingNullPages = minimumPage;
        }
        if (maximumPage >= leadingNullPages + this.mPages.size()) {
            newStorageAllocated = Math.min(this.mTrailingNullCount, (maximumPage + 1 - (leadingNullPages + this.mPages.size())) * this.mPageSize);
            for (int i = this.mPages.size(); i <= maximumPage - leadingNullPages; ++i) {
                this.mPages.add(this.mPages.size(), null);
            }
            this.mStorageCount += newStorageAllocated;
            this.mTrailingNullCount -= newStorageAllocated;
        }
    }

    public void allocatePlaceholders(int index, int prefetchDistance, int pageSize, Callback callback) {
        if (pageSize != this.mPageSize) {
            if (pageSize < this.mPageSize) {
                throw new IllegalArgumentException("Page size cannot be reduced");
            }
            if (this.mPages.size() != 1 || this.mTrailingNullCount != 0) {
                throw new IllegalArgumentException("Page size can change only if last page is only one present");
            }
            this.mPageSize = pageSize;
        }
        int maxPageCount = (this.size() + this.mPageSize - 1) / this.mPageSize;
        int minimumPage = Math.max((index - prefetchDistance) / this.mPageSize, 0);
        int maximumPage = Math.min((index + prefetchDistance) / this.mPageSize, maxPageCount - 1);
        this.allocatePageRange(minimumPage, maximumPage);
        int leadingNullPages = this.mLeadingNullCount / this.mPageSize;
        for (int pageIndex = minimumPage; pageIndex <= maximumPage; ++pageIndex) {
            int localPageIndex = pageIndex - leadingNullPages;
            if (this.mPages.get(localPageIndex) != null) continue;
            this.mPages.set(localPageIndex, PLACEHOLDER_LIST);
            callback.onPagePlaceholderInserted(pageIndex);
        }
    }

    public boolean hasPage(int pageSize, int index) {
        int leadingNullPages = this.mLeadingNullCount / pageSize;
        if (index < leadingNullPages || index >= leadingNullPages + this.mPages.size()) {
            return false;
        }
        List<T> page = this.mPages.get(index - leadingNullPages);
        return page != null && page != PLACEHOLDER_LIST;
    }

    @Override
    public String toString() {
        StringBuilder ret = new StringBuilder("leading " + this.mLeadingNullCount + ", storage " + this.mStorageCount + ", trailing " + this.getTrailingNullCount());
        for (int i = 0; i < this.mPages.size(); ++i) {
            ret.append(" ").append(this.mPages.get(i));
        }
        return ret.toString();
    }

    static interface Callback {
        public void onInitialized(int var1);

        public void onPagePrepended(int var1, int var2, int var3);

        public void onPageAppended(int var1, int var2, int var3);

        public void onPagePlaceholderInserted(int var1);

        public void onPageInserted(int var1, int var2);

        public void onPagesRemoved(int var1, int var2);

        public void onPagesSwappedToPlaceholder(int var1, int var2);

        public void onEmptyPrepend();

        public void onEmptyAppend();
    }
}

