/*
 * Copyright 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.ui.layout

import androidx.annotation.FloatRange
import androidx.compose.Composable
import androidx.ui.core.Alignment
import androidx.ui.core.HorizontalAlignmentLine
import androidx.ui.core.Measured
import androidx.ui.core.Modifier
import androidx.ui.unit.IntPx

/**
 * A layout composable that places its children in a horizontal sequence. For a layout composable
 * that places its children in a vertical sequence, see [Column].
 *
 * The layout model is able to assign children widths according to their weights provided
 * using the [RowScope.weight] modifier. If a child is not provided a weight, it will be
 * asked for its preferred width before the sizes of the children with weights are calculated
 * proportionally to their weight based on the remaining available space.
 *
 * When none of its children have weights, a [Row] will be as small as possible to fit its
 * children one next to the other. In order to change the size of the [Row], use the
 * [LayoutWidth] modifiers; e.g. to make it fill the available width [LayoutWidth.Fill] can be used.
 * If at least one child of a [Row] has a [weight][RowScope.weight], the [Row] will
 * fill the available space, so there is no need for [LayoutWidth.Fill]. However, if [Row]'s
 * size should be limited, the [LayoutWidth] or [LayoutWidth.Max] layout modifiers should be
 * applied.
 *
 * When the size of the [Row] is larger than the sum of its children sizes, a
 * [horizontalArrangement] can be specified to define the positioning of the children inside
 * the [Row]. See [Arrangement] for available positioning behaviors; a custom arrangement can
 * also be defined using the constructor of [Arrangement].
 *
 * Example usage:
 *
 * @sample androidx.ui.layout.samples.SimpleRow
 *
 * @param modifier The modifier to be applied to the Row.
 * @param horizontalArrangement The horizontal arrangement of the layout's children.
 * @param verticalGravity The vertical gravity of the layout's children.
 *
 * @see Column
 */
@Composable
fun Row(
    modifier: Modifier = Modifier,
    horizontalArrangement: Arrangement.Horizontal = Arrangement.Start,
    verticalGravity: Alignment.Vertical = Alignment.Top,
    children: @Composable() RowScope.() -> Unit
) {
    RowColumnImpl(
        orientation = LayoutOrientation.Horizontal,
        modifier = modifier,
        arrangement = horizontalArrangement,
        crossAxisAlignment = verticalGravity,
        crossAxisSize = SizeMode.Wrap,
        children = { RowScope.children() }
    )
}

@Deprecated(
    "RowAlign is deprecated. Please use Alignment instead.",
    ReplaceWith("Alignment", "androidx.ui.core.Alignment")
)
enum class RowAlign {
    Top,
    Center,
    Bottom
}

/**
 * Scope for the children of [Row].
 */
@LayoutScopeMarker
object RowScope {
    /**
     * Position the element vertically within the [Row] according to [align].
     */
    fun Modifier.gravity(align: Alignment.Vertical) = this + GravityModifier(align)

    @Deprecated(
        "gravity(RowAlign) is deprecated. Please use gravity instead.",
        ReplaceWith("gravity(align)")
    )
    @Suppress("Deprecation")
    fun Modifier.gravity(align: RowAlign) = this + when (align) {
        RowAlign.Top -> GravityModifier(Alignment.Top)
        RowAlign.Center -> GravityModifier(Alignment.CenterVertically)
        RowAlign.Bottom -> GravityModifier(Alignment.Bottom)
    }

    /**
     * Position the element vertically such that its [alignmentLine] aligns with sibling elements
     * also configured to [alignWithSiblings] with the same [alignmentLine].
     *
     * Example usage:
     * @sample androidx.ui.layout.samples.SimpleRelativeToSiblingsInRow
     */
    fun Modifier.alignWithSiblings(alignmentLine: HorizontalAlignmentLine) =
        this + SiblingsAlignedModifier.WithAlignmentLine(alignmentLine)

    /**
     * Size the element's width proportional to its [weight] relative to other weighted sibling
     * elements in the [Row]. The parent will divide the horizontal space remaining after measuring
     * unweighted child elements and distribute it according to this weight.
     */
    fun Modifier.weight(
        @FloatRange(from = 0.0, fromInclusive = false) weight: Float,
        fill: Boolean = true
    ): Modifier {
        require(weight > 0.0) { "invalid weight $weight; must be greater than zero" }
        return this + LayoutWeightImpl(weight, fill)
    }

    /**
     * Position the element vertically such that the alignment line for the content as
     * determined by [alignmentLineBlock] aligns with sibling elements also configured to
     * [alignWithSiblings] with an [alignmentLineBlock].
     *
     * Example usage:
     * @sample androidx.ui.layout.samples.SimpleRelativeToSiblings
     */
    fun Modifier.alignWithSiblings(
        alignmentLineBlock: (Measured) -> IntPx
    ) = this + SiblingsAlignedModifier.WithAlignmentLineBlock(alignmentLineBlock)
}
