/*
 * Copyright 2019 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package androidx.ui.text.style

import androidx.compose.Immutable
import androidx.ui.util.lerp

/**
 * Define a geometric transformation on text.
 *
 * @param scaleX The scale of the text on the horizontal direction. The default value is 1.0f, i.e
 * no scaling.
 * @param skewX The shear of the text on the horizontal direction. A pixel at (x, y), where y is
 * the distance above baseline, will be transformed to (x + y * skewX, y). The default value is
 * 0.0f i.e. no skewing.
 */
@Immutable
data class TextGeometricTransform(
    val scaleX: Float = 1.0f,
    val skewX: Float = 0f
) {
    companion object {
        internal val None = TextGeometricTransform(1.0f, 0.0f)
    }
}

fun lerp(
    start: TextGeometricTransform,
    stop: TextGeometricTransform,
    fraction: Float
): TextGeometricTransform {
    return TextGeometricTransform(
        lerp(start.scaleX, stop.scaleX, fraction),
        lerp(start.skewX, stop.skewX, fraction)
    )
}
