package buzz.getcoco.media;

import com.google.gson.annotations.SerializedName;

/**
 * The class representing a Channel in a {@link Network}. A virtual construct
 * to add constraints on {@link Stream}s.
 * A channel can contain any number of {@link Stream}s whose purpose is to
 * send data across clients.
 * Properties of Channels include. But not limited to:
 * <ul>
 *   <li>Once {@link #join(ChannelStatusListener)} is called. It will be processed unless the
 *   {@link Network} gets to {@link Network.State#DISCONNECTED} or an equivalent state</li>
 *
 * </ul>
 */
public class Channel {

  /**
   * An enum indicating the state of the channel.
   */
  public enum State {
    CREATED,
    JOINING,
    JOINED,
    JOIN_FAILED,
    LEFT,
    LEAVE_FAILED,
    DESTROYED,
    DESTROY_FAILED;

    static State getValue(int val) {
      return values()[val];
    }
  }

  @SerializedName(Constants.CHANNEL_NAME)
  private String channelName;
  @SerializedName(Constants.CHANNEL_METADATA)
  private String channelMetadata;
  @SerializedName(Constants.CHANNEL_MAX_STREAMS)
  private int maxStreams;

  @SerializedName(Constants.NETWORK_ID)
  private final String networkId;
  @SerializedName(Constants.CHANNEL_ID)
  private final int channelId;

  private transient State channelState = null;

  protected Channel(String networkId, int channelId) {
    this.networkId = networkId;
    this.channelId = channelId;
  }

  public String getNetworkId() {
    return networkId;
  }

  public int getId() {
    return channelId;
  }

  protected void internalSetState(State state) {
    this.channelState = state;
  }

  protected void internalSetName(String channelName) {
    this.channelName = channelName;
  }

  protected void internalSetMetadata(String channelMetadata) {
    this.channelMetadata = channelMetadata;
  }

  protected void internalSetMaxStreams(int maxStreams) {
    this.maxStreams = maxStreams;
  }

  public State getState() {
    return channelState;
  }

  public String getName() {
    return channelName;
  }

  public String getMetadata() {
    return channelMetadata;
  }

  public int getMaxStreams() {
    return maxStreams;
  }

  public void join(ChannelStatusListener listener) {
    CocoMediaClient.getInstance().getNativeHandler().joinChannel(this, listener);
  }

  public void leave() {
    CocoMediaClient.getInstance().getNativeHandler().leaveChannel(this);
  }

  @Override
  public String toString() {
    return "Channel{"
           + "channelName='" + channelName + '\''
           + ", channelMetadata='" + channelMetadata + '\''
           + ", maxStreams=" + maxStreams
           + ", networkId='" + networkId + '\''
           + ", channelId=" + channelId
           + ", channelState=" + channelState
           + '}';
  }

  /**
   * The listener for updating the stream and channel join statuses.
   */
  public interface ChannelStatusListener extends Listener {

    /**
     * Triggered when the join channel's joining state is changed.
     *
     * @param status The status of the join request
     */
    void onJoinStatusChanged(State status);

    /**
     * Triggered when a new stream is opened/closed.
     *
     * @param stream The stream data container for starting and listening to the data
     * @param status The status of the given stream
     */
    void onStreamInfoUpdated(RxStreamInfo stream, Stream.State status);
  }
}
