package buzz.getcoco.media;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.util.logging.Logger;

/**
 * The base command class which will be extended by corresponding Command classes.
 *
 * @param <T> The capture for Identifying the commandId of this command
 * @param <U> The capture for generating the response of this command
 */
public class Command<T extends CommandIdInterface, U extends CommandResponse<T>> {

  private static final Logger LOGGER = Util.getLogger();
  private static final long DEFAULT_TIMEOUT = 10_000;

  /**
   * An enum denoting the status of a command.
   */
  public enum State {
    SUCCESS,
    INVALID,
    FAILURE,
    PARTIAL_SUCCESS,
    TIMEOUT,
    REJECTED,
    DEVICE_BUSY,
    IN_PROGRESS,
    AUTH_FAILED,
    RESOURCE_NOT_SUPPORTED,
    SUCCESS_INSECURE,
    PARTIAL_SUCCESS_INSECURE,
    CONNECTIVITY_ERROR,
    CMD_NOT_SUPPORTED,
    TOKEN_NOT_SET,
    TOKEN_REFRESH_FAILED,
    DISCOVERY_NOT_APPLICABLE,
    NETWORK_DISCONNECTED,
    MAX_COUNT_REACHED,
    DEVICE_UNREACHABLE,
    NETWORK_BLOCKED;

    static State getValue(int val) {
      return values()[val];
    }
  }

  private final transient T commandId;
  private final transient Class<U> responseClass;

  private transient long timeout = DEFAULT_TIMEOUT;

  protected Command(T commandId, Class<U> responseClass) {
    this.commandId = commandId;
    this.responseClass = responseClass;
  }

  static void init(GsonBuilder builder) {
    builder.registerTypeAdapter(State.class,
        (JsonDeserializer<State>) (json, typeOfT, context) -> State.getValue(json.getAsInt()));
  }

  public T getCommandId() {
    return commandId;
  }

  /**
   * Set the timeout of this command.
   * NOTE: This value can't be modified one command is sent or if it's in progress.
   *
   * @param timeout The timeout which has to be used.
   */
  public void setTimeout(long timeout) {
    this.timeout = timeout;
  }

  U createResponse(String json) {
    LOGGER.fine("responseJson: " + json);

    Gson gson = CocoMediaClient.getInstance().getGson();
    JsonElement je = JsonParser.parseString(json);

    if (!je.isJsonObject()) {
      throw new IllegalStateException("Internal error, jo should be a JsonObject."
                                      + "But, it is: " + je.getClass());
    }

    JsonObject jo = je.getAsJsonObject();
    JsonElement cmdParamsJson = jo.get(Constants.COMMAND_PARAMS);
    JsonElement cmdStatusJson = jo.get(Constants.COMMAND_STATUS);
    JsonElement errorMessageJson = jo.get(Constants.ERROR_MESSAGE);

    State status = gson.fromJson(cmdStatusJson, State.class);
    String errorMessage = gson.fromJson(errorMessageJson, String.class);

    U response =
        gson.fromJson(null != cmdParamsJson ? cmdParamsJson : new JsonObject(), responseClass);

    response.setStatus((null != status) ? status : State.FAILURE);
    response.setErrorMessage(errorMessage);
    response.setCommandId(getCommandId());

    return response;
  }

  JsonObject toJson() {
    Gson gson = CocoMediaClient.getInstance().getGson();

    JsonObject jo = new JsonObject();

    jo.add(Constants.COMMAND_ID, gson.toJsonTree(this.getCommandId()));
    jo.add(Constants.COMMAND_PARAMS, gson.toJsonTree(this));
    jo.addProperty(Constants.TIMEOUT, timeout);

    return jo;
  }

  @Override
  public String toString() {
    return "Command{"
           + "commandId=" + commandId
           + ", responseClass=" + responseClass
           + ", timeout=" + timeout
           + '}';
  }
}
