package buzz.getcoco.media;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.logging.Logger;

/**
 * Construct representing a video/audio/any frame of data.
 */
public final class Frame {

  private static final Logger LOGGER = Util.getLogger();

  public static final int FRAME_TYPE_KEY  = 0;
  public static final int FRAME_TYPE_NONE = 1;

  public static final int MIME_TYPE_VIDEO_H264 = 0;
  public static final int MIME_TYPE_AUDIO_AAC  = 1;
  public static final int MIME_TYPE_TEXT       = 2;

  private final int originalPosition;
  private final long frameIndex;
  private final int frameType;
  private final int mimeType;
  private final long frameDuration;
  private final long framePts;
  private final ByteBuffer data;

  public static final int HEADER_SIZE = 32;

  /**
   * A builder class for building {@link Frame}s.
   */
  public static class Builder {

    private int frameIndex;
    private int frameType;
    private int mimeType;
    private long frameDuration;
    private long framePts;

    public Builder setFrameIndex(int frameIndex) {
      this.frameIndex = frameIndex;
      return this;
    }

    public Builder setFrameType(int frameType) {
      this.frameType = frameType;
      return this;
    }

    public Builder setMimeType(int mimeType) {
      this.mimeType = mimeType;
      return this;
    }

    public Builder setFramePts(long framePts) {
      this.framePts = framePts;
      return this;
    }

    public Builder setFrameDuration(long frameDuration) {
      this.frameDuration = frameDuration;
      return this;
    }

    /**
     * Build frame with the given size.
     *
     * @param frameSize size of the frame which will be allocated after the header.
     * @return Frame built using the data
     */
    public Frame build(int frameSize) {
      assert 0 < frameSize : "frameSize: " + frameSize;

      ByteBuffer buf = ByteBuffer.allocateDirect(HEADER_SIZE + frameSize);

      buf.order(ByteOrder.nativeOrder());

      buf.putInt(frameIndex);
      buf.putInt(mimeType);
      buf.putInt(frameType);
      buf.putLong(frameDuration);
      buf.putLong(framePts);
      buf.putInt(frameSize);

      buf.position(0);

      return new Frame(buf);
    }

    @Override
    public String toString() {
      return "Builder{"
             + "frameIndex=" + frameIndex
             + ", frameType=" + frameType
             + ", mimeType=" + mimeType
             + ", frameDuration=" + frameDuration
             + ", framePts=" + framePts
             + '}';
    }
  }

  /**
   * The constructor for this class.
   * MUST be in sync with coco_media_client_packed_frame_t.
   *
   * @param buf The bytebuffer from which this frame has to be created
   */
  public Frame(ByteBuffer buf) {
    assert null != buf;
    assert HEADER_SIZE < buf.remaining();

    originalPosition = buf.position();

    frameIndex = buf.getInt();
    mimeType = buf.getInt();
    frameType = buf.getInt();
    frameDuration = buf.getLong();
    framePts = buf.getLong();
    int frameSize = buf.getInt();

    LOGGER.fine("frameSize: " + frameSize);

    assert (0 <= frameSize)     : "frameSize: " + frameSize;
    assert (0 <= frameIndex)    : "frameIndex: " + frameIndex;
    assert (0 <= framePts)      : "framePts: " + framePts;
    assert (0 <= frameDuration) : "frameDuration: " + frameDuration;

    assert (FRAME_TYPE_KEY == frameType || FRAME_TYPE_NONE == frameType);

    this.data = buf;
  }

  public int getMimeType() {
    return mimeType;
  }

  public long getFrameIndex() {
    return frameIndex;
  }

  public int getFrameType() {
    return frameType;
  }

  public long getFrameDuration() {
    return frameDuration;
  }

  public long getFramePts() {
    return framePts;
  }

  public ByteBuffer getData() {
    return data;
  }

  ByteBuffer getSendData() {
    data.position(originalPosition);
    return data;
  }

  @Override
  public String toString() {
    return "Frame{"
           + "originalPosition=" + originalPosition
           + ", frameIndex=" + frameIndex
           + ", frameType=" + frameType
           + ", mimeType=" + mimeType
           + ", frameDuration=" + frameDuration
           + ", framePts=" + framePts
           + ", data=" + data
           + '}';
  }
}
