package buzz.getcoco.media;

import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializer;
import com.google.gson.annotations.SerializedName;
import java.util.Arrays;

/**
 * The class representing a coconet.
 * A typical use case like:
 * <ul>
 * <li>Calling would mean: One {@link Network} per one call</li>
 * <li>Chatting would mean: One {@link Network} per one group</li>
 * </ul>
 * The network should be {@link State#CONNECTED} using {@link #connect()}
 * before {@link #sendCommand(Command, CommandStatusListener)}.
 * {@link State} of a network can be determined using {@link #getStatus()}
 * and can be later disconnected using {@link #disconnect()}.
 */
public class Network {

  public static final long MANAGEMENT_NODE_ID = 0;

  /**
   * Determines the type of the network.
   */
  public enum Type {
    IOT,
    MEDIA_NET,
    UNKNOWN;

    int getInt() {
      return ordinal();
    }

    static Type getValue(int value) {
      return values()[value];
    }
  }

  /**
   * Determines the connection status of the network.
   */
  public enum State {
    CONNECTED,
    CONNECTING,
    CONNECTION_ERROR,
    RESET,
    DISCONNECTED,
    BLOCKED,
    REMOTE_CONNECTED;

    int getInt() {
      return ordinal();
    }

    static State getValue(int value) {
      return values()[value];
    }
  }

  /**
   * An enum indicating the role of the user in a given network.
   */
  public enum Role {
    OWNER,
    ADMIN,
    GUEST;

    int getInt() {
      return ordinal();
    }

    static Role getValue(int value) {
      return values()[value];
    }
  }

  /**
   * An enum indicating the accessType of the user in a given network.
   */
  public enum AccessType {
    LOCAL,
    REMOTE;

    int getInt() {
      return ordinal();
    }

    static AccessType getValue(int value) {
      return values()[value];
    }
  }

  /**
   * An enum representing the possible commands that can be sent to manage a network.
   */
  public enum CommandId implements CommandIdInterface {
    CREATE_CHANNEL,
    DELETE_CHANNEL,
    GET_AVAILABLE_CHANNELS;

    @Override
    public int getInt() {
      return ordinal();
    }
  }

  @SerializedName(Constants.NETWORK_NAME)
  private String networkName;
  @SerializedName(Constants.NETWORK_METADATA)
  private String networkMetadata;
  @SerializedName(Constants.NETWORK_TYPE)
  private Type networkType;

  @SerializedName(Constants.ROLE)
  private Role userRole;
  @SerializedName(Constants.ACCESS_TYPE)
  private AccessType accessType;

  @SerializedName(Constants.NETWORK_ID)
  private final String networkId;

  private transient MessagingListener messagingListener;

  private transient State networkStatus = State.DISCONNECTED;

  protected Network(String networkId) {
    this.networkId = networkId;
  }

  static void init(GsonBuilder builder) {

    builder.registerTypeAdapter(CommandId.class,
        (JsonSerializer<CommandId>) (src, typeOfSrc, context) -> new JsonPrimitive(src.getInt()));

    builder.registerTypeAdapter(Role.class,
        (JsonSerializer<Role>) (src, typeOfSrc, context) -> new JsonPrimitive(src.getInt()));

    builder.registerTypeAdapter(Role.class,
        (JsonDeserializer<Role>) (json, typeOfT, context) -> Role.getValue(json.getAsInt()));

    builder.registerTypeAdapter(AccessType.class,
        (JsonSerializer<AccessType>) (src, typeOfSrc, context) -> new JsonPrimitive(src.getInt()));

    builder.registerTypeAdapter(AccessType.class,
        (JsonDeserializer<AccessType>) (json, t, c) -> AccessType.getValue(json.getAsInt()));

    builder.registerTypeAdapter(Type.class,
        (JsonSerializer<Type>) (src, typeOfSrc, context) -> new JsonPrimitive(src.getInt()));

    builder.registerTypeAdapter(Type.class,
        (JsonDeserializer<Type>) (json, typeOfT, context) -> Type.getValue(json.getAsInt()));

    builder.registerTypeAdapter(State.class,
        (JsonSerializer<State>) (src, typeOfSrc, context) -> new JsonPrimitive(src.getInt()));

    builder.registerTypeAdapter(State.class,
        (JsonDeserializer<State>) (json, typeOfT, context) -> State.getValue(json.getAsInt()));
  }

  /**
   * Get the unique ID of this network.
   *
   * @return ID of this network
   */
  public String getId() {
    return networkId;
  }

  protected void internalOnMessageReceived(String message, long sourceNodeId) {
    if (null == messagingListener) {
      return;
    }

    messagingListener.onNewMessageReceived(message, sourceNodeId);
  }

  protected void internalOnContentInfoMessageReceived(String message,
                                                      long sourceNodeId, int contentTime) {
    if (null == messagingListener) {
      return;
    }

    messagingListener.onContentInfoMessageReceived(message, sourceNodeId, contentTime);
  }

  protected void internalNodeConnectionStatusCallback(long nodeId, boolean isOnline) {
    if (null == messagingListener) {
      return;
    }

    messagingListener.onNodeConnectionStatusChanged(nodeId, isOnline);
  }

  protected void internalSetMetadata(String networkMetadata) {
    this.networkMetadata = networkMetadata;
  }

  protected void internalSetName(String networkName) {
    this.networkName = networkName;
  }

  protected void internalSetType(Type networkType) {
    this.networkType = networkType;
  }

  protected void internalSetStatus(State networkStatus) {
    this.networkStatus = networkStatus;
  }

  public String getMetadata() {
    return networkMetadata;
  }

  public String getName() {
    return networkName;
  }

  public Type getType() {
    return networkType;
  }

  public State getStatus() {
    return networkStatus;
  }

  /**
   * Get the role of the current user for this network.
   *
   * @return The role of the user in this network
   */
  public Role getUserRole() {
    return userRole;
  }

  /**
   * Get the access type of the current user for this network.
   *
   * @return The access type of the current user to this network
   */
  public AccessType getAccessType() {
    return accessType;
  }

  /**
   * set the message listener for this network.
   *
   * @param listener The interface which will be triggered on new messages.
   *                 Pass null to unset previous listeners.
   */
  public void setMessagingListener(MessagingListener listener) {
    this.messagingListener = listener;
  }

  /**
   * Send message to the specified nodes.
   *
   * @param message          The message which has to be transmitted
   * @param destinationNodes The target nodes which will receive this message,
   *                         pass nothing (or) null for broadcasting.
   */
  public void sendMessage(String message, long... destinationNodes) {
    CocoMediaClient.getInstance().getNativeHandler()
        .sendMessage(this, message, destinationNodes);
  }

  /**
   * Send content info message to the specified nodes.
   *
   * @param message          The message which has to be passed along with this
   * @param contentTime      The Content sync time in millis
   *                         NOTE: passing other time units will cause issues
   * @param destinationNodes The target nodes which will receive this message,
   *                         pass nothing (or) null for broadcasting.
   */
  public void sendContentInfoMessage(String message, int contentTime, long... destinationNodes) {
    CocoMediaClient.getInstance().getNativeHandler()
        .sendContentInfoMessage(this, message, contentTime, destinationNodes);
  }

  /**
   * connect to this network.
   */
  public void connect() {
    CocoMediaClient.getInstance().getNativeHandler().connect(this);

    if (State.CONNECTED != networkStatus
        && State.CONNECTING != networkStatus) {

      internalSetStatus(State.CONNECTING);
    }
  }

  /**
   * disconnect from this network.
   */
  public void disconnect() {
    CocoMediaClient.getInstance().getNativeHandler().disconnect(this);
  }

  public <V extends CommandResponse<CommandId>, U extends Command<CommandId, V>>
      void sendCommand(U command, CommandStatusListener<V> listener) {

    CocoMediaClient.getInstance().getNativeHandler().sendNetworkCommand(this, command, listener);
  }

  @Override
  public String toString() {
    return "Network{"
           + "networkName='" + networkName + '\''
           + ", networkMetadata='" + networkMetadata + '\''
           + ", networkType=" + networkType
           + ", userRole=" + userRole
           + ", accessType=" + accessType
           + ", networkId='" + networkId + '\''
           + ", messagingListener=" + messagingListener
           + ", networkStatus=" + networkStatus
           + '}';
  }

  /**
   * A command for creating a channel in the network.
   * The network will be identified by the network object on which this command will be sent.
   */
  public static class CreateChannel extends Command<CommandId, CreateChannelResponse> {

    @SerializedName(Constants.CHANNEL_NAME)
    private final String name;

    @SerializedName(Constants.CHANNEL_METADATA)
    private final String metadata;

    @SerializedName(Constants.CHANNEL_MAX_STREAMS)
    private final int maxStreams;

    /**
     * The constructor for this class.
     *
     * @param name       The name of the channels that has to be created
     * @param metadata   The metadata of the channel that has to be created
     * @param maxStreams Max number of streams this channel can contain before killing previous ones
     */
    public CreateChannel(String name, String metadata, int maxStreams) {
      super(CommandId.CREATE_CHANNEL, CreateChannelResponse.class);

      this.name = name;
      this.metadata = metadata;
      this.maxStreams = maxStreams;
    }

    @Override
    public String toString() {
      return "CreateChannel{"
             + "name='" + name + '\''
             + ", metadata='" + metadata + '\''
             + ", maxStreams=" + maxStreams
             + "} "
             + super.toString();
    }
  }

  /**
   * Response for {@link CreateChannel} command.
   */
  public static class CreateChannelResponse extends CommandResponse<CommandId> {

    @SerializedName(Constants.CHANNEL_ID)
    private final int channelId;

    public CreateChannelResponse(int channelId) {
      super(CommandId.CREATE_CHANNEL);
      this.channelId = channelId;
    }

    public int getChannelId() {
      return channelId;
    }

    @Override
    public String toString() {
      return "CreateChannelResponse{"
             + "channelId=" + channelId
             + "} "
             + super.toString();
    }
  }

  /**
   * A command for deleting a channel in the network.
   * The network will be identified by the network object on which this command will be sent.
   */
  public static class DeleteChannel extends Command<CommandId, DeleteChannelResponse> {

    @SerializedName(Constants.CHANNEL_ID)
    private final int channelId;

    protected DeleteChannel(int channelId) {
      super(CommandId.DELETE_CHANNEL, DeleteChannelResponse.class);

      this.channelId = channelId;
    }

    @Override
    public String toString() {
      return "DeleteChannel{"
             + "channelId=" + channelId
             + "} "
             + super.toString();
    }
  }

  /**
   * A response for {@link DeleteChannel} command.
   */
  public static class DeleteChannelResponse extends CommandResponse<CommandId> {

    public DeleteChannelResponse() {
      super(CommandId.DELETE_CHANNEL);
    }
  }

  /**
   * A command for getting all available channels in the network.
   * The network will be identified by the network object on which this command will be sent.
   */
  public static class GetAvailableChannels
      extends Command<CommandId, GetAvailableChannelsResponse> {

    public GetAvailableChannels() {
      super(CommandId.GET_AVAILABLE_CHANNELS, GetAvailableChannelsResponse.class);
    }
  }

  /**
   * A response for {@link GetAvailableChannels} command.
   */
  public static class GetAvailableChannelsResponse extends CommandResponse<CommandId> {

    @SerializedName(Constants.CHANNELS_INFO_ARR)
    private final Channel[] channels;

    /**
     * The constructor for this class.
     *
     * @param channels Available channels array
     */
    public GetAvailableChannelsResponse(Channel[] channels) {
      super(CommandId.GET_AVAILABLE_CHANNELS);

      this.channels = channels;
    }

    public Channel[] getChannels() {
      return channels;
    }

    @Override
    public String toString() {
      return "GetAvailableChannelsResponse{"
             + "channels=" + Arrays.toString(channels)
             + "} "
             + super.toString();
    }
  }

  /**
   * The listener for status updates of sent commands
   * {@link #sendCommand(Command, CommandStatusListener)}.
   *
   * @param <T> The capture for {@link CommandResponse}
   */
  public interface CommandStatusListener<T extends CommandResponse<CommandId>> extends Listener {
    void onResponse(T response, Throwable tr);
  }

  /**
   * The listener for listening to content info and messages sent by other nodes.
   */
  public interface MessagingListener {

    void onNodeConnectionStatusChanged(long nodeId, boolean isOnline);

    void onNewMessageReceived(String message, long sourceNodeId);

    void onContentInfoMessageReceived(String message, long sourceNodeId, int contentTime);
  }
}
