package buzz.getcoco.media;

import java.util.Objects;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Logger;

/**
 * A construct indicating audio/video stream. Even though it's not limited by type.
 * NOTE: State management MUST be handled in application layer. This class is inherently unmanaged.
 * <pre>
 *   State management include, but not limited to:
 *   - Calling {@link #internalStop()} only after {@code 0 != streamHandle}
 *   - Calling {@link #sendData(Frame)} only after {@code 0 != streamHandle}
 *   - Calling {@link #internalStart(StreamStatusListener)} only after {@code 0 == streamHandle}
 * </pre>
 * NOTE:
 * <pre>
 *   - An ideal TXStream's lifecycle would be:
 *     STARTED -&gt; DESTROYED
 *   - An ideal RXStream's lifecycle would be:
 *     CREATED -&gt; STARTED -&gt; STOPPED -&gt; DESTROYED
 * </pre>
 */
public abstract class Stream {

  private static final Logger LOGGER = Util.getLogger();

  /**
   * An enum indicating the state of this stream.
   */
  public enum State {
    CREATED,
    CREATE_FAILED,
    STARTING,
    STARTED,
    STOPPED, // deviated from c-sdk (CLOSED)
    DESTROYED;

    static State getValue(int val, boolean isTx) {
      // to be in sync with c-layer enums
      switch (val) {
        case 0:
          return isTx ? STARTED : CREATED;
        case 1:
          return CREATE_FAILED;
        case 3:
          return STARTED;
        case 2:
          return STARTING;
        case 4:
          return STOPPED;
        case 5:
        default:
          return DESTROYED;
      }
    }
  }

  private final String networkId;
  private final int channelId;
  private final String sdp;
  private final long streamId; // in case of tx this will be 0

  /**
   * Used to prevent {@link DefaultNativeCallbacksHandler#streamStatusChangedCallback}
   * from returning.
   * {@link #streamHandle} is a pointer to a c-struct which will be de-allocated after
   * a setting the status.
   * Any calls ({@link #sendData(Frame)} {@link #internalStop()}) made during that point
   * might fail with SEGV. So, the cb MUST wait for all the senders to complete.
   * The case mentioned is nothing but a Read-Write scenario.
   * Waiting for stream handles is a write operation and stopping/sending data is a read operation.
   */
  private final ReentrantReadWriteLock handleLock = new ReentrantReadWriteLock();

  private volatile long streamHandle = 0; // pointer to c-struct

  private long sourceNodeId;
  private State status = State.STOPPED;

  protected Stream(String sdp, String networkId, int channelId, long streamId) {
    this.sdp = sdp;
    this.networkId = networkId;
    this.channelId = channelId;
    this.streamId = streamId;
  }

  protected abstract boolean isTxStream();

  public State getStatus() {
    return status;
  }

  public String getNetworkId() {
    return networkId;
  }

  public int getChannelId() {
    return channelId;
  }

  public long getId() {
    return streamId;
  }

  public String getSdp() {
    return sdp;
  }

  /**
   * Get the source nodeId of this stream.
   *
   * @return The id of the originator of the stream.
   */
  public long getSourceNodeId() {
    return sourceNodeId;
  }

  protected void internalSetSourceNodeId(long nodeId) {
    this.sourceNodeId = nodeId;
  }

  protected void internalSetStatus(State status) {
    this.status = status;
  }

  void setStreamHandle(long streamHandle) {
    this.streamHandle = streamHandle;
  }

  long getStreamHandle() {
    return streamHandle;
  }

  void waitUntilAllSendersReturn() {
    handleLock.writeLock().lock();
    handleLock.writeLock().unlock();
  }

  /**
   * Starts the stream.
   * If this is a stream object created by SDK
   * then this will probably be a rx stream for listening from other nodes
   * using {@link StreamStatusListener#onStreamDataReceived(Frame)}
   * else this stream object will be for sending data to other nodes
   * using {@link #sendData(Frame)}
   * <p><b>NOTE: If the stream id is not set. Then a new stream will be created.</b></p>
   *
   * @param listener The listener for getting the status updates
   */
  protected final void internalStart(StreamStatusListener listener) {
    if (State.STARTING == status || State.STARTED == status) {
      throw new StateException(status, "cannot internalStart when status is: " + status);
    }

    Objects.requireNonNull(listener);

    CocoMediaClient.getInstance().getNativeHandler().startStream(this, listener);
  }

  /**
   * Send data to other nodes subscribed to this stream.
   *
   * @param frame The audio/video frame which has to be transmitted
   * @throws StateException If this is a listening RX stream
   */
  protected void sendData(Frame frame) {
    if (!frame.getData().isDirect()) {
      throw new IllegalArgumentException("frame data MUST be a direct ByteBuffer");
    }

    handleLock.readLock().lock();

    try {
      if (0 == streamHandle) {
        throw new StateException(status, "stream handle un-set");
      }

      CocoMediaClient.getInstance().getNativeHandler().sendStreamData(this, frame);
    } finally {
      handleLock.readLock().unlock();
    }
  }

  /**
   * Stop listening and sending to the stream.
   * If other nodes are subscribed to this stream
   * They will be notified with {@link State#STOPPED}.
   */
  protected final void internalStop() {
    handleLock.readLock().lock();

    try {
      if (0 == streamHandle) {
        LOGGER.warning("stream handle un-set, ignoring stop call");
        return;
      }

      CocoMediaClient.getInstance().getNativeHandler().stopStream(this);
    } finally {
      handleLock.readLock().unlock();
    }
  }

  @Override
  public String toString() {
    return "Stream{"
           + "networkId='" + networkId + '\''
           + ", channelId=" + channelId
           + ", sdp='" + sdp + '\''
           + ", streamId=" + streamId
           + ", sendLock=" + handleLock
           + ", streamHandle=" + streamHandle
           + ", sourceNodeId=" + sourceNodeId
           + ", status=" + status
           + '}';
  }

  /**
   * A listener which is used for updating the stream status and passing back the stream data.
   */
  public interface StreamStatusListener extends Listener {

    /**
     * Triggered when the stream's status changes.
     *
     * @param status Determines if the stream is opened/closed etc...
     */
    void onStreamStatusChanged(State status);

    /**
     * Triggered when a frame is received from the stream.
     *
     * @param frame The audio/video frame
     */
    default void onStreamDataReceived(Frame frame) {}
  }

  /**
   * Constructs an illegal state exception with the specified detail message and cause.
   */
  public static class StateException extends IllegalStateException {

    private static final long serialVersionUID = 9120120234234L;

    public final Stream.State currentState;

    /**
     * Constructor of this class.
     *
     * @param currentState The state of the stream
     * @param message      The message that has to be passed along
     */
    public StateException(State currentState, String message) {
      super(message);

      this.currentState = currentState;
    }

    @Override
    public String toString() {
      return "StateException{"
             + "currentState=" + currentState
             + "} "
             + super.toString();
    }
  }
}
