package dev.codeflush.baseencoder;

import dev.codeflush.baseencoder.compatibility.*;

import java.io.*;
import java.nio.charset.Charset;

public interface BaseEncoding {

    int bitBlockSize();
    char get(int index);
    int index(char c);

    // region default encodes
    default <IMPL> String encode(IMPL typeInformation, Object obj, Serializer<IMPL> serializer) {
        final String result;

        try (StringWriter writer = new StringWriter()) {
            try (OutputStream out = new BaseEncoder(this, writer)) {
                serializer.serialize(typeInformation, obj, out);
            }

            writer.flush();
            result = writer.toString();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return result;
    }

    default <T> String encode(T obj, TypedSerializer<T> serializer) {
        final String result;

        try (StringWriter writer = new StringWriter()) {
            try (OutputStream out = new BaseEncoder(this, writer)) {
                serializer.serialize(obj, out);
            }

            writer.flush();
            result = writer.toString();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return result;
    }

    default String encode(String str) {
        return encode(str.getBytes());
    }

    default String encode(String str, Charset charset) {
        return encode(str.getBytes(charset));
    }

    default String encode(byte[] bytes) {
        final String result;

        try (StringWriter writer = new StringWriter()) {
            try (OutputStream out = new BaseEncoder(this, writer)) {
                out.write(bytes);
            }

            writer.flush();
            result = writer.toString();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return result;
    }
    // endregion

    // region default decodes
    default <T, IMPL> T decodeObject(IMPL typeInformation, String str, Deserializer<IMPL> deserializer) {
        final T result;

        try (InputStream in = new BaseDecoder(this, new StringReader(str))) {
            result = deserializer.deserialize(typeInformation, in);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return result;
    }

    default <T> T decodeObject(String str, TypedDeserializer<T> deserializer) {
        final T result;

        try (InputStream in = new BaseDecoder(this, new StringReader(str))) {
            result = deserializer.deserialize(in);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return result;
    }

    default String decodeString(String str) {
        final StringBuilder result = new StringBuilder();

        try (InputStream in = new BaseDecoder(this, new StringReader(str))) {
            final byte[] buffer = new byte[8192];
            int bytesRead;

            while ((bytesRead = in.read(buffer)) != -1) {
                result.append(new String(buffer, 0, bytesRead));
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return result.toString();
    }

    default String decodeString(String str, Charset charset) {
        final StringBuilder result = new StringBuilder();

        try (InputStream in = new BaseDecoder(this, new StringReader(str))) {
            final byte[] buffer = new byte[8192];
            int bytesRead;

            while ((bytesRead = in.read(buffer)) != -1) {
                result.append(new String(buffer, 0, bytesRead, charset));
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return result.toString();
    }

    default byte[] decodeBytes(String str) {
        final byte[] result;

        try (InputStream in = new BaseDecoder(this, new StringReader(str))) {
            try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
                final byte[] buffer = new byte[8192];
                int bytesRead;

                while ((bytesRead = in.read(buffer)) != -1) {
                    out.write(buffer, 0, bytesRead);
                }

                out.flush();
                result = out.toByteArray();
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return result;
    }
    // endregion

    BaseEncoding BASE64 = new SimpleBaseEncoding("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/");
    BaseEncoding BASE64_WEB = new SimpleBaseEncoding("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_");
    BaseEncoding BASE32 = new SimpleBaseEncoding("abcdefghijklmnopqrstuvwxyz012345");
    BaseEncoding BASE16 = new SimpleBaseEncoding("0123456789abcdef");
    BaseEncoding BASE8 = new SimpleBaseEncoding("01234567");
    BaseEncoding BASE2 = new SimpleBaseEncoding("01");
}
