package io.k8s.api.core.v1

import dev.hnaderi.k8s._

/** Secret holds secret data of a certain type. The total bytes of the values in the Data field must be less than MaxSecretSize bytes. */
final case class Secret(
  data : Option[Map[String, String]] = None,
  immutable : Option[Boolean] = None,
  metadata : Option[io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta] = None,
  `type` : Option[String] = None,
  stringData : Option[Map[String, String]] = None
) extends KObject {
  protected val _resourceKind = ResourceKind("", "Secret", "v1")


  /** Returns a new data with data set to new value */
  def withData(value: Map[String, String]) : Secret = copy(data = Some(value))
  /** Adds new values to data */
  def addData(newValues: (String, String)*) : Secret = copy(data = Some(data.fold(newValues.toMap)(_ ++ newValues)))
  /** if data has a value, transforms to the result of function*/
  def mapData(f: Map[String, String] => Map[String, String]) : Secret = copy(data = data.map(f))

  /** Returns a new data with immutable set to new value */
  def withImmutable(value: Boolean) : Secret = copy(immutable = Some(value))
  /** if immutable has a value, transforms to the result of function*/
  def mapImmutable(f: Boolean => Boolean) : Secret = copy(immutable = immutable.map(f))

  /** Returns a new data with metadata set to new value */
  def withMetadata(value: io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta) : Secret = copy(metadata = Some(value))
  /** if metadata has a value, transforms to the result of function*/
  def mapMetadata(f: io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta => io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta) : Secret = copy(metadata = metadata.map(f))

  /** Returns a new data with `type` set to new value */
  def withType(value: String) : Secret = copy(`type` = Some(value))
  /** if `type` has a value, transforms to the result of function*/
  def mapType(f: String => String) : Secret = copy(`type` = `type`.map(f))

  /** Returns a new data with stringData set to new value */
  def withStringData(value: Map[String, String]) : Secret = copy(stringData = Some(value))
  /** Adds new values to stringData */
  def addStringData(newValues: (String, String)*) : Secret = copy(stringData = Some(stringData.fold(newValues.toMap)(_ ++ newValues)))
  /** if stringData has a value, transforms to the result of function*/
  def mapStringData(f: Map[String, String] => Map[String, String]) : Secret = copy(stringData = stringData.map(f))
}

