package io.k8s.api.autoscaling.v2

import dev.hnaderi.k8s.utils._

/** HPAScalingRules configures the scaling behavior for one direction. These Rules are applied after calculating DesiredReplicas from metrics for the HPA. They can limit the scaling velocity by specifying scaling policies. They can prevent flapping by specifying the stabilization window, so that the number of replicas is not set instantly, instead, the safest value from the stabilization window is chosen. */
final case class HPAScalingRules(
  policies : Option[Seq[io.k8s.api.autoscaling.v2.HPAScalingPolicy]] = None,
  selectPolicy : Option[String] = None,
  stabilizationWindowSeconds : Option[Int] = None
) {

  /** Returns a new data with policies set to new value */
  def withPolicies(value: Seq[io.k8s.api.autoscaling.v2.HPAScalingPolicy]) : HPAScalingRules = copy(policies = Some(value))
  /** Appends new values to policies */
  def addPolicies(newValues: io.k8s.api.autoscaling.v2.HPAScalingPolicy*) : HPAScalingRules = copy(policies = Some(policies.fold(newValues)(_ ++ newValues)))
  /** if policies has a value, transforms to the result of function*/
  def mapPolicies(f: Seq[io.k8s.api.autoscaling.v2.HPAScalingPolicy] => Seq[io.k8s.api.autoscaling.v2.HPAScalingPolicy]) : HPAScalingRules = copy(policies = policies.map(f))

  /** Returns a new data with selectPolicy set to new value */
  def withSelectPolicy(value: String) : HPAScalingRules = copy(selectPolicy = Some(value))
  /** if selectPolicy has a value, transforms to the result of function*/
  def mapSelectPolicy(f: String => String) : HPAScalingRules = copy(selectPolicy = selectPolicy.map(f))

  /** Returns a new data with stabilizationWindowSeconds set to new value */
  def withStabilizationWindowSeconds(value: Int) : HPAScalingRules = copy(stabilizationWindowSeconds = Some(value))
  /** if stabilizationWindowSeconds has a value, transforms to the result of function*/
  def mapStabilizationWindowSeconds(f: Int => Int) : HPAScalingRules = copy(stabilizationWindowSeconds = stabilizationWindowSeconds.map(f))
}

object HPAScalingRules {

    implicit val encoder : Encoder[io.k8s.api.autoscaling.v2.HPAScalingRules] = new Encoder[io.k8s.api.autoscaling.v2.HPAScalingRules] {
        def apply[T : Builder](o: io.k8s.api.autoscaling.v2.HPAScalingRules) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("policies", o.policies)
            .write("selectPolicy", o.selectPolicy)
            .write("stabilizationWindowSeconds", o.stabilizationWindowSeconds)
            .build
        }
    }

    implicit val decoder: Decoder[HPAScalingRules] = new Decoder[HPAScalingRules] {
      def apply[T : Reader](t: T): Either[String, HPAScalingRules] = for {
          obj <- ObjectReader(t)
          policies <- obj.readOpt[Seq[io.k8s.api.autoscaling.v2.HPAScalingPolicy]]("policies")
          selectPolicy <- obj.readOpt[String]("selectPolicy")
          stabilizationWindowSeconds <- obj.readOpt[Int]("stabilizationWindowSeconds")
      } yield HPAScalingRules (
          policies = policies,
          selectPolicy = selectPolicy,
          stabilizationWindowSeconds = stabilizationWindowSeconds
        )
    }
}

