package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** ResourceFieldSelector represents container resources (cpu, memory) and their output format */
final case class ResourceFieldSelector(
  resource : String,
  containerName : Option[String] = None,
  divisor : Option[io.k8s.apimachinery.pkg.api.resource.Quantity] = None
) {

  /** Returns a new data with resource set to new value */
  def withResource(value: String) : ResourceFieldSelector = copy(resource = value)
  /** transforms resource to result of function */
  def mapResource(f: String => String) : ResourceFieldSelector = copy(resource = f(resource))

  /** Returns a new data with containerName set to new value */
  def withContainerName(value: String) : ResourceFieldSelector = copy(containerName = Some(value))
  /** if containerName has a value, transforms to the result of function*/
  def mapContainerName(f: String => String) : ResourceFieldSelector = copy(containerName = containerName.map(f))

  /** Returns a new data with divisor set to new value */
  def withDivisor(value: io.k8s.apimachinery.pkg.api.resource.Quantity) : ResourceFieldSelector = copy(divisor = Some(value))
  /** if divisor has a value, transforms to the result of function*/
  def mapDivisor(f: io.k8s.apimachinery.pkg.api.resource.Quantity => io.k8s.apimachinery.pkg.api.resource.Quantity) : ResourceFieldSelector = copy(divisor = divisor.map(f))
}

object ResourceFieldSelector {

    implicit val encoder : Encoder[io.k8s.api.core.v1.ResourceFieldSelector] = new Encoder[io.k8s.api.core.v1.ResourceFieldSelector] {
        def apply[T : Builder](o: io.k8s.api.core.v1.ResourceFieldSelector) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("resource", o.resource)
            .write("containerName", o.containerName)
            .write("divisor", o.divisor)
            .build
        }
    }

    implicit val decoder: Decoder[ResourceFieldSelector] = new Decoder[ResourceFieldSelector] {
      def apply[T : Reader](t: T): Either[String, ResourceFieldSelector] = for {
          obj <- ObjectReader(t)
          resource <- obj.read[String]("resource")
          containerName <- obj.readOpt[String]("containerName")
          divisor <- obj.readOpt[io.k8s.apimachinery.pkg.api.resource.Quantity]("divisor")
      } yield ResourceFieldSelector (
          resource = resource,
          containerName = containerName,
          divisor = divisor
        )
    }
}

