package io.k8s.api.discovery.v1

import dev.hnaderi.k8s._
import dev.hnaderi.k8s.utils._

/** EndpointSlice represents a subset of the endpoints that implement a service. For a given service there may be multiple EndpointSlice objects, selected by labels, which must be joined to produce the full set of endpoints. */
final case class EndpointSlice(
  endpoints : Seq[io.k8s.api.discovery.v1.Endpoint],
  addressType : String,
  ports : Option[Seq[io.k8s.api.discovery.v1.EndpointPort]] = None,
  metadata : Option[io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta] = None
) extends KObject {
  protected val _resourceKind = ResourceKind("discovery.k8s.io", "EndpointSlice", "v1")


  /** Returns a new data with endpoints set to new value */
  def withEndpoints(value: Seq[io.k8s.api.discovery.v1.Endpoint]) : EndpointSlice = copy(endpoints = value)
  /** Appends new values to endpoints */
  def addEndpoints(newValues: io.k8s.api.discovery.v1.Endpoint*) : EndpointSlice = copy(endpoints = endpoints ++ newValues)
  /** transforms endpoints to result of function */
  def mapEndpoints(f: Seq[io.k8s.api.discovery.v1.Endpoint] => Seq[io.k8s.api.discovery.v1.Endpoint]) : EndpointSlice = copy(endpoints = f(endpoints))

  /** Returns a new data with addressType set to new value */
  def withAddressType(value: String) : EndpointSlice = copy(addressType = value)
  /** transforms addressType to result of function */
  def mapAddressType(f: String => String) : EndpointSlice = copy(addressType = f(addressType))

  /** Returns a new data with ports set to new value */
  def withPorts(value: Seq[io.k8s.api.discovery.v1.EndpointPort]) : EndpointSlice = copy(ports = Some(value))
  /** Appends new values to ports */
  def addPorts(newValues: io.k8s.api.discovery.v1.EndpointPort*) : EndpointSlice = copy(ports = Some(ports.fold(newValues)(_ ++ newValues)))
  /** if ports has a value, transforms to the result of function*/
  def mapPorts(f: Seq[io.k8s.api.discovery.v1.EndpointPort] => Seq[io.k8s.api.discovery.v1.EndpointPort]) : EndpointSlice = copy(ports = ports.map(f))

  /** Returns a new data with metadata set to new value */
  def withMetadata(value: io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta) : EndpointSlice = copy(metadata = Some(value))
  /** if metadata has a value, transforms to the result of function*/
  def mapMetadata(f: io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta => io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta) : EndpointSlice = copy(metadata = metadata.map(f))

  override def foldTo[T : Builder] : T = EndpointSlice.encoder.apply(this)
}

object EndpointSlice {

    implicit val encoder : Encoder[io.k8s.api.discovery.v1.EndpointSlice] = new Encoder[io.k8s.api.discovery.v1.EndpointSlice] {
        def apply[T : Builder](o: io.k8s.api.discovery.v1.EndpointSlice) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("endpoints", o.endpoints)
            .write("addressType", o.addressType)
            .write("ports", o.ports)
            .write("metadata", o.metadata)
            .write("kind", o.kind)
            .write("apiVersion", o.apiVersion)
            .build
        }
    }

    implicit val decoder: Decoder[EndpointSlice] = new Decoder[EndpointSlice] {
      def apply[T : Reader](t: T): Either[String, EndpointSlice] = for {
          obj <- ObjectReader(t)
          endpoints <- obj.read[Seq[io.k8s.api.discovery.v1.Endpoint]]("endpoints")
          addressType <- obj.read[String]("addressType")
          ports <- obj.readOpt[Seq[io.k8s.api.discovery.v1.EndpointPort]]("ports")
          metadata <- obj.readOpt[io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta]("metadata")
      } yield EndpointSlice (
          endpoints = endpoints,
          addressType = addressType,
          ports = ports,
          metadata = metadata
        )
    }
}

