package io.k8s.api.apps.v1

import dev.hnaderi.k8s._
import dev.hnaderi.k8s.utils._

/** StatefulSet represents a set of pods with consistent identities. Identities are defined as:
  - Network: A single stable DNS and hostname.
  - Storage: As many VolumeClaims as requested.

The StatefulSet guarantees that a given network identity will always map to the same storage identity. */
final case class StatefulSet(
  status : Option[io.k8s.api.apps.v1.StatefulSetStatus] = None,
  spec : Option[io.k8s.api.apps.v1.StatefulSetSpec] = None,
  metadata : Option[io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta] = None
) extends KObject {
  protected val _resourceKind = ResourceKind("apps", "StatefulSet", "v1")


  /** Returns a new data with status set to new value */
  def withStatus(value: io.k8s.api.apps.v1.StatefulSetStatus) : StatefulSet = copy(status = Some(value))
  /** if status has a value, transforms to the result of function*/
  def mapStatus(f: io.k8s.api.apps.v1.StatefulSetStatus => io.k8s.api.apps.v1.StatefulSetStatus) : StatefulSet = copy(status = status.map(f))

  /** Returns a new data with spec set to new value */
  def withSpec(value: io.k8s.api.apps.v1.StatefulSetSpec) : StatefulSet = copy(spec = Some(value))
  /** if spec has a value, transforms to the result of function*/
  def mapSpec(f: io.k8s.api.apps.v1.StatefulSetSpec => io.k8s.api.apps.v1.StatefulSetSpec) : StatefulSet = copy(spec = spec.map(f))

  /** Returns a new data with metadata set to new value */
  def withMetadata(value: io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta) : StatefulSet = copy(metadata = Some(value))
  /** if metadata has a value, transforms to the result of function*/
  def mapMetadata(f: io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta => io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta) : StatefulSet = copy(metadata = metadata.map(f))

  override def foldTo[T : Builder] : T = StatefulSet.encoder.apply(this)
}

object StatefulSet {

    implicit val encoder : Encoder[io.k8s.api.apps.v1.StatefulSet] = new Encoder[io.k8s.api.apps.v1.StatefulSet] {
        def apply[T : Builder](o: io.k8s.api.apps.v1.StatefulSet) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("status", o.status)
            .write("spec", o.spec)
            .write("metadata", o.metadata)
            .write("kind", o.kind)
            .write("apiVersion", o.apiVersion)
            .build
        }
    }

    implicit val decoder: Decoder[StatefulSet] = new Decoder[StatefulSet] {
      def apply[T : Reader](t: T): Either[String, StatefulSet] = for {
          obj <- ObjectReader(t)
          status <- obj.readOpt[io.k8s.api.apps.v1.StatefulSetStatus]("status")
          spec <- obj.readOpt[io.k8s.api.apps.v1.StatefulSetSpec]("spec")
          metadata <- obj.readOpt[io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta]("metadata")
      } yield StatefulSet (
          status = status,
          spec = spec,
          metadata = metadata
        )
    }
}

