package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** Adds and removes POSIX capabilities from running containers. */
final case class Capabilities(
  add : Option[Seq[String]] = None,
  drop : Option[Seq[String]] = None
) {

  /** Returns a new data with add set to new value */
  def withAdd(value: Seq[String]) : Capabilities = copy(add = Some(value))
  /** Appends new values to add */
  def addAdd(newValues: String*) : Capabilities = copy(add = Some(add.fold(newValues)(_ ++ newValues)))
  /** if add has a value, transforms to the result of function*/
  def mapAdd(f: Seq[String] => Seq[String]) : Capabilities = copy(add = add.map(f))

  /** Returns a new data with drop set to new value */
  def withDrop(value: Seq[String]) : Capabilities = copy(drop = Some(value))
  /** Appends new values to drop */
  def addDrop(newValues: String*) : Capabilities = copy(drop = Some(drop.fold(newValues)(_ ++ newValues)))
  /** if drop has a value, transforms to the result of function*/
  def mapDrop(f: Seq[String] => Seq[String]) : Capabilities = copy(drop = drop.map(f))
}

object Capabilities {

    implicit val encoder : Encoder[io.k8s.api.core.v1.Capabilities] = new Encoder[io.k8s.api.core.v1.Capabilities] {
        def apply[T : Builder](o: io.k8s.api.core.v1.Capabilities) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("add", o.add)
            .write("drop", o.drop)
            .build
        }
    }

    implicit val decoder: Decoder[Capabilities] = new Decoder[Capabilities] {
      def apply[T : Reader](t: T): Either[String, Capabilities] = for {
          obj <- ObjectReader(t)
          add <- obj.readOpt[Seq[String]]("add")
          drop <- obj.readOpt[Seq[String]]("drop")
      } yield Capabilities (
          add = add,
          drop = drop
        )
    }
}

