package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** Represents a Persistent Disk resource in Google Compute Engine.

A GCE PD must exist before mounting to a container. The disk must also be in the same GCE project and zone as the kubelet. A GCE PD can only be mounted as read/write once or read-only many times. GCE PDs support ownership management and SELinux relabeling. */
final case class GCEPersistentDiskVolumeSource(
  pdName : String,
  fsType : Option[String] = None,
  partition : Option[Int] = None,
  readOnly : Option[Boolean] = None
) {

  /** Returns a new data with pdName set to new value */
  def withPdName(value: String) : GCEPersistentDiskVolumeSource = copy(pdName = value)
  /** transforms pdName to result of function */
  def mapPdName(f: String => String) : GCEPersistentDiskVolumeSource = copy(pdName = f(pdName))

  /** Returns a new data with fsType set to new value */
  def withFsType(value: String) : GCEPersistentDiskVolumeSource = copy(fsType = Some(value))
  /** if fsType has a value, transforms to the result of function*/
  def mapFsType(f: String => String) : GCEPersistentDiskVolumeSource = copy(fsType = fsType.map(f))

  /** Returns a new data with partition set to new value */
  def withPartition(value: Int) : GCEPersistentDiskVolumeSource = copy(partition = Some(value))
  /** if partition has a value, transforms to the result of function*/
  def mapPartition(f: Int => Int) : GCEPersistentDiskVolumeSource = copy(partition = partition.map(f))

  /** Returns a new data with readOnly set to new value */
  def withReadOnly(value: Boolean) : GCEPersistentDiskVolumeSource = copy(readOnly = Some(value))
  /** if readOnly has a value, transforms to the result of function*/
  def mapReadOnly(f: Boolean => Boolean) : GCEPersistentDiskVolumeSource = copy(readOnly = readOnly.map(f))
}

object GCEPersistentDiskVolumeSource {

    implicit val encoder : Encoder[io.k8s.api.core.v1.GCEPersistentDiskVolumeSource] = new Encoder[io.k8s.api.core.v1.GCEPersistentDiskVolumeSource] {
        def apply[T : Builder](o: io.k8s.api.core.v1.GCEPersistentDiskVolumeSource) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("pdName", o.pdName)
            .write("fsType", o.fsType)
            .write("partition", o.partition)
            .write("readOnly", o.readOnly)
            .build
        }
    }

    implicit val decoder: Decoder[GCEPersistentDiskVolumeSource] = new Decoder[GCEPersistentDiskVolumeSource] {
      def apply[T : Reader](t: T): Either[String, GCEPersistentDiskVolumeSource] = for {
          obj <- ObjectReader(t)
          pdName <- obj.read[String]("pdName")
          fsType <- obj.readOpt[String]("fsType")
          partition <- obj.readOpt[Int]("partition")
          readOnly <- obj.readOpt[Boolean]("readOnly")
      } yield GCEPersistentDiskVolumeSource (
          pdName = pdName,
          fsType = fsType,
          partition = partition,
          readOnly = readOnly
        )
    }
}

