package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** ReplicationControllerSpec is the specification of a replication controller. */
final case class ReplicationControllerSpec(
  minReadySeconds : Option[Int] = None,
  replicas : Option[Int] = None,
  selector : Option[Map[String, String]] = None,
  template : Option[io.k8s.api.core.v1.PodTemplateSpec] = None
) {

  /** Returns a new data with minReadySeconds set to new value */
  def withMinReadySeconds(value: Int) : ReplicationControllerSpec = copy(minReadySeconds = Some(value))
  /** if minReadySeconds has a value, transforms to the result of function*/
  def mapMinReadySeconds(f: Int => Int) : ReplicationControllerSpec = copy(minReadySeconds = minReadySeconds.map(f))

  /** Returns a new data with replicas set to new value */
  def withReplicas(value: Int) : ReplicationControllerSpec = copy(replicas = Some(value))
  /** if replicas has a value, transforms to the result of function*/
  def mapReplicas(f: Int => Int) : ReplicationControllerSpec = copy(replicas = replicas.map(f))

  /** Returns a new data with selector set to new value */
  def withSelector(value: Map[String, String]) : ReplicationControllerSpec = copy(selector = Some(value))
  /** Adds new values to selector */
  def addSelector(newValues: (String, String)*) : ReplicationControllerSpec = copy(selector = Some(selector.fold(newValues.toMap)(_ ++ newValues)))
  /** if selector has a value, transforms to the result of function*/
  def mapSelector(f: Map[String, String] => Map[String, String]) : ReplicationControllerSpec = copy(selector = selector.map(f))

  /** Returns a new data with template set to new value */
  def withTemplate(value: io.k8s.api.core.v1.PodTemplateSpec) : ReplicationControllerSpec = copy(template = Some(value))
  /** if template has a value, transforms to the result of function*/
  def mapTemplate(f: io.k8s.api.core.v1.PodTemplateSpec => io.k8s.api.core.v1.PodTemplateSpec) : ReplicationControllerSpec = copy(template = template.map(f))
}

object ReplicationControllerSpec {

    implicit val encoder : Encoder[io.k8s.api.core.v1.ReplicationControllerSpec] = new Encoder[io.k8s.api.core.v1.ReplicationControllerSpec] {
        def apply[T : Builder](o: io.k8s.api.core.v1.ReplicationControllerSpec) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("minReadySeconds", o.minReadySeconds)
            .write("replicas", o.replicas)
            .write("selector", o.selector)
            .write("template", o.template)
            .build
        }
    }

    implicit val decoder: Decoder[ReplicationControllerSpec] = new Decoder[ReplicationControllerSpec] {
      def apply[T : Reader](t: T): Either[String, ReplicationControllerSpec] = for {
          obj <- ObjectReader(t)
          minReadySeconds <- obj.readOpt[Int]("minReadySeconds")
          replicas <- obj.readOpt[Int]("replicas")
          selector <- obj.readOpt[Map[String, String]]("selector")
          template <- obj.readOpt[io.k8s.api.core.v1.PodTemplateSpec]("template")
      } yield ReplicationControllerSpec (
          minReadySeconds = minReadySeconds,
          replicas = replicas,
          selector = selector,
          template = template
        )
    }
}

