package io.k8s.api.discovery.v1

import dev.hnaderi.k8s.utils._

/** Endpoint represents a single logical "backend" implementing a service. */
final case class Endpoint(
  addresses : Seq[String],
  conditions : Option[io.k8s.api.discovery.v1.EndpointConditions] = None,
  targetRef : Option[io.k8s.api.core.v1.ObjectReference] = None,
  hostname : Option[String] = None,
  nodeName : Option[String] = None,
  hints : Option[io.k8s.api.discovery.v1.EndpointHints] = None,
  deprecatedTopology : Option[Map[String, String]] = None,
  zone : Option[String] = None
) {

  /** Returns a new data with addresses set to new value */
  def withAddresses(value: Seq[String]) : Endpoint = copy(addresses = value)
  /** Appends new values to addresses */
  def addAddresses(newValues: String*) : Endpoint = copy(addresses = addresses ++ newValues)
  /** transforms addresses to result of function */
  def mapAddresses(f: Seq[String] => Seq[String]) : Endpoint = copy(addresses = f(addresses))

  /** Returns a new data with conditions set to new value */
  def withConditions(value: io.k8s.api.discovery.v1.EndpointConditions) : Endpoint = copy(conditions = Some(value))
  /** if conditions has a value, transforms to the result of function*/
  def mapConditions(f: io.k8s.api.discovery.v1.EndpointConditions => io.k8s.api.discovery.v1.EndpointConditions) : Endpoint = copy(conditions = conditions.map(f))

  /** Returns a new data with targetRef set to new value */
  def withTargetRef(value: io.k8s.api.core.v1.ObjectReference) : Endpoint = copy(targetRef = Some(value))
  /** if targetRef has a value, transforms to the result of function*/
  def mapTargetRef(f: io.k8s.api.core.v1.ObjectReference => io.k8s.api.core.v1.ObjectReference) : Endpoint = copy(targetRef = targetRef.map(f))

  /** Returns a new data with hostname set to new value */
  def withHostname(value: String) : Endpoint = copy(hostname = Some(value))
  /** if hostname has a value, transforms to the result of function*/
  def mapHostname(f: String => String) : Endpoint = copy(hostname = hostname.map(f))

  /** Returns a new data with nodeName set to new value */
  def withNodeName(value: String) : Endpoint = copy(nodeName = Some(value))
  /** if nodeName has a value, transforms to the result of function*/
  def mapNodeName(f: String => String) : Endpoint = copy(nodeName = nodeName.map(f))

  /** Returns a new data with hints set to new value */
  def withHints(value: io.k8s.api.discovery.v1.EndpointHints) : Endpoint = copy(hints = Some(value))
  /** if hints has a value, transforms to the result of function*/
  def mapHints(f: io.k8s.api.discovery.v1.EndpointHints => io.k8s.api.discovery.v1.EndpointHints) : Endpoint = copy(hints = hints.map(f))

  /** Returns a new data with deprecatedTopology set to new value */
  def withDeprecatedTopology(value: Map[String, String]) : Endpoint = copy(deprecatedTopology = Some(value))
  /** Adds new values to deprecatedTopology */
  def addDeprecatedTopology(newValues: (String, String)*) : Endpoint = copy(deprecatedTopology = Some(deprecatedTopology.fold(newValues.toMap)(_ ++ newValues)))
  /** if deprecatedTopology has a value, transforms to the result of function*/
  def mapDeprecatedTopology(f: Map[String, String] => Map[String, String]) : Endpoint = copy(deprecatedTopology = deprecatedTopology.map(f))

  /** Returns a new data with zone set to new value */
  def withZone(value: String) : Endpoint = copy(zone = Some(value))
  /** if zone has a value, transforms to the result of function*/
  def mapZone(f: String => String) : Endpoint = copy(zone = zone.map(f))
}

object Endpoint {

    implicit val encoder : Encoder[io.k8s.api.discovery.v1.Endpoint] = new Encoder[io.k8s.api.discovery.v1.Endpoint] {
        def apply[T : Builder](o: io.k8s.api.discovery.v1.Endpoint) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("addresses", o.addresses)
            .write("conditions", o.conditions)
            .write("targetRef", o.targetRef)
            .write("hostname", o.hostname)
            .write("nodeName", o.nodeName)
            .write("hints", o.hints)
            .write("deprecatedTopology", o.deprecatedTopology)
            .write("zone", o.zone)
            .build
        }
    }

    implicit val decoder: Decoder[Endpoint] = new Decoder[Endpoint] {
      def apply[T : Reader](t: T): Either[String, Endpoint] = for {
          obj <- ObjectReader(t)
          addresses <- obj.read[Seq[String]]("addresses")
          conditions <- obj.readOpt[io.k8s.api.discovery.v1.EndpointConditions]("conditions")
          targetRef <- obj.readOpt[io.k8s.api.core.v1.ObjectReference]("targetRef")
          hostname <- obj.readOpt[String]("hostname")
          nodeName <- obj.readOpt[String]("nodeName")
          hints <- obj.readOpt[io.k8s.api.discovery.v1.EndpointHints]("hints")
          deprecatedTopology <- obj.readOpt[Map[String, String]]("deprecatedTopology")
          zone <- obj.readOpt[String]("zone")
      } yield Endpoint (
          addresses = addresses,
          conditions = conditions,
          targetRef = targetRef,
          hostname = hostname,
          nodeName = nodeName,
          hints = hints,
          deprecatedTopology = deprecatedTopology,
          zone = zone
        )
    }
}

